# local_secrets/payment/admin.py

from django.contrib import admin
from django.utils.html import format_html
from django.urls import reverse
from django.utils.translation import gettext_lazy as _
from .models import Payment, PaymentMethod, VendorPayment, PaymentDispute

from ..core.admin import admin_site


class VendorPaymentAdmin(admin.ModelAdmin):
    list_display = ['id', 'vendor', 'currency', 'status']
    list_filter = ['status', 'currency']
    search_fields = ['vendor__name', 'payment_gateway_reference']

    def get_readonly_fields(self, request, obj=None):
        if obj:  # Editing an existing object
            return ['amount', 'currency', 'payment_gateway_reference']
        return []

    fieldsets = (
        (_('Payment Information'), {
            'fields': ('vendor', 'currency', 'status')
        }),
        (_('Payment Details'), {
            'fields': ('payment_method', 'payment_gateway_reference')
        }),
    )
 
class PaymentMethodAdmin(admin.ModelAdmin):
    list_display = ['vendor', 'payment_type', 'is_default']
    list_filter = ['payment_type', 'is_default']
    search_fields = ['vendor__name']

    fieldsets = (
        (_('Method Information'), {
            'fields': ('vendor', 'payment_type', 'is_default')
        }),
        (_('Gateway Details'), {
            'fields': (
                'stripe_payment_method_id',
                'paypal_payment_method_id',
                'razorpay_payment_method_id',
                'syspay_payment_method_id'
            )
        }),
    )
 
class PaymentDisputeAdmin(admin.ModelAdmin):
    list_display = ['id', 'payment', 'status']
    list_filter = ['status']
    search_fields = ['payment__vendor__name', 'reason']

    fieldsets = (
        (_('Dispute Information'), {
            'fields': ('payment', 'status', 'reason')
        }),
        (_('Dispute Details'), {
            'fields': ('description', 'resolution')
        }),
    )

    def get_readonly_fields(self, request, obj=None):
        if obj:
            return ['payment', 'reason']
        return []
 
class PaymentAdmin(admin.ModelAdmin):
    list_display = ['id', 'currency', 'status']
    list_filter = ['status', 'currency']
    search_fields = ['vendor_payment__vendor__name']

    fieldsets = (
        (_('Basic Information'), {
            'fields': ('currency', 'status')
        }),
        (_('Payment Details'), {
            'fields': ('payment_gateway_reference',)
        }),
        (_('Error Information'), {
            'fields': ('error_message',),
            'classes': ('collapse',)
        }),
    )

    def get_readonly_fields(self, request, obj=None):
        if obj:
            return ['amount', 'currency']
        return []

    def has_delete_permission(self, request, obj=None):
        if obj and obj.status in ['completed', 'success']:
            return False
        return super().has_delete_permission(request, obj)

admin_site.register(Payment, PaymentAdmin)
admin_site.register(PaymentMethod, PaymentMethodAdmin)
admin_site.register(VendorPayment, VendorPaymentAdmin)
admin_site.register(PaymentDispute, PaymentDisputeAdmin)
