# local_secrets/payment/config/gateway_config.py

class PaymentGatewayConfig:
    def __init__(self, settings):
        self.settings = settings
        self.required_keys = {
            'stripe': ['STRIPE_API_KEY', 'STRIPE_SECRET_KEY', 'STRIPE_WEBHOOK_SECRET'],
            'paypal': ['PAYPAL_CLIENT_ID', 'PAYPAL_CLIENT_SECRET', 'PAYPAL_WEBHOOK_SECRET'],
        }

    def validate_config(self):
        validations = {
            'stripe': self._validate_stripe_config(),
            'paypal': self._validate_paypal_config(),
            'environment': self._validate_environment(),
            'urls': self._validate_urls()
        }
        return validations

    def _validate_stripe_config(self):
        stripe_config = {
            'valid': True,
            'issues': []
        }
        
        # Validate API keys format
        if not self.settings.get('STRIPE_API_KEY', '').startswith('pk_'):
            stripe_config['issues'].append('Invalid STRIPE_API_KEY format')
            stripe_config['valid'] = False
            
        if not self.settings.get('STRIPE_SECRET_KEY', '').startswith('sk_'):
            stripe_config['issues'].append('Invalid STRIPE_SECRET_KEY format')
            stripe_config['valid'] = False
            
        if not self.settings.get('STRIPE_WEBHOOK_SECRET', '').startswith('whsec_'):
            stripe_config['issues'].append('Invalid STRIPE_WEBHOOK_SECRET format')
            stripe_config['valid'] = False
            
        return stripe_config

    def _validate_paypal_config(self):
        paypal_config = {
            'valid': True,
            'issues': []
        }
        
        # Validate PayPal mode
        if self.settings.get('PAYPAL_MODE') not in ['sandbox', 'live']:
            paypal_config['issues'].append('Invalid PAYPAL_MODE. Must be either "sandbox" or "live"')
            paypal_config['valid'] = False
            
        # Validate credentials presence
        for key in ['PAYPAL_CLIENT_ID', 'PAYPAL_CLIENT_SECRET', 'PAYPAL_WEBHOOK_SECRET']:
            if not self.settings.get(key):
                paypal_config['issues'].append(f'Missing {key}')
                paypal_config['valid'] = False
                
        return paypal_config

    def _validate_environment(self):
        env_config = {
            'valid': True,
            'issues': []
        }
        
        # Check environment consistency
        environments = {
            'STRIPE_ENVIRONMENT': self.settings.get('STRIPE_ENVIRONMENT'),
            'PAYPAL_ENVIRONMENT': self.settings.get('PAYPAL_ENVIRONMENT'),
            'SAGEPAY_ENVIRONMENT': self.settings.get('SAGEPAY_ENVIRONMENT'),
            'MERCADOPAGO_ENVIRONMENT': self.settings.get('MERCADOPAGO_ENVIRONMENT')
        }
        
        # All environments should match in production
        if any(env == 'production' for env in environments.values()):
            if not all(env == 'production' for env in environments.values()):
                env_config['issues'].append('Inconsistent environment settings across payment gateways')
                env_config['valid'] = False
                
        return env_config

    def _validate_urls(self):
        url_config = {
            'valid': True,
            'issues': []
        }
        
        # Validate success and cancel URLs
        for url_key in ['STRIPE_SUCCESS_URL', 'STRIPE_CANCEL_URL']:
            url = self.settings.get(url_key)
            if not url or not url.startswith(('http://', 'https://')):
                url_config['issues'].append(f'Invalid {url_key}')
                url_config['valid'] = False
                
        # Validate SSL usage in production
        if any(env == 'production' for env in [
            self.settings.get('STRIPE_ENVIRONMENT'),
            self.settings.get('PAYPAL_ENVIRONMENT')
        ]):
            if not self.settings.get('PAYMENT_USES_SSL'):
                url_config['issues'].append('SSL should be enabled in production')
                url_config['valid'] = False
                
        return url_config

