# local_secrets/payment/gateways/paypal.py

from decimal import Decimal
from .base import PaymentGatewayInterface
from typing import Dict, Any
from django.conf import settings
from django.utils.translation import gettext as _
from django.core.exceptions import ValidationError
from django.core.cache import cache
from django.utils import timezone
from django.db import transaction
from django.db.models import Q
from django.urls import reverse
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.utils.html import strip_tags
from django.utils.translation import gettext_lazy as _
from django.utils.translation import gettext as _
from local_secrets.payment.models import VendorPayment, PaymentMethod
from local_secrets.payment.exceptions import PaymentError 
import paypalrestsdk
 

class PayPalGateway(PaymentGatewayInterface):
    """Implementation of PayPal payment gateway"""

    def __init__(self):
        self.client_id = settings.PAYPAL_CLIENT_ID
        self.client_secret = settings.PAYPAL_CLIENT_SECRET
        self.mode = 'sandbox' if settings.PAYPAL_SANDBOX else 'live'
        
        paypalrestsdk.configure({
            "mode": self.mode,
            "client_id": self.client_id,
            "client_secret": self.client_secret
        })

    def process_payment(self, payment: VendorPayment) -> Dict[str, Any]:
        """Process a payment through PayPal"""
        try:
            payment_data = {
                "intent": "sale",
                "payer": {
                    "payment_method": "paypal"
                },
                "transactions": [{
                    "amount": {
                        "total": str(payment.amount),
                        "currency": payment.currency
                    },
                    "description": f"Payment for {payment.vendor.company_name}",
                    "custom": str(payment.id),
                    "invoice_number": str(payment.id)
                }],
                "redirect_urls": {
                    "return_url": settings.PAYPAL_RETURN_URL,
                    "cancel_url": settings.PAYPAL_CANCEL_URL
                }
            }

            paypal_payment = paypalrestsdk.Payment(payment_data)
            
            if paypal_payment.create():
                approval_url = next(link.href for link in paypal_payment.links 
                                 if link.rel == "approval_url")
                return {
                    "payment_id": paypal_payment.id,
                    "approval_url": approval_url,
                    "status": paypal_payment.state
                }
            else:
                raise PaymentError(f"PayPal payment creation failed: {paypal_payment.error}")

        except Exception as e:
            raise PaymentError(f"PayPal payment processing error: {str(e)}")

    def refund_payment(self, payment: VendorPayment, amount: Decimal) -> Dict[str, Any]:
        """Process a refund through PayPal"""
        try:
            paypal_payment = paypalrestsdk.Payment.find(payment.payment_gateway_reference)
            sale = paypal_payment.transactions[0].related_resources[0].sale

            refund = sale.refund({
                "amount": {
                    "total": str(amount),
                    "currency": payment.currency
                }
            })

            if refund.success():
                return {
                    "refund_id": refund.id,
                    "status": refund.state,
                    "amount": amount
                }
            else:
                raise PaymentError(f"PayPal refund failed: {refund.error}")

        except Exception as e:
            raise PaymentError(f"PayPal refund processing error: {str(e)}")

    def verify_webhook(self, payload: Dict[str, Any], signature: str) -> bool:
        """Verify webhook signature from PayPal"""
        try:
            # PayPal webhook verification logic
            verification_response = paypalrestsdk.notifications.WebhookEvent.verify(
                payload,
                signature,
                settings.PAYPAL_WEBHOOK_ID
            )
            return verification_response
        except Exception as e:
            raise PaymentError(f"PayPal webhook verification error: {str(e)}")

    def get_payment_status(self, payment_id: str) -> str:
        """Get payment status from PayPal"""
        try:
            paypal_payment = paypalrestsdk.Payment.find(payment_id)
            return paypal_payment.state
        except Exception as e:
            raise PaymentError(f"Error retrieving PayPal payment status: {str(e)}")

    def execute_payment(self, payment_id: str, payer_id: str) -> Dict[str, Any]:
        """Execute a PayPal payment after user approval"""
        try:
            paypal_payment = paypalrestsdk.Payment.find(payment_id)
            if paypal_payment.execute({"payer_id": payer_id}):
                return {
                    "payment_id": paypal_payment.id,
                    "status": paypal_payment.state,
                    "transaction_id": paypal_payment.transactions[0].related_resources[0].sale.id
                }
            else:
                raise PaymentError(f"PayPal payment execution failed: {paypal_payment.error}")
        except Exception as e:
            raise PaymentError(f"PayPal payment execution error: {str(e)}")
