# local_secrets/payment/gateways/stripe.py
from decimal import Decimal
from django.conf import settings 
from django.core.exceptions import ValidationError
from django.core.cache import cache
from django.utils import timezone
from django.db import transaction
from django.db.models import Q
from django.urls import reverse
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.utils.html import strip_tags
from django.utils.translation import gettext_lazy as _
from django.utils.translation import gettext as _
from local_secrets.payment.config.gateway_config import PaymentGatewayConfig
from local_secrets.payment.models import VendorPayment, PaymentMethod
from local_secrets.payment.exceptions import PaymentError
from local_secrets.payment.security.payment_security import PaymentSecurity
from local_secrets.payment.utils.error_limiting import PaymentErrorHandler
from local_secrets.users.models import Vendor
from .base import PaymentGatewayInterface
from typing import Dict, Any
from stripe import stripe, error as stripe_error
from stripe import PaymentIntent, Refund, Webhook 
from local_secrets.payment.services import PaymentService
 

class StripeGateway(PaymentGatewayInterface):
    """Implementation of Stripe payment gateway"""
    def __init__(self):
        # Validate configuration
        config = PaymentGatewayConfig()
        self.config = config.validate_gateway_config('stripe')
        
        # Initialize error handler
        self.error_handler = PaymentErrorHandler()
        
        # Initialize Stripe
        stripe.api_key = self.config['STRIPE_SECRET_KEY']

    def process_payment(self, payment: VendorPayment) -> Dict[str, Any]:
        """Process a payment through Stripe"""
        try:
            intent = stripe.PaymentIntent.create(
                amount=int(payment.amount * 100),  # Convert to cents
                currency=payment.currency.lower(),
                payment_method=payment.payment_method.stripe_payment_method_id,
                metadata={
                    "vendor_payment_id": str(payment.id),
                    "vendor_id": str(payment.vendor.id)
                },
                confirm=True,
                return_url=settings.STRIPE_RETURN_URL
            )
            
            return {
                "payment_intent_id": intent.id,
                "client_secret": intent.client_secret,
                "status": intent.status
            }
        except stripe.error.StripeError as e:
            raise PaymentError(f"Stripe payment processing error: {str(e)}")
        except Exception as e:
            error_data = self.error_handler.handle_payment_error(e, payment)
            raise PaymentError(error_data['error_message'])

    def refund_payment(self, payment: VendorPayment, amount: Decimal) -> Dict[str, Any]:
        """Process a refund through Stripe"""
        try:
            refund = stripe.Refund.create(
                payment_intent=payment.payment_gateway_reference,
                amount=int(amount * 100),  # Convert to cents
                metadata={
                    "vendor_payment_id": str(payment.id),
                    "vendor_id": str(payment.vendor.id)
                }
            )
            
            return {
                "refund_id": refund.id,
                "status": refund.status,
                "amount": Decimal(refund.amount) / 100  # Convert back to decimal
            }
        except stripe.error.StripeError as e:
            raise PaymentError(f"Stripe refund processing error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Unexpected error during refund processing: {str(e)}")

    def verify_webhook(self, payload: Dict[str, Any], signature: str) -> bool:
        return PaymentSecurity.validate_webhook_signature('stripe', payload, signature)

    def verify_webhook(self, payload: Dict[str, Any], signature: str) -> bool:
        """Verify webhook signature from Stripe"""
        try:
            stripe.Webhook.construct_event(
                payload,
                signature,
                settings.STRIPE_WEBHOOK_SECRET
            )
            return True
        except stripe.error.SignatureVerificationError:
            return False
        except Exception as e:
            raise PaymentError(f"Webhook verification error: {str(e)}")

    def get_payment_status(self, payment_id: str) -> str:
        """Get payment status from Stripe"""
        try:
            payment_intent = stripe.PaymentIntent.retrieve(payment_id)
            return payment_intent.status
        except stripe.error.StripeError as e:
            raise PaymentError(f"Error retrieving payment status: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Unexpected error getting payment status: {str(e)}")

    def create_setup_intent(self, vendor: 'Vendor') -> Dict[str, Any]:
        """Create a setup intent for adding a payment method"""
        try:
            setup_intent = stripe.SetupIntent.create(
                customer=vendor.stripe_customer_id,
                metadata={
                    "vendor_id": str(vendor.id)
                }
            )
            return {
                "setup_intent_id": setup_intent.id,
                "client_secret": setup_intent.client_secret
            }
        except stripe.error.StripeError as e:
            raise PaymentError(f"Error creating setup intent: {str(e)}")

            
class StripeRefund(PaymentGatewayInterface):
    """Implementation of Stripe refund gateway"""

    def __init__(self, api_key: str = settings.STRIPE_API_KEY):
        self.api_key = api_key
        stripe.api_key = self.api_key

    def refund_payment(self, payment: VendorPayment, amount: float) -> Dict[str, Any]:
        """Process a refund through Stripe"""
        try:
            refund = Refund.create(
                payment_intent=payment.payment_intent_id,
                amount=int(amount * 100),  # Amount in cents
            )
            return {
                "refund_id": refund.id,
                "status": refund.status,
            }
        except stripe_error.StripeError as e:
            raise PaymentError(f"Stripe error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Refund processing failed: {str(e)}")

    def verify_webhook(self, payload: Dict[str, Any], signature: str) -> bool:
        """Verify webhook signature from Stripe"""
        try:
            event = Webhook.construct_event(
                payload, signature, settings.STRIPE_WEBHOOK_SECRET
            )
            return True
        except stripe_error.SignatureVerificationError as e:
            raise PaymentError(f"Webhook signature verification failed: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Webhook processing failed: {str(e)}")


    def get_payment_status(self, payment_id: str) -> str:
        """Get payment status from Stripe"""
        try:
            payment = PaymentIntent.retrieve(payment_id)
            return payment.status
        except stripe_error.StripeError as e:
            raise PaymentError(f"Stripe error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Payment status retrieval failed: {str(e)}")


    def get_payment_method(self, payment_method_id: str) -> PaymentMethod:
        """Get payment method details from Stripe"""
        try:
            payment_method = PaymentMethod.retrieve(payment_method_id)
            return PaymentMethod(
                id=payment_method.id,
                brand=payment_method.card.brand,
                last4=payment_method.card.last4,
                exp_month=payment_method.card.exp_month,
                exp_year=payment_method.card.exp_year,
            )
        except stripe_error.StripeError as e:
            raise PaymentError(f"Stripe error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Payment method retrieval failed: {str(e)}")
 
class StripePaymentService(PaymentService):
    """Service layer for handling Stripe payment operations"""

    def __init__(self, gateway: StripeGateway):
        super().__init__(gateway)

    def create_payment(self, vendor: 'Vendor', amount: float, 
                       currency: str, payment_method: 'PaymentMethod') -> 'VendorPayment':
        """Create and process a new payment"""
        try:
            payment = VendorPayment.objects.create(
                vendor=vendor,
                amount=amount,
                currency=currency,
                payment_method=payment_method
            )
            
            result = self.gateway.process_payment(payment)
            payment.update_status(result)
            
            return payment
        except Exception as e:
            raise PaymentError(f"Payment failed: {str(e)}")


    def handle_webhook(self, payload: Dict[str, Any], signature: str) -> None:
        """Handle webhook notifications from Stripe"""
        if not self.gateway.verify_webhook(payload, signature):
            raise PaymentError("Invalid webhook signature")
            
        event_type = payload.get('type')
        handler = self.get_webhook_handler(event_type)
        if handler:
            handler(payload)

        else:   
            raise PaymentError(f"Unhandled event type: {event_type}")

    def get_webhook_handler(self, event_type: str):
        """Get appropriate webhook handler for event type"""
        handlers = {
            'payment_intent.succeeded': self._handle_payment_intent_succeeded,
            'payment_intent.payment_failed': self._handle_payment_intent_failed,
            'charge.refunded': self._handle_charge_refunded,
            'dispute.created': self._handle_dispute_created,
        }
        return handlers.get(event_type)

    def _handle_payment_intent_succeeded(self, payload: Dict[str, Any]) -> None:    
        """Handle payment intent succeeded event"""
        payment_intent_id = payload['data']['object']['id']
        payment = VendorPayment.objects.get(payment_intent_id=payment_intent_id)
        payment.update_status('succeeded')
        payment.save()
        # Notify vendor about successful payment
        self.notify_vendor(payment, "Payment succeeded")    

    def _handle_payment_intent_failed(self, payload: Dict[str, Any]) -> None:
        """Handle payment intent failed event"""
        payment_intent_id = payload['data']['object']['id']
        payment = VendorPayment.objects.get(payment_intent_id=payment_intent_id)
        payment.update_status('failed')
        payment.save()
        # Notify vendor about failed payment
        self.notify_vendor(payment, "Payment failed")

    def _handle_charge_refunded(self, payload: Dict[str, Any]) -> None: 
        """Handle charge refunded event"""
        charge_id = payload['data']['object']['id']
        payment = VendorPayment.objects.get(charge_id=charge_id)
        payment.update_status('refunded')
        payment.save()
        # Notify vendor about refunded payment
        self.notify_vendor(payment, "Payment refunded")

    def _handle_dispute_created(self, payload: Dict[str, Any]) -> None:
        """Handle dispute created event"""
        dispute_id = payload['data']['object']['id']
        payment = VendorPayment.objects.get(dispute_id=dispute_id)
        payment.update_status('disputed')
        payment.save()
        # Notify vendor about disputed payment
        self.notify_vendor(payment, "Payment disputed")

    def notify_vendor(self, payment: 'VendorPayment', message: str) -> None:
        """Notify vendor about payment status"""
        subject = _("Payment Notification")
        message = render_to_string(
            'payment/payment_notification_email.html',
            {'payment': payment, 'message': message}
        )
        plain_message = strip_tags(message)
        send_mail(subject, plain_message, settings.DEFAULT_FROM_EMAIL, [payment.vendor.email])
        # Optionally, you can use Django's messaging framework to notify the vendor in the UI
        # messages.success(payment.vendor.user, message)    
 
    def get_payment_method(self, payment_method_id: str) -> PaymentMethod:
        """Get payment method details from Stripe"""
        try:
            payment_method = PaymentMethod.retrieve(payment_method_id)
            return PaymentMethod(
                id=payment_method.id,
                brand=payment_method.card.brand,
                last4=payment_method.card.last4,
                exp_month=payment_method.card.exp_month,
                exp_year=payment_method.card.exp_year,
            )
        except stripe_error.StripeError as e:
            raise PaymentError(f"Stripe error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Payment method retrieval failed: {str(e)}")

    def update_payment_method(self, payment_method: PaymentMethod) -> None:
        """Update payment method details in Stripe"""
        try:
            payment_method.save()
        except stripe_error.StripeError as e:
            raise PaymentError(f"Stripe error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Payment method update failed: {str(e)}")

    def remove_payment_method(self, payment_method: PaymentMethod) -> None:
        """Remove payment method from Stripe"""
        try:
            payment_method.delete()
        except stripe_error.StripeError as e:
            raise PaymentError(f"Stripe error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Payment method removal failed: {str(e)}")

    def list_payment_methods(self, vendor: 'Vendor') -> list:   
        """List payment methods for a vendor"""
        try:
            payment_methods = PaymentMethod.list(vendor=vendor.id)
            return payment_methods.data
        except stripe_error.StripeError as e:
            raise PaymentError(f"Stripe error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Payment method listing failed: {str(e)}")

    def add_payment_method(self, vendor: 'Vendor', payment_method: PaymentMethod) -> None:
        """Add a new payment method for a vendor"""
        try:
            payment_method.save(vendor=vendor.id)
        except stripe_error.StripeError as e:
            raise PaymentError(f"Stripe error: {str(e)}")
        except Exception as e:
            raise PaymentError(f"Adding payment method failed: {str(e)}")


