# local_secrets/payment/processors.py

from abc import ABC, abstractmethod
from decimal import Decimal
from typing import Optional, Dict, Any
from django.utils.translation import gettext as _
from django.db import transaction
from django.core.cache import cache
from django.conf import settings
from django.utils import timezone

from .models import VendorPayment, PaymentMethod
from .exceptions import PaymentProcessError
from .gateways.base import PaymentGatewayInterface
from .notifications import PaymentNotificationService

class BasePaymentProcessor(ABC):
    """Abstract base class for payment processors"""

    def __init__(self, payment: VendorPayment, gateway: PaymentGatewayInterface):
        self.payment = payment
        self.gateway = gateway
        self.notification_service = PaymentNotificationService()
        self.cache_key = f"payment_processing_{payment.id}"
        self.cache_timeout = 300  # 5 minutes

    @abstractmethod
    def process_payment(self) -> Dict[str, Any]:
        """Process the payment through the gateway"""
        pass

    @abstractmethod
    def handle_callback(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Handle gateway callback/webhook"""
        pass

    def validate(self) -> bool:
        """Validate payment before processing"""
        if self.payment.amount <= 0:
            raise PaymentProcessError(_("Invalid payment amount"))
        
        if not self.payment.payment_method:
            raise PaymentProcessError(_("No payment method provided"))
        
        if self.payment.status not in ['pending', 'failed']:
            raise PaymentProcessError(_("Invalid payment status for processing"))
        
        if cache.get(self.cache_key):
            raise PaymentProcessError(_("Payment is already being processed"))
        
        return True

    def pre_process(self) -> None:
        """Pre-processing steps"""
        with transaction.atomic():
            self.payment.status = 'processing'
            self.payment.attempts += 1
            self.payment.last_processed_at = timezone.now()
            self.payment.save()
            
            # Set processing lock
            cache.set(self.cache_key, True, self.cache_timeout)

    def post_process(self, success: bool, response: Dict[str, Any]) -> None:
        """Post-processing steps"""
        try:
            with transaction.atomic():
                if success:
                    self.handle_success(response)
                else:
                    self.handle_failure(response)
        finally:
            # Clear processing lock
            cache.delete(self.cache_key)

    def handle_success(self, response: Dict[str, Any]) -> None:
        """Handle successful payment"""
        self.payment.status = 'completed'
        self.payment.completed_at = timezone.now()
        self.payment.payment_gateway_reference = response.get('transaction_id')
        self.payment.response_data = response
        self.payment.save()
        
        self.notification_service.send_payment_success(self.payment)

    def handle_failure(self, response: Dict[str, Any]) -> None:
        """Handle failed payment"""
        self.payment.status = 'failed'
        self.payment.error_message = response.get('error_message', '')
        self.payment.response_data = response
        self.payment.save()
        
        self.notification_service.send_payment_failure(self.payment)

    def refund(self, amount: Optional[Decimal] = None) -> Dict[str, Any]:
        """Process refund"""
        if not amount:
            amount = self.payment.amount
            
        if amount > self.payment.amount:
            raise PaymentProcessError(_("Refund amount cannot exceed payment amount"))
            
        try:
            response = self.gateway.refund_payment(self.payment, amount)
            
            with transaction.atomic():
                self.payment.status = 'refunded'
                self.payment.refunded_amount = amount
                self.payment.refunded_at = timezone.now()
                self.payment.save()
                
                self.notification_service.send_refund_notification(self.payment)
                
            return response
            
        except Exception as e:
            raise PaymentProcessError(f"Refund failed: {str(e)}")

class StripePaymentProcessor(BasePaymentProcessor):
    """Stripe payment processor implementation"""

    def process_payment(self) -> Dict[str, Any]:
        """Process payment through Stripe"""
        try:
            self.validate()
            self.pre_process()
            
            response = self.gateway.process_payment(self.payment)
            success = response.get('status') in ['succeeded', 'requires_confirmation']
            
            self.post_process(success, response)
            return response
            
        except Exception as e:
            self.post_process(False, {'error_message': str(e)})
            raise PaymentProcessError(f"Stripe payment failed: {str(e)}")

    def handle_callback(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Handle Stripe webhook"""
        event_type = data.get('type')
        event_handlers = {
            'payment_intent.succeeded': self._handle_payment_success,
            'payment_intent.payment_failed': self._handle_payment_failure,
            'charge.refunded': self._handle_refund,
            'charge.dispute.created': self._handle_dispute
        }
        
        handler = event_handlers.get(event_type)
        if handler:
            return handler(data)
        return {'status': 'ignored', 'message': f'Unhandled event type: {event_type}'}

    def _handle_payment_success(self, data: Dict[str, Any]) -> Dict[str, Any]:
        payment_intent = data['data']['object']
        self.payment.status = 'completed'
        self.payment.completed_at = timezone.now()
        self.payment.payment_gateway_reference = payment_intent['id']
        self.payment.save()
        return {'status': 'success'}

    def _handle_payment_failure(self, data: Dict[str, Any]) -> Dict[str, Any]:
        payment_intent = data['data']['object']
        self.payment.status = 'failed'
        self.payment.error_message = payment_intent.get('last_payment_error', {}).get('message', '')
        self.payment.save()
        return {'status': 'failed'}

    def _handle_refund(self, data: Dict[str, Any]) -> Dict[str, Any]:
        charge = data['data']['object']
        self.payment.status = 'refunded'
        self.payment.refunded_amount = Decimal(charge['amount_refunded']) / 100
        self.payment.refunded_at = timezone.now()
        self.payment.save()
        return {'status': 'refunded'}

    def _handle_dispute(self, data: Dict[str, Any]) -> Dict[str, Any]:
        dispute = data['data']['object']
        self.payment.status = 'disputed'
        self.payment.dispute_reason = dispute.get('reason')
        self.payment.disputed_at = timezone.now()
        self.payment.save()
        return {'status': 'disputed'}

class PayPalPaymentProcessor(BasePaymentProcessor):
    """PayPal payment processor implementation"""
    
    def process_payment(self) -> Dict[str, Any]:
        """Process payment through PayPal"""
        try:
            self.validate()
            self.pre_process()
            
            response = self.gateway.process_payment(self.payment)
            success = response.get('status') in ['COMPLETED', 'APPROVED']
            
            self.post_process(success, response)
            return response
            
        except Exception as e:
            self.post_process(False, {'error_message': str(e)})
            raise PaymentProcessError(f"PayPal payment failed: {str(e)}")

    def handle_callback(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Handle PayPal webhook"""
        event_type = data.get('event_type')
        event_handlers = {
            'PAYMENT.CAPTURE.COMPLETED': self._handle_payment_success,
            'PAYMENT.CAPTURE.DENIED': self._handle_payment_failure,
            'REFUND.COMPLETED': self._handle_refund
        }
        
        handler = event_handlers.get(event_type)
        if handler:
            return handler(data)
        return {'status': 'ignored', 'message': f'Unhandled event type: {event_type}'}

    def _handle_payment_success(self, data: Dict[str, Any]) -> Dict[str, Any]:
        resource = data['resource']
        self.payment.status = 'completed'
        self.payment.completed_at = timezone.now()
        self.payment.payment_gateway_reference = resource['id']
        self.payment.save()
        return {'status': 'success'}

    def _handle_payment_failure(self, data: Dict[str, Any]) -> Dict[str, Any]:
        resource = data['resource']
        self.payment.status = 'failed'
        self.payment.error_message = resource.get('status_details', {}).get('reason', '')
        self.payment.save()
        return {'status': 'failed'}

    def _handle_refund(self, data: Dict[str, Any]) -> Dict[str, Any]:
        resource = data['resource']
        self.payment.status = 'refunded'
        self.payment.refunded_amount = Decimal(resource['amount']['value'])
        self.payment.refunded_at = timezone.now()
        self.payment.save()
        return {'status': 'refunded'}
