# local_secrets/payment/providers/syspay_provider.py

import json
import requests
from django.conf import settings
from django.utils import timezone
from ..exceptions import PaymentGatewayError, PaymentMethodError, PaymentProcessError, PaymentProcessingError
import hmac
import hashlib
import base64
from datetime import datetime, time, timedelta 
from django.utils.translation import gettext_lazy as _
import logging
from django.core.cache import cache
from typing import Dict, Any, Optional
from local_secrets.payment.processors import PaymentMethod 
from local_secrets.payment.models import Payment, PaymentMethod, AutomaticBilling
logger = logging.getLogger(__name__)

class SysPayProvider:
    logger = logging.getLogger(__name__)

    API_VERSION = 'v1'
    RETRY_COUNT = 3
    RETRY_DELAY = 2  # seconds
    CACHE_TTL = 3600  # 1 hour
    
    # Payment status constants
    PAYMENT_STATUS = {
        'PENDING': 'pending',
        'SUCCESS': 'success',
        'FAILED': 'failed',
        'CANCELLED': 'cancelled',
        'REFUNDED': 'refunded'
    }

    def __init__(self):
        self.api_url = settings.SYSPAY_API_URL
        self.partner_id = settings.SYSPAY_PARTNER_ID
        self.api_key = settings.SYSPAY_API_KEY
        self.return_url = settings.SYSPAY_RETURN_URL
        self.ems_url = settings.SYSPAY_EMS_URL
        self.headers = {
            'Authorization': f'Bearer {self.api_key}',
            'Partner-ID': self.partner_id,
            'Content-Type': 'application/json'
        }
        self.retry_count = settings.SYSPAY_API_RETRIES
        self.retry_delay = settings.SYSPAY_API_RETRY_DELAY

    def _validate_settings(self) -> None:
        """Validate that all required settings are present"""
        required_settings = [
            'SYSPAY_API_KEY',
            'SYSPAY_PARTNER_ID',
            'SYSPAY_WEBHOOK_SECRET',
            'SYSPAY_RETURN_URL',
            'SYSPAY_EMS_URL'
        ]
        
        missing_settings = [
            setting for setting in required_settings 
            if not hasattr(settings, setting)
        ]
        
        if missing_settings:
            raise ValueError(
                f"Missing required SysPay settings: {', '.join(missing_settings)}"
            )

    def _make_request(self, method, endpoint, data=None):
        """Make request to SysPay API"""
        try:
            response = requests.request(
                method,
                f"{self.base_url}/{endpoint}",
                headers=self.headers,
                json=data
            )
            response.raise_for_status()
            return response.json()
        except requests.exceptions.RequestException as e:
            self.logger.error(f"SysPay API error: {str(e)}", extra={
                'endpoint': endpoint,
                'method': method
            })
            raise PaymentGatewayError(f"SysPay API error: {str(e)}")
    
    def _prepare_headers(self, data: Optional[Dict[str, Any]], timestamp: str) -> Dict[str, str]:
        """Prepare request headers with authentication"""
        return {
            'X-Partner-Id': self.partner_id,
            'X-Timestamp': timestamp,
            'X-Signature': self._generate_signature(
                json.dumps(data) if data else '', 
                timestamp
            ),
            'Content-Type': 'application/json',
            'X-Api-Version': self.API_VERSION,
            'User-Agent': f'SysPay-Python/{self.API_VERSION}'
        }

    def _execute_request(
        self, 
        method: str, 
        url: str, 
        data: Optional[Dict[str, Any]], 
        headers: Dict[str, str]
    ) -> Dict[str, Any]:
        """Execute HTTP request and handle common responses"""
        response = self.session.request(
            method,
            url,
            json=data,
            headers=headers,
            timeout=self.timeout
        )
        
        if response.status_code == 429:
            retry_after = int(response.headers.get('Retry-After', self.RETRY_DELAY))
            logger.warning(f"Rate limit hit, waiting {retry_after} seconds")
            time.sleep(retry_after)
            return self._execute_request(method, url, data, headers)
            
        response.raise_for_status()
        return response.json()

    def _handle_http_error(self, error: requests.exceptions.HTTPError, attempt: int, max_attempts: int) -> None:
        """Handle HTTP errors with proper logging and retrying"""
        error_data = error.response.json() if error.response.content else {}
        logger.error(
            f"SysPay API HTTP error: {str(error)}", 
            extra={
                'error_data': error_data,
                'attempt': attempt + 1,
                'max_attempts': max_attempts
            }
        )
        
        if attempt == max_attempts - 1:
            raise SysPayError(error_data)

    def _handle_request_error(self, error: requests.exceptions.RequestException, attempt: int, max_attempts: int) -> None:
        """Handle general request errors"""
        logger.error(
            f"SysPay API request failed: {str(error)}",
            extra={
                'attempt': attempt + 1,
                'max_attempts': max_attempts
            }
        )
        
        if attempt == max_attempts - 1:
            raise APIConnectionError("Failed to connect to SysPay API")

    def get_onboarding_link(self, vendor) -> Dict[str, Any]:
        """Get onboarding link for vendor with caching"""
        cache_key = f'syspay_onboarding_link_{vendor.id}'
        cached_link = cache.get(cache_key)
        
        if cached_link:
            return cached_link
            
        try:
            result = self._make_request(
                'GET', 
                f'users/{vendor.syspay_user_id}/onboarding-link'
            )
            cache.set(cache_key, result, timeout=300)  # Cache for 5 minutes
            return result
        except Exception as e:
            logger.error(f"Failed to get onboarding link: {str(e)}")
            raise

    def get_user_info(self, vendor) -> Dict[str, Any]:
        """Get SysPay user information with caching"""
        cache_key = f'syspay_user_info_{vendor.id}'
        cached_info = cache.get(cache_key)
        
        if cached_info:
            return cached_info
            
        try:
            result = self._make_request('GET', f'users/{vendor.syspay_user_id}')
            self._cache_user_info(vendor.id, result)
            return result
        except requests.exceptions.HTTPError as e:
            if e.response.status_code == 404:
                raise UserNotFoundError(f"User not found: {vendor.syspay_user_id}")
            raise

    def verify_webhook(self, payload: str, signature: str, timestamp: str) -> bool:
        """Verify webhook signature with additional security checks"""
        try:
            # Verify timestamp is recent
            webhook_time = datetime.strptime(timestamp, '%Y-%m-%d %H:%M:%S')
            time_diff = datetime.utcnow() - webhook_time
            
            if time_diff > timedelta(minutes=5):
                logger.warning("Webhook timestamp too old")
                return False
                
            expected_signature = self._generate_signature(payload, timestamp)
            return hmac.compare_digest(signature, expected_signature)
            
        except Exception as e:
            logger.error(f"Webhook verification failed: {str(e)}")
            return False

    def _cache_user_info(self, vendor_id: int, user_info: Dict[str, Any]) -> None:
        """Cache user information"""
        cache_key = f'syspay_user_info_{vendor_id}'
        cache.set(cache_key, user_info, timeout=self.CACHE_TTL)

    def create_hosted_token(self, vendor, payment_data):
        """Create a hosted token page URL"""
        data = {
            'flow': 'BUYER',
            'source': vendor.id,
            'description': payment_data.get('description'),
            'return_url': settings.SYSPAY_RETURN_URL,
            'ems_url': settings.SYSPAY_EMS_URL,
            'notify': payment_data.get('notify', 'EMAIL'),
            'mandate': True,  # Enable future payments
            'customer': {
                'email': vendor.contact_email,
                'reference': f"vendor_{vendor.id}"
            }
        }
        return self._make_request('POST', 'merchant/token', data)

    def import_token(self, token_data):
        """Import existing token"""
        return self._make_request('POST', 'merchant/token/import', token_data)

    def validate_token(self, token_id):
        """Validate an existing token"""
        return self._make_request('POST', f'merchant/token/{token_id}/validate')

    def update_token(self, token_id, update_data):
        """Update token information"""
        return self._make_request('PUT', f'merchant/token/{token_id}', update_data)

    def get_token_info(self, token_id):
        """Get token information"""
        return self._make_request('GET', f'merchant/token/{token_id}')
 
    def _generate_signature(self, data, timestamp):
        """Generate HMAC signature for API requests"""
        message = f"{timestamp}{self.partner_id}{data}"
        signature = hmac.new(
            self.api_key.encode(),
            message.encode(),
            hashlib.sha256
        ).digest()
        return base64.b64encode(signature).decode()

    def create_user(self, user):
        """Create merchant account in SysPay"""
        if not user.is_vendor:
            raise ValueError("User does not have a vendor profile")
            
        vendor = user.vendor_profile
        data = {
            'email': vendor.contact_email,
            'first_name': user.first_name,
            'last_name': user.last_name,
            'company_name': vendor.company_name,
            'reference': f"vendor_{vendor.id}",
            'country': vendor.primary_address.country.code if vendor.primary_address else None,
        }
        
        response = self._make_request('POST', 'users', data)
        if response.get('status') == 'success':
            return {
                'user_id': response['user_id'],
                'status': response['status']
            }
        raise PaymentError(response.get('message', 'Failed to create SysPay user'))

    def setup_payment_method(self, user, payment_data):
        """
        Setup payment method for a vendor
        Args:
            user: The user object
            payment_data: Dictionary containing payment details
        Returns:
            Dictionary with payment method details
        """
        try:
            self.logger.info("Setting up payment method", extra={
                'user_id': user.id
            })

            if not hasattr(user, 'vendor_profile'):
                raise ValueError("User does not have a vendor profile")

            vendor = user.vendor_profile
            
            # Prepare payment method data
            method_data = {
                'type': payment_data.get('type', 'card'),
                'card_number': payment_data.get('card_number'),
                'exp_month': payment_data.get('exp_month'),
                'exp_year': payment_data.get('exp_year'),
                'cvv': payment_data.get('cvv'),
                'currency': payment_data.get('currency', 'USD'),
                'customer_reference': f"vendor_{vendor.id}",
                'metadata': {
                    'vendor_id': str(vendor.id),
                    'company_name': vendor.company_name
                }
            }

            # Make API request to SysPay
            response = self._make_request('POST', 'payment-methods', method_data)

            if response.get('status') == 'success':
                payment_method = PaymentMethod.objects.create(
                    vendor=vendor,
                    payment_type='syspay',
                    syspay_payment_method_id=response['payment_method_id'],
                    is_default=payment_data.get('is_default', False)
                )

                self.logger.info("Payment method setup successful", extra={
                    'user_id': user.id,
                    'payment_method_id': payment_method.id
                })

                return {
                    'id': payment_method.id,
                    'payment_method_id': response['payment_method_id'],
                    'status': 'success'
                }
            else:
                raise PaymentProcessError(response.get('message', 'Failed to setup payment method'))

        except Exception as e:
            self.logger.error("Failed to setup payment method", extra={
                'user_id': user.id,
                'error': str(e)
            })
            raise

    def process_subscription_payment(self, subscription, payment_method):
        """Process subscription payment"""
        data = {
            'user_id': subscription.vendor.syspay_user_id,
            'payment_method_id': payment_method.syspay_payment_method_id,
            'amount': subscription.plan.price,
            'currency': subscription.plan.currency,
            'description': f"Subscription: {subscription.plan.name}",
            'reference': f"sub_{subscription.id}_{datetime.now().timestamp()}"
        }
        
        response = self._make_request('POST', 'payments', data)
        if response.get('status') == 'success':
            return {
                'transaction_id': response['transaction_id'],
                'status': response['status']
            }
        raise PaymentError(response.get('message', 'Payment processing failed'))

    def setup_recurring_payment(self, subscription):
        """Setup recurring payment for subscription"""
        data = {
            'user_id': subscription.vendor.syspay_user_id,
            'payment_method_id': subscription.payment_method.syspay_payment_method_id,
            'amount': subscription.plan.price,
            'currency': subscription.plan.currency,
            'interval': subscription.plan.billing_interval,
            'interval_count': subscription.plan.billing_interval_count,
            'description': f"Recurring: {subscription.plan.name}",
            'reference': f"recurring_{subscription.id}"
        }
        
        response = self._make_request('POST', 'recurring', data)
        if response.get('status') == 'success':
            return {
                'subscription_id': response['subscription_id'],
                'status': response['status']
            }
        raise PaymentError(response.get('message', 'Failed to setup recurring payment'))

    def cancel_recurring_payment(self, subscription):
        """Cancel recurring payment"""
        if not subscription.syspay_subscription_id:
            return {'status': 'success'}
            
        response = self._make_request(
            'POST', 
            f'recurring/{subscription.syspay_subscription_id}/cancel'
        )
        return {'status': response.get('status', 'failed')}

    def create_hosted_payment(self, payment_data):
        """Create a hosted payment page URL"""
        data = {
            'flow': 'BUYER',
            'source': payment_data.get('source'),
            'description': payment_data.get('description'),
            'return_url': settings.SYSPAY_RETURN_URL,
            'ems_url': settings.SYSPAY_EMS_URL,
            'notify': payment_data.get('notify', 'EMAIL'),
            'customer': payment_data.get('customer'),
            'payment': {
                'amount': payment_data['amount'],
                'currency': payment_data['currency'],
                'reference': payment_data.get('reference'),
                'preauth': payment_data.get('preauth', False)
            }
        }
        return self._make_request('POST', 'merchant/payment', data)

    def process_server_payment(self, payment_data):
        """Process server-to-server payment"""
        return self._make_request('POST', 'merchant/payment/process', payment_data)

    def capture_payment(self, payment_id, amount=None):
        """Capture pre-authorized payment"""
        data = {'amount': amount} if amount else {}
        return self._make_request('POST', f'merchant/payment/{payment_id}/capture', data)

    def void_payment(self, payment_id):
        """Void pre-authorized payment"""
        return self._make_request('POST', f'merchant/payment/{payment_id}/void')

    def refund_payment(self, payment_id, amount=None):
        """Refund payment"""
        data = {'amount': amount} if amount else {}
        return self._make_request('POST', f'merchant/payment/{payment_id}/refund', data)

    def create_phone_booking(self, booking_data):
        """Create phone booking request"""
        return self._make_request('POST', 'merchant/phone-booking', booking_data)

    def retrieve_phone_booking_token(self, booking_id):
        """Retrieve token from phone booking"""
        return self._make_request('GET', f'merchant/phone-booking/{booking_id}/token')

    def _get_test_response(self, method, endpoint, data):
        """Generate mock responses for test mode"""
        if 'payment-methods' in endpoint:
            return {
                'id': 'pm_test_123',
                'status': 'active',
                'last4': '4242',
                'card_type': 'visa'
            }
        elif 'payments' in endpoint:
            return {
                'id': 'pay_test_123',
                'status': 'succeeded',
                'amount': data.get('amount'),
                'currency': data.get('currency')
            }
        return {}



    def create_payment_intent(self, payment_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Create a new payment intent.
        
        Args:
            payment_data: Dictionary containing payment details
            
        Returns:
            Payment intent details
        """
        required_fields = ['amount', 'currency', 'payment_method']
        self._validate_required_fields(payment_data, required_fields)
        
        return self._make_request('POST', 'payment-intents', payment_data)

    def _validate_required_fields(self, data: Dict[str, Any], required_fields: list) -> None:
        """Validate that all required fields are present in the data"""
        missing_fields = [
            field for field in required_fields 
            if field not in data
        ]
        
        if missing_fields:
            raise ValueError(
                f"Missing required fields: {', '.join(missing_fields)}"
            )

    def handle_webhook_event(self, payload: Dict[str, Any], headers: Dict[str, str]) -> Dict[str, Any]:
        """
        Handle incoming webhook events.
        
        Args:
            payload: Webhook payload
            headers: Request headers containing signature
            
        Returns:
            Processed webhook data
            
        Raises:
            SysPayError: When webhook verification fails
        """
        if not self.verify_webhook(
            payload=json.dumps(payload),
            signature=headers.get('X-Signature'),
            timestamp=headers.get('X-Timestamp')
        ):
            raise SysPayError("Invalid webhook signature")
            
        event_type = payload.get('type')
        event_handlers = {
            'payment.succeeded': self._handle_payment_succeeded,
            'payment.failed': self._handle_payment_failed,
            'refund.succeeded': self._handle_refund_succeeded
        }
        
        handler = event_handlers.get(event_type)
        if handler:
            return handler(payload)
            
        logger.warning(f"Unhandled webhook event type: {event_type}")
        return {'status': 'ignored'}

    def _handle_payment_succeeded(self, payload: Dict[str, Any]) -> Dict[str, Any]:
        """Handle successful payment webhook"""
        payment_data = payload.get('data', {}).get('object', {})
        # Implement payment success logic
        return {'status': 'processed', 'payment_id': payment_data.get('id')}

    def _handle_payment_failed(self, payload: Dict[str, Any]) -> Dict[str, Any]:
        """Handle failed payment webhook"""
        payment_data = payload.get('data', {}).get('object', {})
        # Implement payment failure logic
        return {'status': 'processed', 'payment_id': payment_data.get('id')}

    def _handle_refund_succeeded(self, payload: Dict[str, Any]) -> Dict[str, Any]:
        """Handle successful refund webhook"""
        refund_data = payload.get('data', {}).get('object', {})
        # Implement refund success logic
        return {'status': 'processed', 'refund_id': refund_data.get('id')}
 
class SysPayError(Exception):
    ERROR_CODES = {
        'invalid_request': 'Invalid request parameters',
        'authentication_failed': 'Authentication failed',
        'insufficient_funds': 'Insufficient funds',
        'payment_declined': 'Payment was declined',
        'expired_card': 'Card has expired',
        'invalid_card': 'Invalid card information',
        'processing_error': 'Payment processing error',
        'invalid_amount': 'Invalid payment amount',
        'currency_not_supported': 'Currency not supported',
        'token_expired': 'Payment token has expired',
        'token_already_used': 'Payment token already used',
        'invalid_token': 'Invalid payment token'
    }

    def __init__(self, error_data):
        self.error_data = error_data
        self.error_code = error_data.get('error', {}).get('code')
        self.error_message = self.ERROR_CODES.get(
            self.error_code, 
            error_data.get('error', {}).get('message', 'Unknown error')
        )
        super().__init__(self.error_message)

        logger.error(f"SysPayError: {self.error_message}", extra={'error_data': self.error_data})

class APIConnectionError(SysPayError):
    """Raised when API connection fails"""
    pass

class UserNotFoundError(SysPayError):
    """Raised when user is not found"""
    pass

class PaymentError(SysPayError):
    """Raised when payment processing fails"""
    pass
 
class SysPaySandboxProvider:
    def __init__(self):
        self.base_url = settings.SYSPAY_SANDBOX_API_URL
        self.partner_id = settings.SYSPAY_SANDBOX_PARTNER_ID
        self.api_key = settings.SYSPAY_SANDBOX_API_KEY
        
        self.headers = {
            'Authorization': f'Bearer {self.api_key}',
            'Partner-ID': self.partner_id,
            'Content-Type': 'application/json'
        }

        self.logger = logging.getLogger(__name__)
        self.logger.setLevel(logging.DEBUG)
        handler = logging.StreamHandler()
        handler.setLevel(logging.DEBUG)
        formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
        handler.setFormatter(formatter) 