# local_secrets/payment/services.py

from decimal import Decimal
from typing import Dict, Any, Optional, List, Tuple
from django.conf import settings
from django.db import transaction
from django.utils import timezone
from django.core.cache import cache
from local_secrets.users.models import Vendor
from .models import PaymentTransaction, VendorPayment, PaymentMethod 
from .gateways.base import PaymentGatewayInterface
from .processors import BasePaymentProcessor
from .exceptions import PaymentError, PaymentServiceError
from .notifications import  PaymentNotificationService 

class BasePaymentService:
    """Base service class for payment operations"""
    
    def __init__(self, gateway: PaymentGatewayInterface):
        self.gateway = gateway
        self.notification_service = PaymentNotificationService()
        self.cache_timeout = settings.PAYMENT_CACHE_TIMEOUT

    def _get_cache_key(self, key_type: str, identifier: str) -> str:
        """Generate cache key"""
        return f"payment_service_{key_type}_{identifier}"

    def _log_transaction(self, payment: VendorPayment, 
                        transaction_type: str, details: Dict[str, Any]) -> None:
        """Log transaction details"""
        PaymentTransaction.objects.create(
            payment=payment,
            transaction_type=transaction_type,
            amount=details.get('amount'),
            status=details.get('status'),
            gateway_reference=details.get('gateway_reference'),
            response_data=details
        )

class PaymentService(BasePaymentService):
    """Service layer for handling payment operations"""

    def create_payment(self, vendor: Vendor, amount: Decimal, currency: str, payment_method: PaymentMethod, metadata: Optional[Dict[str, Any]] = None) -> VendorPayment:
        """Create and process a new payment"""
        
        self._validate_payment_request(vendor, amount, currency, payment_method)
        
        try:
            with transaction.atomic():
                # Create payment record
                payment = VendorPayment.objects.create(
                    vendor=vendor,
                    amount=amount,
                    currency=currency,
                    payment_method=payment_method,
                    metadata=metadata or {},
                    status='pending'
                )
                
                # Process payment through gateway
                result = self.gateway.process_payment(payment)
                
                # Update payment with gateway response
                self._update_payment_status(payment, result)
                
                # Log transaction
                self._log_transaction(payment, 'payment', result)
                
                # Send notifications
                self._handle_payment_notifications(payment)
                
                return payment
                
        except Exception as e:
            raise PaymentServiceError(f"Payment creation failed: {str(e)}")

    def get_payment_status(self, payment_id: str) -> Dict[str, Any]:
        """Get current payment status"""
        cache_key = self._get_cache_key('status', payment_id)
        
        # Try to get from cache first
        status = cache.get(cache_key)
        if status:
            return status
            
        try:
            status = self.gateway.get_payment_status(payment_id)
            cache.set(cache_key, status, self.cache_timeout)
            return status
        except Exception as e:
            raise PaymentServiceError(f"Failed to get payment status: {str(e)}")

    def handle_webhook(self, payload: Dict[str, Any], signature: str) -> Dict[str, Any]:
        """Handle webhook notifications from payment gateway"""
        try:
            if not self.gateway.verify_webhook(payload, signature):
                raise PaymentServiceError("Invalid webhook signature")
                
            event_type = payload.get('type')
            handler = self._get_webhook_handler(event_type)
            
            if handler:
                return handler(payload)
            else:
                return {'status': 'ignored', 'message': f'Unhandled event type: {event_type}'}
                
        except Exception as e:
            raise PaymentServiceError(f"Webhook handling failed: {str(e)}")

    def _get_webhook_handler(self, event_type: str):
        """Get appropriate webhook handler for event type"""
        handlers = {
            'payment.succeeded': self._handle_payment_succeeded,
            'payment.failed': self._handle_payment_failed,
            'refund.succeeded': self._handle_refund_succeeded,
            'dispute.created': self._handle_dispute_created,
            'dispute.resolved': self._handle_dispute_resolved
        }
        return handlers.get(event_type)

    def _validate_payment_request(self, vendor: Vendor, 
                                amount: Decimal, 
                                currency: str, 
                                payment_method: PaymentMethod) -> None:
        """Validate payment request parameters"""
        if amount <= 0:
            raise PaymentServiceError("Invalid payment amount")
            
        if currency not in settings.SUPPORTED_CURRENCIES:
            raise PaymentServiceError(f"Unsupported currency: {currency}")
            
        if not payment_method.is_valid():
            raise PaymentServiceError("Invalid payment method")
            
        if not vendor.can_process_payments():
            raise PaymentServiceError("Vendor cannot process payments")

    def _update_payment_status(self, payment: VendorPayment, 
                             result: Dict[str, Any]) -> None:
        """Update payment status based on gateway response"""
        payment.status = result.get('status', 'failed')
        payment.payment_gateway_reference = result.get('transaction_id')
        payment.response_data = result
        
        if payment.status == 'completed':
            payment.completed_at = timezone.now()
            
        payment.save()

class RefundService(BasePaymentService):
    """Service layer for handling refund operations"""

    def process_refund(self, 
                      payment: VendorPayment, 
                      amount: Decimal,
                      reason: Optional[str] = None) -> Dict[str, Any]:
        """Process a refund for the given payment"""
        self._validate_refund_request(payment, amount)
        
        try:
            with transaction.atomic():
                result = self.gateway.refund_payment(payment, amount)
                
                # Update payment record
                payment.amount_refunded = amount
                payment.refund_reason = reason
                payment.status = 'refunded' if amount == payment.amount else 'partially_refunded'
                payment.save()
                
                # Log transaction
                self._log_transaction(payment, 'refund', result)
                
                # Send notifications
                self.notification_service.send_refund_notification(payment)
                
                return result
                
        except Exception as e:
            raise PaymentServiceError(f"Refund processing failed: {str(e)}")

    def _validate_refund_request(self, payment: VendorPayment, 
                               amount: Decimal) -> None:
        """Validate refund request parameters"""
        if payment.status != 'completed':
            raise PaymentServiceError("Payment must be completed to process refund")
            
        if amount > payment.amount - payment.amount_refunded:
            raise PaymentServiceError("Refund amount exceeds available amount")
            
        if amount <= 0:
            raise PaymentServiceError("Invalid refund amount")

class DisputeService(BasePaymentService):
    """Service layer for handling payment disputes"""

    def handle_dispute(self, 
                      payment: VendorPayment, 
                      reason: str,
                      evidence: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        """Handle a new dispute"""
        try:
            with transaction.atomic():
                payment.has_dispute = True
                payment.dispute_reason = reason
                payment.dispute_created_at = timezone.now()
                payment.status = 'disputed'
                payment.save()
                
                # Log dispute
                self._log_transaction(payment, 'dispute', {
                    'reason': reason,
                    'evidence': evidence
                })
                
                # Send notifications
                self.notification_service.send_dispute_notification(payment)
                
                return {'status': 'dispute_created', 'payment_id': payment.id}
                
        except Exception as e:
            raise PaymentServiceError(f"Dispute handling failed: {str(e)}")

    def resolve_dispute(self, 
                       payment: VendorPayment, 
                       resolution: str) -> Dict[str, Any]:
        """Resolve an existing dispute"""
        try:
            with transaction.atomic():
                payment.has_dispute = False
                payment.dispute_resolved_at = timezone.now()
                payment.status = 'dispute_resolved'
                payment.save()
                
                # Log resolution
                self._log_transaction(payment, 'dispute_resolution', {
                    'resolution': resolution
                })
                
                # Send notifications
                self.notification_service.send_dispute_resolution_notification(payment)
                
                return {'status': 'dispute_resolved', 'payment_id': payment.id}
                
        except Exception as e:
            raise PaymentServiceError(f"Dispute resolution failed: {str(e)}")
