
from datetime import timezone
import logging


class PaymentErrorHandler:
    def __init__(self, payment_service):
        self.payment_service = payment_service
        self.logger = logging.getLogger(__name__)

    def handle_payment_error(self, error, payment=None):
        error_data = {
            'timestamp': timezone.now(),
            'error_type': type(error).__name__,
            'error_message': str(error)
        }
        
        if payment:
            error_data.update({
                'payment_id': payment.id,
                'vendor_id': payment.vendor.id,
                'amount': payment.amount
            })

        # Log error
        self.logger.error(
            "Payment processing error",
            extra=error_data,
            exc_info=True
        )

        # Rollback transaction if needed
        if payment and payment.status == 'processing':
            try:
                self.payment_service.rollback_payment(payment)
            except Exception as rollback_error:
                self.logger.error(
                    "Payment rollback failed",
                    extra={'original_error': str(error), 'rollback_error': str(rollback_error)}
                )

        # Notify relevant parties
        self._send_error_notifications(error_data)
        
        return error_data

    def _send_error_notifications(self, error_data):
        if error_data.get('payment_id'):
            # Notify vendor
            self._notify_vendor(error_data)
            
        # Notify admin if critical error
        if self._is_critical_error(error_data):
            self._notify_admin(error_data)
