
from typing import Tuple
from rest_framework.throttling import SimpleRateThrottle
from django.core.cache import cache


class PaymentRateThrottle:
    """Rate limiting for payment endpoints"""
    
    def __init__(self, key_prefix: str, rate: str):
        self.key_prefix = key_prefix
        self.rate = rate
        
    def allow_request(self, identifier: str) -> bool:
        """Check if request is allowed based on rate limiting"""
        cache_key = f"{self.key_prefix}:{identifier}"
        
        # Get current count
        count = cache.get(cache_key, 0)
        
        # Parse rate limit
        num_requests, period = self._parse_rate(self.rate)
        
        if count >= num_requests:
            return False
            
        # Increment counter
        cache.incr(cache_key)
        
        return True
        
    def _parse_rate(self, rate: str) -> Tuple[int, int]:
        """Parse rate string (e.g., '100/hour')"""
        num, period = rate.split('/')
        return int(num), period
