# local_secrets/payment/views.py

from datetime import timedelta
from decimal import Decimal
from pydantic import ValidationError
from rest_framework import viewsets, status
from rest_framework.decorators import action
from rest_framework.response import Response 
from django.utils import timezone 
from local_secrets.payment.webhooks import WebhookHandler
from .models import PaymentMethod, AutomaticBilling, Payment, PaymentGateway, VendorPayment
import logging
from rest_framework.throttling import UserRateThrottle
from django.views.generic import TemplateView
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.http import JsonResponse
from django.shortcuts import redirect, render
from django.views import View 
from django.views.decorators.csrf import csrf_exempt
from django.core.exceptions import PermissionDenied
from django.utils.decorators import method_decorator
import json
from rest_framework.views import APIView
from django.conf import settings
from .throttling import PaymentRateThrottle
from django.utils.decorators import method_decorator 
from rest_framework.permissions import AllowAny, IsAuthenticated
from  local_secrets.users.serializers import IsVendor
from local_secrets.payment.serializers import (
    SysPayTokenSerializer,
    SysPayPaymentMethodSerializer,
    SysPayPaymentSerializer,
    VendorPaymentSerializer,
    VendorPaymentCreateSerializer, 
    VendorPaymentListSerializer
)

from rest_framework import generics, status
from rest_framework.response import Response
from rest_framework.filters import OrderingFilter, SearchFilter
from django_filters.rest_framework import DjangoFilterBackend
from django.db.models import Q
from django.utils import timezone
from django.core.paginator import Paginator  
from .filters import PaymentFilterSet
from .pagination import StandardResultsSetPagination
from .models import VendorPayment, PaymentRefund, PaymentDispute
from .serializers import (
    VendorPaymentDetailSerializer,
    PaymentRefundSerializer,
    PaymentDisputeSerializer
)
from django.utils.decorators import method_decorator
from django.views.decorators.csrf import csrf_protect, csrf_exempt
from .exceptions import PaymentError, CardDeclinedError, PAYMENT_ERROR_MESSAGES, PaymentGatewayError
from .services import PaymentService, RefundService
from .permissions import IsVendorOwner
from django.db import transaction
import stripe
from local_secrets.users.models import VendorSubscription, SubscriptionPlan
import uuid
logger = logging.getLogger(__name__)
 
class PaymentConfigError(Exception):
    pass

class PaymentProcessError(Exception):
    pass

class TestPaymentProcessor:
    @transaction.atomic
    def process_test_payment(self, card_number, vendor, plan):
        """Process test payment and update subscription status"""
        try:
            # Validate test scenario
            scenario = self._get_test_scenario(card_number)
            
            if scenario.get('status') == 'success':
                # Create transaction record
                transaction = VendorPayment.objects.create(
                    vendor=vendor,
                    total=plan.price,
                    currency='USD',
                    status='completed',
                    transaction_id=f'test_tx_{uuid.uuid4()}',
                    payment_type='subscription',
                    description=f'Test payment for {plan.name} subscription'
                )

                # Update or create subscription
                subscription = self._update_subscription(vendor, plan, transaction)
                
                # Update vendor status
                self._update_vendor_status(vendor, subscription)
                
                return {
                    'status': 'success',
                    'transaction_id': transaction.transaction_id,
                    'subscription_id': subscription.id
                }
            else:
                raise CardDeclinedError(
                    message=scenario.get('error_message', 'Payment declined'),
                    code=scenario.get('error_code', 'card_declined')
                )

        except Exception as e:
            logger.exception("Error in test payment processing")
            raise

    def _get_test_scenario(self, card_number):
        scenarios = {
            '4242424242424242': {
                'status': 'success'
            },
            '4000000000000002': {
                'status': 'failed',
                'error_message': 'Your card was declined',
                'error_code': 'card_declined'
            }
        }
        return scenarios.get(card_number.replace(' ', ''), {
            'status': 'failed',
            'error_message': 'Invalid card number',
            'error_code': 'invalid_number'
        })

    @transaction.atomic
    def _update_subscription(self, vendor, plan, transaction):
        # Deactivate existing subscription
        VendorSubscription.objects.filter(
            vendor=vendor,
            status='active'
        ).update(
            status='cancelled',
            cancelled_at=timezone.now()
        )

        # Create new subscription
        subscription = VendorSubscription.objects.create(
            vendor=vendor,
            plan=plan,
            start_date=timezone.now(),
            end_date=timezone.now() + timedelta(days=30),
            status='active',
            payment_status='completed',
            payment_id=transaction.transaction_id,
            sites_quota_used=0,
            events_quota_used=0,
            auto_renew=True
        )

        return subscription

    def _update_vendor_status(self, vendor, subscription):
        """Update vendor and associated sites status"""
        vendor.subscription_status = 'active'
        vendor.current_subscription = subscription
        vendor.save()

        # Update all vendor's sites visibility based on quota
        sites_to_update = vendor.sites.all()[:subscription.plan.max_sites]
        vendor.sites.all().update(is_published=False)
        vendor.sites.filter(id__in=[site.id for site in sites_to_update]).update(
            is_published=True,
            subscription_status='subscribed'
        )


class TestPaymentInterface(LoginRequiredMixin, TemplateView):
    template_name = 'payment/test_interface.html'

    def get_context_data(self, **kwargs):
        if not settings.DEBUG:
            logger.warning("Attempted to access test interface in production")
            return {}

        context = super().get_context_data(**kwargs)
        
        # Get subscription plans from database
        subscription_plans = SubscriptionPlan.objects.filter(
            is_active=True
        ).order_by('price')
        billing_period = SubscriptionPlan.objects.filter()
        context.update({
            'stripe_key': settings.STRIPE_PUBLISHABLE_KEY,
            'subscription_plans': subscription_plans,
            'test_cards': self.get_test_cards()
        })
        
        return context

    def get_test_cards(self):
        return {
            'success': [
                {
                    'type': 'Visa',
                    'number': '4242 4242 4242 4242',
                    'exp_month': '12',
                    'exp_year': '2025',
                    'cvc': '123',
                    'description': 'Successful payment'
                }
            ],
            'authentication_required': [
                {
                    'type': 'Visa',
                    'number': '4000 0000 0000 3220',
                    'exp_month': '12',
                    'exp_year': '2025',
                    'cvc': '123',
                    'description': '3D Secure authentication'
                }
            ],
            'decline': [
                {
                    'type': 'Visa',
                    'number': '4000 0000 0000 0002',
                    'exp_month': '12',
                    'exp_year': '2025',
                    'cvc': '123',
                    'description': 'Declined payment'
                }
            ]
        }

@method_decorator(csrf_protect, name='dispatch')
class TestPaymentProcessView(View):
    def post(self, request, *args, **kwargs):
        try:
            data = json.loads(request.body)
            card_number = data.get('cardNumber', '').replace(' ', '')
            plan_id = data.get('planId')
            
            # Validate plan
            try:
                plan = SubscriptionPlan.objects.get(id=plan_id, is_active=True)
            except SubscriptionPlan.DoesNotExist:
                return JsonResponse({
                    'status': 'error',
                    'message': 'Invalid subscription plan'
                }, status=400)

            # Get vendor
            vendor = request.user.vendor_profile
            
            # Process payment
            processor = TestPaymentProcessor()
            result = processor.process_test_payment(card_number, vendor, plan)
            
            return JsonResponse(result)
            
        except CardDeclinedError as e:
            logger.warning(f"Test card declined: {str(e)}")
            return JsonResponse({
                'status': 'error',
                'error_type': 'card_declined',
                'message': str(e),
                'code': e.code
            }, status=402)
            
        except Exception as e:
            logger.exception("Unexpected error in test payment processing")
            return JsonResponse({
                'status': 'error',
                'message': str(e)
            }, status=500)
 
class StripeTestInterface(TemplateView):
    template_name = 'payment/stripe_test.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        stripe_key = settings.STRIPE_PUBLISHABLE_KEY

        if not stripe_key:
            logger.error("Stripe publishable key is not configured")
            context['stripe_error'] = "Payment system configuration error"
        else:
            logger.debug(f"Using Stripe key: {stripe_key[:6]}...{stripe_key[-4:]}")
            context['stripe_key'] = stripe_key

        if not settings.DEBUG:
            logger.warning("Attempted to access test interface in production")
            return context

        # Make sure this line is present
        context['stripe_key'] = settings.STRIPE_PUBLISHABLE_KEY

        # Test cards for different scenarios
        context['test_cards'] = {
            'success': [
                {
                    'type': 'Visa',
                    'number': '4242 4242 4242 4242',
                    'exp_month': '12',
                    'exp_year': '2025',
                    'cvc': '123',
                    'description': 'Successful payment'
                },
                {
                    'type': 'Mastercard',
                    'number': '5555 5555 5555 4444',
                    'exp_month': '12',
                    'exp_year': '2025',
                    'cvc': '123',
                    'description': 'Successful payment'
                }
            ],
            'authentication_required': [
                {
                    'type': 'Visa',
                    'number': '4000 0000 0000 3220',
                    'exp_month': '12',
                    'exp_year': '2025',
                    'cvc': '123',
                    'description': '3D Secure 2 authentication'
                }
            ],
            'decline': [
                {
                    'type': 'Visa',
                    'number': '4000 0000 0000 0002',
                    'exp_month': '12',
                    'exp_year': '2025',
                    'cvc': '123',
                    'description': 'Generic decline'
                },
                {
                    'type': 'Visa',
                    'number': '4000 0000 0000 9995',
                    'exp_month': '12',
                    'exp_year': '2025',
                    'cvc': '123',
                    'description': 'Insufficient funds decline'
                }
            ]
        }

        # Test subscription plans
        context['test_plans'] = [
            {
                'id': 'basic_monthly',
                'name': 'Basic Monthly',
                'price': 9.99,
                'currency': 'USD',
                'interval': 'month',
                'stripe_price_id': 'price_H5ggYwtDq4fbrJ'  # Replace with your Stripe price ID
            },
            {
                'id': 'pro_monthly',
                'name': 'Pro Monthly',
                'price': 29.99,
                'currency': 'USD',
                'interval': 'month',
                'stripe_price_id': 'price_H5ggYwtDq4fbrK'  # Replace with your Stripe price ID
            }
        ]
        
        return context

class StripeTestPaymentView(View):
    def post(self, request, *args, **kwargs):
        try:
            stripe.api_key = settings.STRIPE_SECRET_KEY
            data = json.loads(request.body)
            
            # Create payment intent
            intent = self._create_payment_intent(data)
            
            return JsonResponse({
                'client_secret': intent.client_secret,
                'status': intent.status,
            })
            
        except stripe.error.CardError as e:
            logger.warning(f"Stripe card error: {str(e)}")
            return JsonResponse({
                'error': {
                    'message': e.error.message,
                    'code': e.error.code,
                    'decline_code': e.error.decline_code,
                }
            }, status=400)
            
        except stripe.error.StripeError as e:
            logger.error(f"Stripe error: {str(e)}")
            return JsonResponse({
                'error': {
                    'message': 'An error occurred while processing your payment.',
                    'code': e.error.code,
                }
            }, status=400)
            
        except Exception as e:
            logger.exception("Unexpected error in Stripe payment processing")
            return JsonResponse({
                'error': {
                    'message': 'An unexpected error occurred.',
                }
            }, status=500)

    def _create_payment_intent(self, data):
        """Create Stripe PaymentIntent"""
        return stripe.PaymentIntent.create(
            amount=self._get_amount_in_cents(data.get('amount')),
            currency=data.get('currency', 'usd'),
            payment_method_types=['card'],
            metadata={
                'plan_id': data.get('plan_id'),
                'test_interface': 'true'
            }
        )

    def _get_amount_in_cents(self, amount):
        """Convert amount to cents"""
        return int(float(amount) * 100)

class VendorSubscriptionTestView(LoginRequiredMixin, UserPassesTestMixin, TemplateView):
    template_name = 'payment/vendor_subscription_test.html'

    def test_func(self):
        return hasattr(self.request.user, 'vendor_profile')

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        
        # Get vendor's current subscription if any
        vendor = self.request.user.vendor_profile
        current_subscription = VendorSubscription.objects.filter(
            vendor=vendor, 
            status='active'
        ).first()

        # Get available subscription plans
        subscription_plans = SubscriptionPlan.objects.filter(
            is_active=True
        ).order_by('price')

        context.update({
            'stripe_key': settings.STRIPE_PUBLISHABLE_KEY,
            'current_subscription': current_subscription,
            'subscription_plans': subscription_plans,
            'vendor': vendor,
            'test_cards': self.get_test_cards()
        })
        
        return context

    def get_test_cards(self):
        return {
            'success': {
                'number': '4242 4242 4242 4242',
                'exp_month': '12',
                'exp_year': '2025',
                'cvc': '123'
            },
            'authentication_required': {
                'number': '4000 0000 0000 3220',
                'exp_month': '12',
                'exp_year': '2025',
                'cvc': '123'
            },
            'decline': {
                'number': '4000 0000 0000 9995',
                'exp_month': '12',
                'exp_year': '2025',
                'cvc': '123'
            }
        }

class ProcessVendorSubscriptionView(LoginRequiredMixin, UserPassesTestMixin, View):
    def test_func(self):
        vendor = hasattr(self.request.user, 'vendor_profile')
        logger.info(f"Vendor access check: {vendor}")
        return vendor

    def post(self, request, *args, **kwargs):
        logger.info("Starting vendor subscription process")
        try:
            stripe.api_key = settings.STRIPE_SECRET_KEY
            data = json.loads(request.body)
            plan_id = data.get('plan_id')
            
            logger.debug(f"Received subscription request - Plan ID: {plan_id}")
            logger.debug(f"Request user: {request.user.email}")
            
            # Validate plan
            try:
                plan = SubscriptionPlan.objects.get(id=plan_id, is_active=True)
                logger.info(f"Found plan: {plan.name} (${plan.price})")
            except SubscriptionPlan.DoesNotExist:
                logger.error(f"Plan not found: {plan_id}")
                return JsonResponse({'error': 'Invalid subscription plan'}, status=400)

            vendor = request.user.vendor_profile
            logger.info(f"Processing for vendor: {vendor.company_name} (ID: {vendor.id})")

            # Check current subscription
            current_sub = VendorSubscription.objects.filter(
                vendor=vendor, 
                status='active'
            ).first()
            if current_sub:
                logger.info(f"Current subscription found: {current_sub.plan.name}")

            # Create Stripe Payment Intent
            try:
                amount = int(Decimal(plan.price) * 100)  # Convert to cents
                logger.debug(f"Creating PaymentIntent for amount: {amount} cents")
                
                intent = stripe.PaymentIntent.create(
                    amount=amount,
                    currency='USD', #plan.currency.lower(),
                    metadata={
                        'vendor_id': vendor.id,
                        'plan_id': plan.id,
                        'test_mode': 'true'
                    }
                )
                logger.info(f"PaymentIntent created: {intent.id}")

                return JsonResponse({
                    'client_secret': intent.client_secret,
                    'plan_details': {
                        'name': plan.name,
                        'price': float(plan.price),
                        'currency': 'USD'
                    }
                })

            except stripe.error.StripeError as e:
                logger.error(f"Stripe error: {str(e)}")
                return JsonResponse({'error': f"Payment processing error: {str(e)}"}, status=400)

        except json.JSONDecodeError as e:
            logger.error(f"JSON decode error: {str(e)}")
            return JsonResponse({'error': 'Invalid request data'}, status=400)
        except Exception as e:
            logger.exception("Unexpected error in subscription processing")
            return JsonResponse({'error': 'An unexpected error occurred'}, status=500)

    @transaction.atomic
    def process_successful_payment(self, vendor, plan, payment_intent):
        """Create or update vendor subscription after successful payment"""
        try:
            # Deactivate current subscription if exists
            VendorSubscription.objects.filter(
                vendor=vendor,
                status='active'
            ).update(
                status='cancelled',
                cancelled_at=timezone.now()
            )

            # Create new subscription
            subscription = VendorSubscription.objects.create(
                vendor=vendor,
                plan=plan,
                start_date=timezone.now(),
                end_date=timezone.now() + timedelta(days=30),  # Assuming monthly
                status='active',
                payment_status='completed',
                payment_id=payment_intent.id,
                auto_renew=True
            )

            logger.info(f"Created subscription for vendor {vendor.id}: {subscription.id}")
            return subscription

        except Exception as e:
            logger.exception("Error processing successful payment")
            raise

class PaymentMethodCreateView(LoginRequiredMixin, View):
    def post(self, request):
        try:
            # Create payment method
            payment_method = PaymentMethod.objects.create(
                vendor=request.user.vendor,
                type=request.POST.get('type', 'card'),  # Default to card
                token=request.POST.get('token'),
                billing_first_name=request.user.first_name,
                billing_last_name=request.user.last_name,
                billing_email=request.user.email,
                description=request.POST.get('description', '')
            )
            
            return JsonResponse({
                'status': 'success',
                'payment_method_id': payment_method.id
            })
            
        except Exception as e:
            logger.exception("Payment method creation failed")
            return JsonResponse({
                'status': 'error',
                'message': str(e)
            }, status=500)
 
class PaymentMethodListView(LoginRequiredMixin, View):
    def get(self, request):
        try:
            # Get all payment methods for the vendor
            payment_methods = PaymentMethod.objects.filter(vendor=request.user.vendor)
            
            # Serialize and return the payment methods
            serializer = SysPayPaymentMethodSerializer(payment_methods, many=True)
            return JsonResponse(serializer.data, safe=False)
            
        except Exception as e:
            logger.exception("Failed to retrieve payment methods")
            return JsonResponse({
                'status': 'error',
                'message': str(e)
            }, status=500)

class PaymentMethodDeleteView(LoginRequiredMixin, View):
    def post(self, request, payment_method_id):
        try:
            # Delete the specified payment method
            payment_method = PaymentMethod.objects.get(id=payment_method_id, vendor=request.user.vendor)
            payment_method.delete()
            
            return JsonResponse({
                'status': 'success',
                'message': 'Payment method deleted successfully'
            })
            
        except PaymentMethod.DoesNotExist:
            return JsonResponse({
                'status': 'error',
                'message': 'Payment method not found'
            }, status=404)
        except Exception as e:
            logger.exception("Failed to delete payment method")
            return JsonResponse({
                'status': 'error',
                'message': str(e)
            }, status=500)
 
class PaymentCreateView(LoginRequiredMixin, View):
    def post(self, request):
        try:
            # Create VendorPayment instance
            payment = VendorPayment.objects.create(
                vendor=request.user.vendor,
                variant=request.POST.get('gateway', 'stripe'),  # Default to stripe
                total=Decimal(request.POST.get('amount')),
                currency=request.POST.get('currency', 'USD'),
                billing_first_name=request.user.first_name,
                billing_last_name=request.user.last_name,
                billing_email=request.user.email,
                description=request.POST.get('description', '')
            )
            
            return redirect(payment.get_payment_url())
            
        except Exception as e:
            logger.exception("Payment creation failed")
            return JsonResponse({
                'status': 'error',
                'message': str(e)
            }, status=500)

class PaymentSuccessView(LoginRequiredMixin, View):
    def get(self, request, payment_id):
        try:
            payment = VendorPayment.objects.get(id=payment_id)
            if payment.status == 'confirmed':
                return JsonResponse({'status': 'success'})
            return JsonResponse({'status': 'pending'})
        except VendorPayment.DoesNotExist:
            return JsonResponse({'error': 'Payment not found'}, status=404)

class PaymentFailureView(LoginRequiredMixin, View):
    def get(self, request, payment_id):
        try:
            payment = VendorPayment.objects.get(id=payment_id)
            return JsonResponse({
                'status': 'failed',
                'error': payment.error
            })
        except VendorPayment.DoesNotExist:
            return JsonResponse({'error': 'Payment not found'}, status=404)

class VendorPaymentListView(generics.ListAPIView):
    """
    List view for vendor payments with filtering, searching and ordering
    """
    serializer_class = VendorPaymentListSerializer
    permission_classes = [IsAuthenticated]
    pagination_class = StandardResultsSetPagination
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_class = PaymentFilterSet
    search_fields = ['description', 'transaction_id', 'billing_email']
    ordering_fields = ['created', 'modified', 'total', 'status']
    ordering = ['-created']  # Default ordering

    def get_queryset(self):
        """
        Get payments queryset with optimized queries and filters
        """
        vendor = self.request.user.vendor_profile
        
        # Base queryset with select_related for foreign keys
        queryset = VendorPayment.objects.select_related(
            'vendor',
            'subscription',
            'payment_method'
        ).filter(vendor=vendor)

        # Add date range filters if provided
        start_date = self.request.query_params.get('start_date')
        end_date = self.request.query_params.get('end_date')
        
        if start_date:
            queryset = queryset.filter(created__gte=start_date)
        if end_date:
            queryset = queryset.filter(created__lte=end_date)

        # Status filter
        status = self.request.query_params.get('status')
        if status:
            queryset = queryset.filter(status=status)

        # Payment type filter
        payment_type = self.request.query_params.get('payment_type')
        if payment_type:
            queryset = queryset.filter(payment_type=payment_type)

        # Amount range filter
        min_amount = self.request.query_params.get('min_amount')
        max_amount = self.request.query_params.get('max_amount')
        
        if min_amount:
            queryset = queryset.filter(total__gte=float(min_amount))
        if max_amount:
            queryset = queryset.filter(total__lte=float(max_amount))

        return queryset

    def list(self, request, *args, **kwargs):
        """
        Enhanced list method with additional metadata
        """
        try:
            # Get filtered queryset
            queryset = self.filter_queryset(self.get_queryset())
            
            # Calculate totals for different payment statuses
            stats = {
                'total_payments': queryset.count(),
                'total_amount': sum(payment.total for payment in queryset),
                'completed_payments': queryset.filter(status='completed').count(),
                'pending_payments': queryset.filter(status='pending').count(),
                'failed_payments': queryset.filter(status='failed').count()
            }

            # Paginate results
            page = self.paginate_queryset(queryset)
            if page is not None:
                serializer = self.get_serializer(page, many=True)
                response = self.get_paginated_response(serializer.data)
                response.data['stats'] = stats
                return response

            serializer = self.get_serializer(queryset, many=True)
            return Response({
                'results': serializer.data,
                'stats': stats
            })

        except Exception as e:
            return Response({
                'error': str(e)
            }, status=status.HTTP_400_BAD_REQUEST)

class VendorPaymentDetailView(generics.RetrieveAPIView):
    """
    Detailed view for a specific vendor payment with related information
    and action handlers for refunds, disputes, and receipts
    """
    serializer_class = VendorPaymentDetailSerializer
    permission_classes = [IsAuthenticated, IsVendorOwner]
    lookup_field = 'id'

    def get_queryset(self):
        """
        Get the base queryset for the payment with all related data
        """
        return VendorPayment.objects.select_related(
            'vendor',
            'subscription',
            'payment_method',
            'billing_address'
        ).prefetch_related(
            'refunds',
            'disputes',
            'transactions'
        ).filter(
            vendor=self.request.user.vendor_profile
        )

    def get_object(self):
        """
        Get the payment object with additional validation
        """
        payment = super().get_object()
        
        # Additional validation if needed
        if payment.vendor != self.request.user.vendor_profile:
            raise PermissionDenied("You don't have permission to view this payment")
            
        return payment

    def retrieve(self, request, *args, **kwargs):
        """
        Enhanced retrieve method with additional payment information
        """
        try:
            payment = self.get_object()
            serializer = self.get_serializer(payment)
            
            # Get additional payment details from payment processor
            payment_service = PaymentService()
            external_payment_info = payment_service.get_payment_details(payment.transaction_id)
            
            response_data = {
                **serializer.data,
                'external_payment_info': external_payment_info,
                'can_refund': payment.can_be_refunded(),
                'can_dispute': payment.can_be_disputed(),
                'related_documents': self.get_related_documents(payment)
            }
            
            return Response(response_data)
            
        except PaymentProcessError as e:
            return Response({
                'error': str(e)
            }, status=status.HTTP_400_BAD_REQUEST)
        except Exception as e:
            return Response({
                'error': 'An unexpected error occurred'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    @action(detail=True, methods=['post'])
    def refund(self, request, *args, **kwargs):
        """
        Handle payment refund requests
        """
        payment = self.get_object()
        
        try:
            with transaction.atomic():
                # Validate refund possibility
                if not payment.can_be_refunded():
                    raise PaymentProcessError("Payment cannot be refunded")

                amount = request.data.get('amount')
                reason = request.data.get('reason')
                
                refund_service = RefundService()
                refund = refund_service.process_refund(
                    payment=payment,
                    amount=amount,
                    reason=reason,
                    user=request.user
                )
                
                return Response(PaymentRefundSerializer(refund).data)
                
        except PaymentProcessError as e:
            return Response({
                'error': str(e)
            }, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=['post'])
    def dispute(self, request, *args, **kwargs):
        """
        Handle payment dispute creation
        """
        payment = self.get_object()
        
        try:
            if not payment.can_be_disputed():
                raise PaymentProcessError("Payment cannot be disputed")

            dispute_data = {
                'payment': payment,
                'reason': request.data.get('reason'),
                'description': request.data.get('description'),
                'evidence': request.data.get('evidence', [])
            }
            
            dispute = PaymentDispute.objects.create(**dispute_data)
            return Response(PaymentDisputeSerializer(dispute).data)
            
        except PaymentProcessError as e:
            return Response({
                'error': str(e)
            }, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=['get'])
    def receipt(self, request, *args, **kwargs):
        """
        Generate and return payment receipt
        """
        payment = self.get_object()
        
        try:
            receipt_service = PaymentService()
            receipt_url = receipt_service.generate_receipt(payment)
            
            return Response({
                'receipt_url': receipt_url
            })
            
        except Exception as e:
            return Response({
                'error': 'Failed to generate receipt'
            }, status=status.HTTP_400_BAD_REQUEST)

    def get_related_documents(self, payment):
        """
        Get all related documents for the payment
        """
        return {
            'invoices': payment.get_invoices(),
            'receipts': payment.get_receipts(),
            'statements': payment.get_statements()
        }
 
class WebhookView(APIView):
    """Webhook endpoint for payment gateways"""
    
    @method_decorator(csrf_exempt)
    def post(self, request, gateway_name):
        try:
            handler = WebhookHandler(gateway_name)
            result = handler.process_webhook(request)
            
            return Response(
                data={'status': 'success', 'result': result},
                status=status.HTTP_200_OK
            )
            
        except PaymentError as e:
            logger.error(f"Payment error in webhook: {str(e)}")
            return Response(
                data={'status': 'error', 'message': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )
            
        except Exception as e:
            logger.error(f"Unexpected error in webhook: {str(e)}")
            return Response(
                data={'status': 'error', 'message': 'Internal server error'},
                status=status.HTTP_500_INTERNAL_SERVER_ERROR
            )
 
class PaymentRateThrottle(UserRateThrottle):
    rate = '100/day'
 
class PaymentListView(generics.ListAPIView):
    """List view for vendor payments with filtering, searching and ordering"""
    serializer_class = VendorPaymentListSerializer
    permission_classes = [IsAuthenticated]
    pagination_class = StandardResultsSetPagination
    filter_backends = [DjangoFilterBackend, SearchFilter, OrderingFilter]
    filterset_class = PaymentFilterSet
    search_fields = ['description', 'transaction_id', 'billing_email']
    ordering_fields = ['created', 'modified', 'total', 'status']
    ordering = ['-created']  # Default ordering

    def get_queryset(self):
        """Get payments queryset with optimized queries and filters"""
        vendor = self.request.user.vendor_profile
        
        # Base queryset with select_related for foreign keys
        queryset = VendorPayment.objects.select_related(
            'vendor',
            'subscription',
            'payment_method'
        ).filter(vendor=vendor)

        # Add date range filters if provided
        start_date = self.request.query_params.get('start_date')
        end_date = self.request.query_params.get('end_date')
        
        if start_date:
            queryset = queryset.filter(created__gte=start_date)
        if end_date:
            queryset = queryset.filter(created__lte=end_date)

        # Status filter
        status = self.request.query_params.get('status')
        if status:
            queryset = queryset.filter(status=status)

        # Payment type filter
        payment_type = self.request.query_params.get('payment_type')
        if payment_type:
            queryset = queryset.filter(payment_type=payment_type)

        # Amount range filter
        min_amount = self.request.query_params.get('min_amount')
        max_amount = self.request.query_params.get('max_amount')
        
        if min_amount:
            queryset = queryset.filter(total__gte=float(min_amount))
        if max_amount:
            queryset = queryset.filter(total__lte=float(max_amount))

        return queryset

class PaymentDetailView(generics.RetrieveAPIView):
    """Detailed view for a specific vendor payment with related information"""
    serializer_class = VendorPaymentDetailSerializer
    permission_classes = [IsAuthenticated, IsVendorOwner]
    lookup_field = 'id'

    def get_queryset(self):
        """Get the base queryset for the payment with all related data"""
        return VendorPayment.objects.select_related(
            'vendor',
            'subscription',
            'payment_method',
            'billing_address'
        ).prefetch_related(
            'refunds',
            'disputes',
            'transactions'
        ).filter(
            vendor=self.request.user.vendor_profile
        )

    def get_object(self):
        """Get the payment object with additional validation"""
        payment = super().get_object()
        
        # Additional validation if needed
        if payment.vendor != self.request.user.vendor_profile:
            raise PermissionDenied("You don't have permission to view this payment")
            
        return payment
 
    def retrieve(self, request, *args, **kwargs):
        """Enhanced retrieve method with additional payment information"""
        try:
            payment = self.get_object()
            serializer = self.get_serializer(payment)
            
            # Get additional payment details from payment processor
            payment_service = PaymentService()
            external_payment_info = payment_service.get_payment_details(payment.transaction_id)
            
            response_data = {
                **serializer.data,
                'external_payment_info': external_payment_info,
                'can_refund': payment.can_be_refunded(),
                'can_dispute': payment.can_be_disputed(),
                'related_documents': self.get_related_documents(payment)
            }
            
            return Response(response_data)
            
        except PaymentProcessError as e:
            return Response({
                'error': str(e)
            }, status=status.HTTP_400_BAD_REQUEST)
        except Exception as e:
            return Response({
                'error': 'An unexpected error occurred'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
 
class PaymentRefundView(generics.CreateAPIView):
    """View for creating payment refunds"""
    serializer_class = PaymentRefundSerializer
    permission_classes = [IsAuthenticated, IsVendorOwner]

    def perform_create(self, serializer):
        """Override to handle refund creation"""
        payment = self.get_object()
        
        # Validate and process refund
        if not payment.can_be_refunded():
            raise PaymentProcessError("Payment cannot be refunded")
        
        refund_service = RefundService()
        refund = refund_service.process_refund(
            payment=payment,
            amount=serializer.validated_data['amount'],
            reason=serializer.validated_data['reason'],
            user=self.request.user
        )
        
        return Response(PaymentRefundSerializer(refund).data)
 
class PaymentMethodViewSet(viewsets.ViewSet):
    """ViewSet for managing payment methods"""
    throttle_classes = [PaymentRateThrottle]
    permission_classes = [IsAuthenticated]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.logger = logging.getLogger(__name__)

    def get_queryset(self):
        return PaymentMethod.objects.filter(vendor=self.request.user.vendor_profile)

class PaymentViewSet(viewsets.ViewSet):
    throttle_classes = [PaymentRateThrottle]
    permission_classes = [IsAuthenticated]
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.logger = logging.getLogger(__name__)
    
    def get_queryset(self):
        return VendorPayment.objects.filter(vendor=self.request.user.vendor)
    
    def get_serializer_class(self):
        if self.action in ['create', 'update']:
            return VendorPaymentCreateSerializer
        return VendorPaymentSerializer
    
    @action(detail=False, methods=['post'])
    def add_payment_method(self, request):
        try:
            variant = request.data.get('gateway')
            if variant not in dict(PaymentGateway.choices):
                raise ValidationError("Invalid payment gateway")
                
            payment = VendorPayment.objects.create(
                vendor=request.user.vendor,
                variant=variant,
                total=0,  # Token creation doesn't require amount
                currency=request.data.get('currency', 'USD'),
                billing_first_name=request.user.first_name,
                billing_last_name=request.user.last_name,
                billing_email=request.user.email,
                token=request.data.get('token')
            )
            
            return Response({
                'payment_method_id': payment.id,
                'redirect_url': payment.get_payment_url()
            })
            
        except Exception as e:
            self.logger.error("Error adding payment method", extra={
                'user_id': request.user.id,
                'error': str(e)
            })
            return self.handle_payment_error(e, {
                'action': 'add_payment_method',
                'user_id': request.user.id
            })

    @action(detail=True, methods=['delete'])
    def remove_payment_method(self, request, pk=None):
        """Remove a payment method"""
        try:
            self.logger.info("Removing payment method", extra={
                'user_id': request.user.id,
                'payment_method_id': pk
            })
            
            payment_method = PaymentMethod.objects.get(
                id=pk,
                vendor=request.user.vendor_profile
            )
            
            if payment_method.is_default:
                raise PaymentConfigError("Cannot delete default payment method")
            
            payment_method.delete()
            
            self.logger.info("Payment method removed successfully", extra={
                'user_id': request.user.id,
                'payment_method_id': pk
            })
            
            return Response(status=status.HTTP_204_NO_CONTENT)
            
        except Exception as e:
            return self.handle_payment_error(e, {
                'action': 'remove_payment_method',
                'payment_method_id': pk,
                'user_id': request.user.id
            })
    
    @action(detail=False, methods=['post'])
    def setup_automatic_billing(self, request):
        """Setup automatic billing"""
        try:
            self.logger.info("Setting up automatic billing", extra={
                'user_id': request.user.id
            })
            
            vendor = request.user.vendor_profile
            payment_method_id = request.data.get('payment_method_id')
            billing_day = request.data.get('billing_day', 1)
            
            payment_method = PaymentMethod.objects.get(
                id=payment_method_id,
                vendor=vendor
            )
            
            AutomaticBilling.objects.update_or_create(
                vendor=vendor,
                defaults={
                    'enabled': True,
                    'billing_day': billing_day,
                    'payment_method': payment_method,
                    'next_billing_date': timezone.now().date()
                }
            )
            
            self.logger.info("Automatic billing setup complete", extra={
                'user_id': request.user.id,
                'payment_method_id': payment_method_id
            })
            
            return Response({'status': 'automatic billing enabled'})
            
        except Exception as e:
            return self.handle_payment_error(e, {
                'action': 'setup_automatic_billing',
                'user_id': request.user.id
            })
    
    def handle_payment_error(self, error, context=None):
        """Centralized error handling for payment operations"""
        error_msg = str(error)
        error_data = {
            'error': error_msg,
            'context': context
        }
        
        self.logger.error(
            f"Payment error: {error_msg}",
            extra={'context': context},
            exc_info=True
        )
        
        if isinstance(error, PaymentMethod.DoesNotExist):
            return Response(
                error_data,
                status=status.HTTP_404_NOT_FOUND
            )
        elif isinstance(error, (ValueError, PaymentConfigError)):
            return Response(
                error_data,
                status=status.HTTP_400_BAD_REQUEST
            )
        else:
            return Response(
                {'error': 'Internal server error'},
                status=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

class SubscriptionFeatures:
    @staticmethod
    def check_feature_access(vendor, feature_name):
        if not vendor.get_active_subscription():
            return False
            
        plan_features = vendor.subscriptions.filter(
            status='active'
        ).first().plan.features
        
        return feature_name in plan_features.get('allowed_features', [])

    @staticmethod
    def get_feature_limit(vendor, feature_name):
        if not vendor.get_active_subscription():
            return 0
            
        plan_features = vendor.subscriptions.filter(
            status='active'
        ).first().plan.features
        
        return plan_features.get('limits', {}).get(feature_name, 0)

    def _validate_billing_day(self, billing_day):
        """Validate billing day is between 1 and 28"""
        try:
            day = int(billing_day)
            if not 1 <= day <= 28:
                raise ValueError("Billing day must be between 1 and 28")
            return day
        except (TypeError, ValueError):
            raise ValueError("Invalid billing day")

    def _calculate_next_billing_date(self, billing_day):
        """Calculate next billing date based on billing day"""
        today = timezone.now().date()
        if today.day > billing_day:
            next_date = today.replace(
                day=billing_day,
                month=today.month + 1 if today.month < 12 else 1,
                year=today.year + 1 if today.month == 12 else today.year
            )
        else:
            next_date = today.replace(day=billing_day)
        return next_date

    @action(detail=False, methods=['post'])
    def setup_automatic_billing(self, request):
        vendor = request.user.vendor_profile
        payment_method_id = request.data.get('payment_method_id')
        billing_day = request.data.get('billing_day', 1)

        try:
            billing_day = self._validate_billing_day(billing_day)
            
            payment_method = PaymentMethod.objects.get(
                id=payment_method_id,
                vendor=vendor,
                is_active=True
            )

            next_billing_date = self._calculate_next_billing_date(billing_day)

            automatic_billing, created = AutomaticBilling.objects.update_or_create(
                vendor=vendor,
                defaults={
                    'enabled': True,
                    'billing_day': billing_day,
                    'payment_method': payment_method,
                    'next_billing_date': next_billing_date,
                    'last_updated': timezone.now()
                }
            )

            return Response({
                'status': 'automatic billing enabled',
                'next_billing_date': next_billing_date,
                'payment_method': {
                    'id': payment_method.id,
                    'type': payment_method.payment_type,
                    'last4': payment_method.last4
                }
            })
        except PaymentMethod.DoesNotExist:
            return Response(
                {'error': 'Invalid payment method'},
                status=status.HTTP_400_BAD_REQUEST
            )
        except ValueError as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )
 
class SubscriptionFeatureViewSet(viewsets.ViewSet):
    def _get_active_subscription(self, vendor):
        """Get vendor's active subscription with cached features"""
        from django.core.cache import cache
        
        cache_key = f'vendor_subscription_{vendor.id}'
        subscription = cache.get(cache_key)
        
        if not subscription:
            subscription = vendor.subscriptions.filter(
                status='active'
            ).select_related('plan').first()
            if subscription:
                cache.set(cache_key, subscription, timeout=3600)
        
        return subscription


    @action(detail=False, methods=['get'])
    def check_feature_access(self, request):
        """Check if vendor has access to specific feature"""
        vendor = request.user.vendor_profile
        feature_name = request.query_params.get('feature')
        
        if not feature_name:
            return Response(
                {'error': 'Feature name is required'},
                status=status.HTTP_400_BAD_REQUEST
            )
        
        subscription = self._get_active_subscription(vendor)
        if not subscription:
            return Response({
                'has_access': False,
                'reason': 'No active subscription'
            })
        
        features = subscription.plan.features
        has_access = feature_name in features.get('allowed_features', [])
        limit = features.get('limits', {}).get(feature_name, 0)
        
        return Response({
            'has_access': has_access,
            'feature_limit': limit,
            'subscription_plan': subscription.plan.name
        })

    def _validate_payment_method(self, payment_method):
        """Validate payment method status and expiration"""
        if not payment_method.is_active:
            raise ValueError("Payment method is inactive")
            
        if payment_method.payment_type == 'card' and payment_method.expiry_date:
            if payment_method.expiry_date < timezone.now().date():
                raise ValueError("Payment method has expired")
                
        return True
 
 