# scripts/translate_utils.py

import polib
import openai
from pathlib import Path
from django.conf import settings
from typing import Any, List, Dict
import time
import asyncio
import aiohttp
import logging
from typing import Optional, Union
from dataclasses import dataclass
from concurrent.futures import ThreadPoolExecutor
from functools import partial
from tqdm import tqdm

logger = logging.getLogger(__name__)
 
@dataclass
class TranslationConfig:
    api_key: str
    base_locale_path: Path
    batch_size: int = 10
    rate_limit_delay: float = 1.0
    model: str = 'gpt-3.5-turbo'
    max_retries: int = 3
    timeout: int = 30
    supported_languages: Dict[str, str] = None

    def __post_init__(self):
        self.supported_languages = self.supported_languages or {
            'fr': 'French',
            'de': 'German',
            'pt': 'Portuguese',
            'po': 'Portuguese',
            'es': 'Spanish',
            'en': 'English'
        }

class TranslationError(Exception):
    """Custom exception for translation errors"""
    pass

async def handle_translation_error(self, error: Exception, entry: polib.POEntry) -> polib.POEntry:
    """Handle translation errors gracefully"""
    logger.error(f"Translation error for '{entry.msgid}': {str(error)}")
    return polib.POEntry(
        msgid=entry.msgid,
        msgstr=entry.msgid,
        occurrences=entry.occurrences,
        flags=['fuzzy']  # Mark as needs review
    )

class RateLimiter:
    def __init__(self, calls_per_minute: int = 60):
        self.calls_per_minute = calls_per_minute
        self.calls: List[float] = []
        self._lock = asyncio.Lock()

    async def acquire(self):
        async with self._lock:
            now = time.time()
            self.calls = [call for call in self.calls if call > now - 60]
            
            if len(self.calls) >= self.calls_per_minute:
                wait_time = self.calls[0] - (now - 60)
                await asyncio.sleep(wait_time)
            
            self.calls.append(now)

class ProgressTracker:
    def __init__(self, total: int, desc: str):
        self.pbar = tqdm(total=total, desc=desc)
    
    def update(self, n: int = 1):
        self.pbar.update(n)
    
    def close(self):
        self.pbar.close()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()

class OpenAITranslator:
    def __init__(self, config: TranslationConfig):
        self.config = config
        self.client = openai.AsyncOpenAI(api_key='')
        self.rate_limiter = RateLimiter()
        self._examples_cache: Dict[str, List[Dict[str, str]]] = {}

    async def translate_with_retry(self, text: str, target_lang: str, context: Optional[str] = None) -> str:
        for attempt in range(self.config.max_retries):
            try:
                await self.rate_limiter.acquire()
                return await self.translate_text(text, target_lang, context)
            except Exception as e:
                if attempt == self.config.max_retries - 1:
                    logger.error(f"Failed to translate after {self.config.max_retries} attempts: {str(e)}")
                    raise TranslationError(f"Translation failed: {str(e)}")
                await asyncio.sleep(2 ** attempt)  # Exponential backoff

    def _get_language_specific_examples(self, target_lang: str) -> List[Dict[str, Any]]:
        if target_lang in self._examples_cache:
            return self._examples_cache[target_lang]
        
        examples = {
            "es": 
            [
                {
                "role": "user",
                "content": "Please confirm your email address at %(site_name)s."
                },
                {
                "role": "assistant",
                "content": "Por favor, confirma tu dirección de correo electrónico en %(site_name)s."
                },
                {
                "role": "user",
                "content": "Invalid username or password. <strong>Try again</strong>."
                },
                {
                "role": "assistant",
                "content": "Nombre de usuario o contraseña inválidos. <strong>Intenta de nuevo</strong>."
                }
            ],
            "fr": [
                {
                "role": "user",
                "content": "You have %(count)d new message(s)."
                },
                {
                "role": "assistant",
                "content": "Vous avez %(count)d nouveau(x) message(s)."
                }
            ]
 
            }
        self._examples_cache[target_lang] = examples.get(target_lang, [])
        return self._examples_cache[target_lang]
 
    async def translate_text(self, text: str, target_lang: str, context: str = None) -> str:
        """
        Translate text using OpenAI API with specific handling for Django localization files.
        Args:
            text: Text to translate.
            target_lang: Target language code (e.g., 'es', 'fr', 'de').
            context: Optional context about where the string is used.
        Returns:
            Translated text.
        """
        try:
            system_prompt = f"""
            You are a specialized translator for Django web applications. Follow these critical rules:

            PRESERVE ALL TECHNICAL ELEMENTS:
            Keep all Django template variables: %(name)s, %(count)d, {0}, {1}, etc.
            Maintain HTML tags: <strong>, <em>, <span>, etc.
            Keep all Python string formatting: %s, %d, %f
            Preserve all placeholder formats:  user ,  date, etc.

            CONTEXT AWARENESS:
            Maintain the same technical meaning as the original.
            Consider the provided context about where the string appears.
            Keep URLs, code references, and technical terms unchanged.
            Preserve any pluralization handling formats.

            TRANSLATION GUIDELINES:
            Translate only the human-readable text.
            Keep similar length when possible for UI elements.
            Maintain the same tone (technical, user-friendly, error message, etc.).
            Keep capitalizations consistent with the original.

            STRICTLY FORBIDDEN:
            DO NOT translate variable names or technical terms.
            DO NOT modify any formatting syntax.
            DO NOT add or remove placeholders.
            DO NOT change HTML structure.

            OUTPUT FORMAT:
            Return ONLY the translated text.
            Maintain exact same formatting as input.
            Keep line breaks and spacing identical.

            Target Language: {target_lang}
            """

            messages = [
                {
                    "role": "system",
                    "content": system_prompt
                }
            ]
            if context:
                messages.append({
                    "role": "user",
                    "content": f"Context: {context}\nText to translate: {text}"
                })
            else:
                messages.append({
                    "role": "user",
                    "content": text
                })
 
            examples = self._get_language_specific_examples(target_lang)
            if examples:
                messages.extend(examples)

            response = await self.client.chat.completions.create(
                model=self.config.model,  # Use the model from config
                messages=messages,
                temperature=0.2,
                max_tokens=250,
                presence_penalty=0.0,
                frequency_penalty=0.0
            )
            return response.choices[0].message.content.strip()
        except Exception as e:
            logger.error(f"Translation error: {str(e)}")
            return text

 
class TranslationManager:
    def __init__(self, config: TranslationConfig):
        self.config = config
        self.translator = OpenAITranslator(config)
        
    async def translate_batch(self, texts: List[str], target_lang: str) -> List[str]:
        """Translate a batch of texts concurrently"""
        async with aiohttp.ClientSession() as session:
            tasks = [
                self.translator.translate_text(text, self.language_names[target_lang])
                for text in texts
            ]
            return await asyncio.gather(*tasks)

    async def process_po_entry(self, entry: polib.POEntry, target_lang: str) -> polib.POEntry:
        try:
            translated_text = await self.translator.translate_with_retry(
                entry.msgid,
                target_lang,
                context=entry.comment
            )
            
            new_entry = polib.POEntry(
                msgid=entry.msgid,
                msgstr=translated_text,
                occurrences=entry.occurrences,
                comment=entry.comment,
                flags=entry.flags
            )

            if entry.msgid_plural:
                plural_translated = await self.translator.translate_with_retry(
                    entry.msgid_plural,
                    target_lang,
                    context=f"Plural form of: {entry.msgid}"
                )
                new_entry.msgid_plural = entry.msgid_plural
                new_entry.msgstr_plural = {0: translated_text, 1: plural_translated}

            return new_entry
        except TranslationError as e:
            logger.error(f"Failed to translate entry '{entry.msgid}': {str(e)}")
            return polib.POEntry(
                msgid=entry.msgid,
                msgstr=entry.msgid,
                occurrences=entry.occurrences,
                flags=['fuzzy']
            )

    async def translate_po_file(self, source_lang: str = 'en', target_lang: str = 'fr'):
        if target_lang not in self.config.supported_languages:
            raise ValueError(f"Unsupported target language: {target_lang}")

        source_po_path = self.config.base_locale_path / source_lang / 'LC_MESSAGES' / 'django.po'
        target_po_path = self.config.base_locale_path / target_lang / 'LC_MESSAGES' / 'django.po'

        if not source_po_path.exists():
            raise FileNotFoundError(f"Source PO file not found: {source_po_path}")

        source_po = polib.pofile(str(source_po_path))
        target_po = polib.POFile()
        target_po.metadata = source_po.metadata.copy()
        target_po.metadata['Language'] = target_lang

        entries = [entry for entry in source_po if entry.msgid]
        
        with ProgressTracker(len(entries), f"Translating to {target_lang}") as pbar:
            for i in range(0, len(entries), self.config.batch_size):
                batch = entries[i:i + self.config.batch_size]
                tasks = [self.process_po_entry(entry, target_lang) for entry in batch]
                translated_entries = await asyncio.gather(*tasks)
                
                for entry in translated_entries:
                    target_po.append(entry)
                    pbar.update()

                await asyncio.sleep(self.config.rate_limit_delay)

        # Ensure the target directory exists
        target_po_path.parent.mkdir(parents=True, exist_ok=True)
        
        # Save translations
        target_po.save(str(target_po_path))
        target_po.save_as_mofile(str(target_po_path.parent / 'django.mo'))
        
        logger.info(f"Translation completed for {target_lang}")

    async def translate_all_languages(self, source_lang: str = 'en', target_langs: List[str] = None):
        """Translate to multiple target languages"""
        if target_langs is None:
            target_langs = ['fr', 'de']
            
        for target_lang in target_langs:
            await self.translate_po_file(source_lang, target_lang)
    
    async def update_existing_translations(self, languages: List[str] = None):
        """Update existing translations with new strings"""
        if languages is None:
            languages = ['fr', 'de', 'po']
        
        for lang in languages:
            po_path = self.base_locale_path / lang / 'LC_MESSAGES' / 'django.po'
            if po_path.exists():
                po = polib.pofile(str(po_path))
                untranslated = [entry for entry in po if not entry.msgstr]
                
                if untranslated:
                    print(f"Translating {len(untranslated)} missing strings for {lang}")
                    texts_to_translate = [entry.msgid for entry in untranslated]
                    
                    try:
                        translated_texts = await self.translate_batch(texts_to_translate, lang)
                        
                        for entry, translated_text in zip(untranslated, translated_texts):
                            entry.msgstr = translated_text
                            
                    except Exception as e:
                        print(f"Error updating translations: {str(e)}")
                
                # Save updated translations
                po.save()
                po.save_as_mofile(str(po_path.parent / 'django.mo'))
                print(f"Updated translations for {lang}")

"""
python manage.py translate_messages --languages pt

"""