# local_secrets/searches/services/search_service.py
from django.conf import settings
from django.db.models import Q
from local_secrets.sites.models import Site
import logging

logger = logging.getLogger(__name__)

class SearchIndexService:
    """
    Manages search indexing for sites
    """
    def index_site(self, site: Site) -> bool:
        """
        Indexes a site for search
        """
        try:
            # Update search fields
            site.search_vector = self._generate_search_vector(site)
            site.save(update_fields=['search_vector', 'updated_at'])
            
            # Update related indexed content
            self._index_related_content(site)
            
            return True
        except Exception as e:
            logger.error(f"Failed to index site {site.id}: {str(e)}")
            return False

    def remove_site(self, site: Site) -> bool:
        """
        Removes site from search index
        """
        try:
            # Clear search fields
            site.search_vector = None
            site.save(update_fields=['search_vector', 'updated_at'])
            
            # Remove related indexed content
            self._remove_related_content(site)
            
            return True
        except Exception as e:
            logger.error(f"Failed to remove site {site.id} from index: {str(e)}")
            return False

    def _generate_search_vector(self, site: Site) -> dict:
        """
        Generates search vector for site
        """
        return {
            'title': site.title,
            'description': site.description,
            'tags': list(site.tags.values_list('name', flat=True)),
            'city': site.city.name if site.city else None,
            'address': str(site.address) if site.address else None
        }

    def _index_related_content(self, site: Site):
        """
        Indexes related content
        """
        # Index comments
        for comment in site.comments.all():
            comment.update_search_vector()

        # Index images
        for image in site.images.all():
            image.update_search_vector()

    def _remove_related_content(self, site: Site):
        """
        Removes related content from index
        """
        # Clear comment search vectors
        site.comments.update(search_vector=None)
        
        # Clear image search vectors
        site.images.update(search_vector=None)
