"""Custom servie apis"""
import logging
import asyncio
from django.db import transaction
from django.http import Http404
from django.db.models import F
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework import status
from drf_yasg.utils import swagger_auto_schema
from drf_yasg import openapi
from local_secrets.sites.models import (
    Site, Level, Category, SubCategory, CustomUser, Day, HourRange,
    TranslatedSite, TranslatedLevel, TranslatedCategory, TranslatedSubCategory)
from local_secrets.cities.models import Country, City, Address, TranslatedCountry, TranslatedCity
from local_secrets.languages.models import Language
from local_secrets.service_integration.helper import (
    OperationTimings, UploadSiteImages, QueryFilterGenerator, TagLink)
from local_secrets.core import constants as const
from local_secrets.core.exception_handlers import log_exception
from local_secrets.service_integration.decorators import validate_signature, language_required
from local_secrets.service_integration.swagger_doc.serializer import (
    AutomationMappingSerializer, LanguageSerializer, CountrySerializer,
    CitySerializer, LevelSerializer, CategorySerializer, SubCategorySerializer)
from local_secrets.service_integration.swagger_doc.helper import manual_params
from utils.nominatim import Nominatim
from local_secrets.cities.helpers.city_helper import DataMigration

logger = logging.getLogger(__name__)


class MoveToApp(APIView):
    """
    Class responsible for automating the process of mapping fields from automation 
    service to a local secret service.
    """

    @swagger_auto_schema(
        operation_id="Move to Production",
        operation_description="Responsible for automating the process of mapping fields from automation service to a local secret service",
        manual_parameters=manual_params(),
        request_body=AutomationMappingSerializer, responses={200: "Success"},)
    @validate_signature(topic="move-to-app")
    def post(self, request):
        """
        Keys in the payload:

            level_id -> 
                The level_id would be the reference value found in the local_secret 
                Level table, and it is stored in the automation 
                table for reference matching.

            category_id -> 
                The category_id would be the reference value found in the local_secret 
                Category table, and it is stored in the automation 
                table for reference matching.

            sub_category_id -> 
                The sub_category_id would be the reference value found in the local_secret 
                Subcategory table, and it is stored in the automation 
                table for reference matching.

            city_id -> 
                The city_id would be the reference value found in the local_secret 
                City table, and it is stored in the automation 
                table for reference matching.

            country_id -> 
                The sub_category_id would be the reference value found in the local_secret 
                Country table, and it is stored in the automation 
                table for reference matching.
        """
        try:
            business_data = request.data

            # Initialize variables
            latitude = business_data.get("latitude")
            longitude = business_data.get("longitude")
            buisness_id = business_data.get("id")
            user_data = business_data.get("user")

            geo_service = Nominatim()

            field_mapping = {
                "title": "title",
                "description_esp": "description",
                "phone": "phone",
                "website": "url",
            }
            language_mapping = {
                "fr": ("title_fr", "description_fr", "Français"),
                "en-GB": ("title_eng", "description_eng", "English (UK)"),
                "en": ("title", "description", "English (US)"),
            }
            tag_mapping = {
                "es": ("types_esp", "Español"),
                "fr": ("types_fr", "Français"),
                "en-GB": ("types_eng", "English (UK)"),
            }
            user_fields = [
                "email", "username", "first_name", "last_name", "password",
                "is_staff", "is_active", "is_superuser"]

            sub_category_obj = None
            hour_objects = []
            business_filter = QueryFilterGenerator(business_data)

            # Validate fields
            if not (street := business_data.get("street")):
                raise ValueError("Street address is required.")
            if not (postal_code := business_data.get("postal_code")):
                raise ValueError("Postal code is required.")

            # Initialize transaction
            with transaction.atomic():
                site_obj = Site()
                for buisness_field, site_field in field_mapping.items():
                    setattr(
                        site_obj, site_field, business_data.get(
                            buisness_field, None))

                # Retreive country
                try:
                    result = business_filter.get_query_filter("country")
                    Country.objects.get(**result.query_filter)
                except Country.DoesNotExist as ne:
                    raise Http404(
                        f"Country {result.description} not found.") from ne

                # Retrieve City
                try:
                    result = business_filter.get_query_filter("city")
                    city_obj = City.objects.get(**result.query_filter)
                except City.DoesNotExist as ne:
                    raise Http404(
                        f"City {result.description} not found.") from ne

                # Save Address
                address_obj = Address.objects.create(
                    street=street,
                    cp=postal_code,
                    city=city_obj,
                    latitude=latitude,
                    longitude=longitude,
                )

                # Get address details using laititude and longitude of the current address
                task = geo_service.get_location_data(latitude, longitude)
                address_data = asyncio.run(task)

                print("address_data processing")

                print(city_obj.id)

                # Update Region, County, Continent value to the City
                city_sync = DataMigration(
                    address_info=address_data,
                    city=city_obj,
                    country=city_obj.country
                )
                city_sync.sync_region_and_county()
                city_sync.sync_neighbour()

                if hasattr(city_sync, "region_obj"):
                    city_obj.region = getattr(city_sync, "region_obj", None)

                if hasattr(city_sync, "county_obj"):
                    city_obj.county = getattr(city_sync, "county_obj", None)

                if city_obj.country.continent:
                    city_obj.continent = city_obj.country.continent

                city_obj.save()

                # Update Neighbourhood value to the Address
                if hasattr(city_sync, "neighbour_obj"):
                    neighbour = getattr(city_sync, "neighbour_obj", None)
                    address_obj.neighborhood = neighbour
                    address_obj.save()

                site_obj.address = address_obj
                site_obj.city = city_obj
                site_obj.save()

                # Save operation hours
                op_timing = OperationTimings()
                if operation_hours := business_data.get("operating_hours"):
                    for day in const.WEEK_DAY_ORDER:
                        if day in operation_hours and day in Day.values:
                            timings = operation_hours[day]

                            # If timing is closed for the day, then skip the insertion.
                            if timings.lower() in ["close", "closed", "none"]:
                                continue
                            else:
                                hour_objects.extend(
                                    op_timing.process_time_slots(
                                        day, timings, site_obj))
                    if hour_objects:
                        HourRange.objects.bulk_create(hour_objects)

                # Retrieve Level
                try:
                    result = business_filter.get_query_filter("level")
                    level_obj = Level.objects.get(**result.query_filter)
                except Level.DoesNotExist as ne:
                    raise Http404(
                        f"Level {result.description} not found.") from ne

                site_obj.levels.add(level_obj)

                # Retrieve Category
                try:
                    result = business_filter.get_query_filter("category")
                    category_obj = Category.objects.get(**result.query_filter)
                except Category.DoesNotExist as ne:
                    raise Http404(
                        f"Category {result.description} not found.") from ne

                site_obj.categories.add(category_obj)

                # Retrieve Sub Category
                result = business_filter.get_query_filter(
                    "subcategory", category_obj=category_obj)
                if result.query_filter:
                    try:
                        sub_category_obj = SubCategory.objects.get(
                            **result.query_filter)
                    except SubCategory.DoesNotExist as ne:
                        raise Http404(
                            f"SubCategory {result.description} not found.") from ne

                    site_obj.subcategories.add(sub_category_obj)

                # Retrieve or create a User object based on the username.
                # If a matching object does not exist, create a new user object
                # with the provided user details.
                user_obj, _ = CustomUser.objects.get_or_create(
                    username=user_data.get("username"),
                    defaults={
                        field: business_data.get("user")[field]
                        for field in user_fields
                    }
                )
                site_obj.created_by = user_obj

                # Transform and store tags in the Tags table and their corresponding translations
                # in the Translated Tags table.
                if tag_objs := TagLink(business_data).assign():
                    for tag in tag_objs:
                        site_obj.tags.add(tag)

                site_obj.save()

                # Retrieve or create TranslatedSite
                for lang_code, (title_key, des_key, lang_name) in language_mapping.items():
                    if title_value := business_data.get(title_key):
                        description_value = business_data.get(des_key)

                        # Retrieve/Save language
                        language_obj, _ = Language.objects.get_or_create(
                            code=lang_code,
                            defaults={"name": lang_name}
                        )
                        TranslatedSite.objects.get_or_create(
                            title__iexact=title_value,
                            site=site_obj.id,
                            language=language_obj.id,
                            defaults={
                                "site": site_obj,
                                "language": language_obj,
                                "title": title_value,
                                "description": description_value,
                            }
                        )

                # Save images
                if images_urls := business_data.get("images_urls"):
                    site_image = UploadSiteImages(images_urls, buisness_id)
                    site_image.download_image()
                    site_image.upload_to_bucket(site_obj)

            return Response(
                {
                    "hasError": False,
                    "errorCode": -1,
                    "message": "success",
                }
            )
        except Exception as e:
            log_exception(e)  # write to log file
            return Response(
                {
                    "hasError": True,
                    "errorCode": 1002,
                    "message": str(e),
                }
            )


class LoadCountries(APIView):
    """
    This class is responsible for loading countries.
    """
    serializer_class = CountrySerializer

    @swagger_auto_schema(
        operation_id="List Countries",
        operation_description="Retrieves a list of countries, translating country names based on the provided language parameter.",
        manual_parameters=manual_params(["language"]),
        request=LanguageSerializer,
        responses={200: openapi.Response(
            description="List of countries",
            schema=serializer_class(many=True))})
    @validate_signature()
    @language_required
    def get(self, request):
        """
        Handles GET requests to load countries.
        This method executes a database query to retrieve the required countries. 
        """
        try:
            language = request.GET.get('language')
            if language == 'es':
                countries = Country.objects.all().values(
                    'id', 'name', 'code', 'phone_code').order_by("name")
            else:
                countries = (
                    TranslatedCountry.objects.filter(language__code=language).
                    select_related('country', 'language').
                    values('id', 'name', 'country__id').
                    annotate(code=F('country__code'), phone_code=F('country__phone_code')).
                    order_by("name")
                )
            return Response(self.serializer_class(countries, many=True).data)

        except Exception as e:
            log_exception(e)  # write to log file
            return Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class LoadCities(APIView):
    """
    This class is responsible for loading cities based on country id.
    """
    serializer_class = CitySerializer

    @swagger_auto_schema(
        operation_id="List Cities by Country",
        operation_description="Retrieves a list of cities based on country, translating city names based on the provided language parameter.",
        manual_parameters=manual_params(["language", "country"]),
        request=LanguageSerializer,
        responses={200: openapi.Response(
            description="List of Cities",
            schema=serializer_class(many=True))})
    @validate_signature()
    @language_required
    def get(self, request):
        """
        Handles GET requests to load cities.
        This method executes a database query to retrieve the required cities based on country. 
        """
        try:
            language = request.GET.get('language')
            country_id = request.GET.get('country_id')

            if language == 'es':
                cities = (
                    City.objects.filter(country=country_id).
                    select_related('country').values(
                        'id', 'name', 'cp', 'province', 'description', 'link',
                        'latitude', 'longitude', 'slogan').order_by("name"))
            else:
                cities = (
                    TranslatedCity.objects.
                    filter(
                        city__country=country_id,
                        language__code=language).
                    select_related('city', 'language').
                    values(
                        'id', 'name', 'province', 'description',
                              'slogan', 'city__id').annotate(
                        link=F('city__link'),
                        latitude=F('city__latitude'),
                        longitude=F('city__longitude')).order_by("name"))

            return Response(self.serializer_class(cities, many=True).data)

        except Exception as e:
            log_exception(e)  # write to log file
            return Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class LoadLevels(APIView):
    """
    This class is responsible for loading levels.
    """
    serializer_class = LevelSerializer

    @swagger_auto_schema(
        operation_id="List Levels",
        operation_description="Retrieves a list of Levels, translating level names based on the provided language parameter.",
        manual_parameters=manual_params(["language"]),
        request=LanguageSerializer,
        responses={200: openapi.Response(
            description="List of Levels",
            schema=serializer_class(many=True))})
    @validate_signature()
    @language_required
    def get(self, request):
        """
        Handles GET requests to load levels.
        This method executes a database query to retrieve the required levels. 
        """
        try:
            language = request.GET.get('language')
            if language == 'es':
                levels = Level.objects.filter(
                    type="place").all().values(
                    'id', 'title').order_by("title")
            else:
                levels = (
                    TranslatedLevel.objects.filter(
                        language__code=language, level__type="place").
                    select_related('level', 'language').
                    values('id', 'title', 'level__id').order_by("title")
                )

            return Response(self.serializer_class(levels, many=True).data)

        except Exception as e:
            log_exception(e)  # write to log file
            return Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class LoadCategories(APIView):
    """
    This class is responsible for loading categories based on the level they are assosiated with.
    """
    serializer_class = CategorySerializer

    @swagger_auto_schema(
        operation_id="List Categories by Level",
        operation_description="Retrieves a list of categories by level, translating category names based on the provided language parameter.",
        manual_parameters=manual_params(["language", "level"]),
        request=LanguageSerializer,
        responses={200: openapi.Response(
            description="List of Categories",
            schema=serializer_class(many=True))})
    @validate_signature()
    @language_required
    def get(self, request):
        """
        Handles GET requests to load categories based on the level id.
        This method executes a database query to retrieve the required categories. 
        """
        try:
            level_id = request.GET.get('level_id')
            language = request.GET.get('language')

            if language == "es":
                category = (
                    Category.objects.filter(level=level_id).all().
                    order_by("title").
                    values("id", "title")
                )
            else:
                category = (
                    TranslatedCategory.objects.filter(
                        language__code=language, category__level__id=level_id
                    ).select_related('category', 'language').
                    all().
                    order_by("title").
                    values("id", "title", "category__id")
                )

            return Response(self.serializer_class(category, many=True).data)

        except Exception as e:
            log_exception(e)  # write to log file
            return Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class LoadSubCategories(APIView):
    """
    This class is responsible for loading subcategories based on the category they are 
    assosiated with.
    """
    serializer_class = SubCategorySerializer

    @swagger_auto_schema(
        operation_id="List Sub categories by Category",
        operation_description="Retrieves a list of sub categories by category, translating sub category names based on the provided language parameter.",
        manual_parameters=manual_params(["language", "category"]),
        request=LanguageSerializer,
        responses={200: openapi.Response(
            description="List of Sub Categories",
            schema=serializer_class(many=True))})
    @validate_signature()
    @language_required
    def get(self, request):
        """
        Handles GET requests to load subcategories based on the category id.
        This method executes a database query to retrieve the required sub categories. 
        """
        try:
            category_id = request.GET.get('category_id')
            language = request.GET.get('language')

            if language == "es":
                sub_category = (
                    SubCategory.objects.filter(category=category_id).all().
                    order_by("title").
                    values("id", "title")
                )
            else:
                sub_category = (
                    TranslatedSubCategory.objects.
                    filter(
                        language__code=language,
                        subcategory__category__id=category_id).
                    select_related('subcategory', 'language').
                    all().order_by("title").values(
                        "id", "title", "subcategory__id"))

            return Response(self.serializer_class(
                sub_category, many=True).data)

        except Exception as e:
            log_exception(e)  # write to log file
            return Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)
