# local_secrets/sites/cron.py

from django.core.exceptions import ObjectDoesNotExist
from django.utils import timezone
from django.db import transaction
from django.conf import settings
import logging
from datetime import timedelta

from local_secrets.sites.models import Site, SiteVisibilityChoices
from local_secrets.sites.services.site_visibility import VisibilityService
from local_secrets.users.models import VendorSubscription 

logger = logging.getLogger(__name__)

class VisibilityCronJob:
    def __init__(self):
        self.visibility_service = VisibilityService()

    def process_pending_visibility_updates(self):
        """
        Process pending visibility updates
        Runs every 5 minutes via cron
        """
        cutoff_time = timezone.now() - timedelta(minutes=settings.VISIBILITY_RETRY_INTERVAL)
        
        # Get sites that need visibility update
        pending_sites = Site.objects.filter(
            visibility_update_pending=True,
            last_update_attempt__lt=cutoff_time,
            update_attempts__lt=settings.VISIBILITY_MAX_RETRIES
        )

        for site in pending_sites:
            try:
                with transaction.atomic():
                    subscription = site.vendor.active_subscription
                    target_visibility = self._determine_target_visibility(subscription)
                    
                    self.visibility_service.update_site_visibility(
                        site=site,
                        new_visibility=target_visibility,
                        reason="Cron retry attempt",
                        metadata={"attempt": site.update_attempts + 1}
                    )

            except Exception as e:
                logger.error(f"Failed to process visibility update for site {site.id}: {str(e)}")

    def _determine_target_visibility(self, subscription: VendorSubscription) -> str:
        """
        Determine target visibility based on subscription status
        """
        if not subscription:
            return SiteVisibilityChoices.SUBSCRIPTION_EXPIRED
            
        if subscription.status == 'active' and subscription.end_date > timezone.now():
            return SiteVisibilityChoices.VISIBLE
        elif subscription.payment_status == 'pending':
            return SiteVisibilityChoices.PAYMENT_PENDING
        else:
            return SiteVisibilityChoices.SUBSCRIPTION_EXPIRED
