# sites/notifications.py
from django.core.mail import send_mail
from django.conf import settings
from django.utils.translation import gettext_lazy as _
from django.utils.html import strip_tags
from local_secrets.sites.choices import SiteStatusChoices
from local_secrets.users.models import Notification
from django.template.loader import render_to_string
from django.contrib.auth.models import User

class SiteNotificationManager:
    def __init__(self):
        self.email_templates = {
            'pending_review': 'sites/emails/pending_review.html',
            'approved': 'sites/emails/site_approved.html',
            'rejected': 'sites/emails/site_rejected.html',
            'published': 'sites/emails/site_published.html',
            'subscription_required': 'sites/emails/subscription_required.html'
        }

    def create_status_notification(self, site, status):
        """Create and send notification based on site status"""
        if status == SiteStatusChoices.PENDING_REVIEW:
            self._notify_admin_pending_review(site)
            self._notify_vendor_submitted(site)
        elif status == SiteStatusChoices.APPROVED:
            self._notify_vendor_approved(site)
        elif status == SiteStatusChoices.REJECTED:
            self._notify_vendor_rejected(site)
        elif status == SiteStatusChoices.PUBLISHED:
            self._notify_vendor_published(site)

    def _notify_admin_pending_review(self, site):
        """Notify admins about new pending review"""
        context = {
            'site': site,
            'vendor': site.vendor,
            'review_url': self._get_admin_review_url(site)
        }
        
        html_content = render_to_string(
            self.email_templates['pending_review'],
            context
        )
        
        send_mail(
            subject=_('New Site Review Request: {}').format(site.title),
            message=strip_tags(html_content),
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=self._get_admin_emails(),
            html_message=html_content
        )

    def _notify_vendor_approved(self, site):
        """Notify vendor about site approval"""
        context = {
            'site': site,
            'vendor': site.vendor,
            'dashboard_url': self._get_vendor_dashboard_url(site)
        }
        
        html_content = render_to_string(
            self.email_templates['approved'],
            context
        )
        
        send_mail(
            subject=_('Your Site Has Been Approved: {}').format(site.title),
            message=strip_tags(html_content),
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=[site.vendor.user.email],
            html_message=html_content
        )

    def _get_admin_review_url(self, site):
        """Get admin review URL for site"""
        return f"{settings.SITE_URL}/admin/sites/review/{site.id}/"

    def _get_vendor_dashboard_url(self, site):
        """Get vendor dashboard URL for site"""
        return f"{settings.SITE_URL}/vendor/sites/{site.id}/"

    def _get_admin_emails(self):
        """Get list of admin email addresses"""
        return User.objects.filter(is_staff=True).values_list('email', flat=True)    
  