Hi, Saurav,

When running the makemigrations you might have issues with the migrate due to the constraints of the new field.

You will need to:

1- Execute from using python manage.py shell the script below or use a direct SQL fix.
2- Run a fake translation: 
python manage.py migrate sites 0012_site_slug --fake
 


# This script should be executed in your Django shell (python manage.py shell)
# It will populate all empty slugs with unique values and then you can run the migration with 

# python manage.py migrate sites 0012_site_slug --fake

from django.db import connection
from django.utils.text import slugify
import random

def check_if_column_exists():
    """Check if the slug column already exists in the sites_site table"""
    with connection.cursor() as cursor:
        cursor.execute("""
            SELECT column_name 
            FROM information_schema.columns 
            WHERE table_name='sites_site' AND column_name='slug';
        """)
        return cursor.fetchone() is not None

def add_slug_column_if_not_exists():
    """Add the slug column if it doesn't exist yet"""
    if not check_if_column_exists():
        with connection.cursor() as cursor:
            cursor.execute("""
                ALTER TABLE sites_site
                ADD COLUMN slug VARCHAR(500);
            """)
            print("Added slug column to sites_site table")

def generate_unique_slug(site_id, title):
    """Generate a unique slug based on title"""
    base_slug = slugify(title or f"site-{site_id}")
    if not base_slug:
        base_slug = f"site-{site_id}"
    
    # Add a random suffix to ensure uniqueness
    return f"{base_slug}-{random.randint(1000, 9999)}"

def get_sites_without_slug():
    """Get all sites that have NULL or empty slugs"""
    with connection.cursor() as cursor:
        cursor.execute("""
            SELECT id, title FROM sites_site
            WHERE slug IS NULL OR slug = '';
        """)
        return cursor.fetchall()

def update_site_slug(site_id, new_slug):
    """Update the slug for a specific site"""
    with connection.cursor() as cursor:
        cursor.execute("""
            UPDATE sites_site
            SET slug = %s
            WHERE id = %s;
        """, [new_slug, site_id])

def check_for_duplicate_slugs():
    """Check for any duplicate slugs in the database"""
    with connection.cursor() as cursor:
        cursor.execute("""
            SELECT slug, COUNT(*) 
            FROM sites_site 
            GROUP BY slug 
            HAVING COUNT(*) > 1;
        """)
        return cursor.fetchall()

def fix_duplicate_slugs():
    """Fix any duplicate slugs found in the database"""
    duplicates = check_for_duplicate_slugs()
    if not duplicates:
        print("No duplicate slugs found")
        return
    
    print(f"Found {len(duplicates)} duplicate slug values")
    for slug, count in duplicates:
        with connection.cursor() as cursor:
            # Get all sites with this slug except the first one
            cursor.execute("""
                SELECT id, title FROM sites_site
                WHERE slug = %s
                ORDER BY id
                LIMIT %s OFFSET 1;
            """, [slug, count - 1])
            sites_to_update = cursor.fetchall()
            
            for site_id, title in sites_to_update:
                new_slug = generate_unique_slug(site_id, title)
                update_site_slug(site_id, new_slug)
                print(f"Updated site {site_id} slug to {new_slug}")

def main():
    """Main function to fix all slug issues"""
    print("Starting slug repair process...")
    
    # Add slug column if it doesn't exist
    add_slug_column_if_not_exists()
    
    # Get sites without slugs
    sites_without_slug = get_sites_without_slug()
    print(f"Found {len(sites_without_slug)} sites without slugs")
    
    # Update each site with a unique slug
    for site_id, title in sites_without_slug:
        new_slug = generate_unique_slug(site_id, title)
        update_site_slug(site_id, new_slug)
        print(f"Updated site {site_id} slug to {new_slug}")
    
    # Fix any duplicate slugs
    fix_duplicate_slugs()
    
    print("Slug repair process completed successfully")

if __name__ == "__main__":
    main()

