from datetime import datetime, timedelta, timezone
from functools import cache

from django.db.models import Avg
from django.utils.timezone import now, localtime
from django.utils.translation import gettext_lazy as _
from rest_framework import serializers
from rest_framework.exceptions import ValidationError
from rest_framework.response import Response
from local_secrets.cities.models import City, Address 
from local_secrets.cities.serializers import AddressCreateSerializer, AddressRetrieveSerializer, AddressSerializer, BaseCitySerializer, CitySerializer, AddressPointSerializer
from local_secrets.core.events import SystemEvent
from local_secrets.core.serializers import ThumbnailJSONSerializer
from local_secrets.events.models import Event
from local_secrets.sites.choices import Day, FrequencyChoices, SiteStatusChoices, SiteSubscriptionStatus, SiteType, SiteVisibilityChoices
from local_secrets.sites.models import (
    Category,
    Comment,
    DefaultImage,
    Level, 
    Day,
    Site, 
    Schedule,
    HourRange,
    SpecialSchedule,
    SubCategory,
    
)
from rest_framework.decorators import action
from rest_framework.permissions import IsAuthenticated
from local_secrets.sites.services.review_workflow import ReviewWorkflowManager
from local_secrets.sites.state_machine import StateTransitionError
from local_secrets.users.models import LocationPreference, Tag
from local_secrets.users.serializers import TagOutputWithoutSelectionSerializer, UserCommentOutputSerializer
from .models import SiteReview, ReviewChecklistItem, ReviewChecklistResponse, SpecialHourRange
from django.contrib.gis.geos import Point  
from .models import Site, SiteReview, Promotion
from .validators import SitePublicationValidator
from decimal import Decimal 
from django.db import transaction
from rest_framework import serializers
from local_secrets.sites.models import Site, SiteVisibilityChoices
from local_secrets.core import constants as const

class SubCategorySerializer(serializers.ModelSerializer):
    def to_representation(self, instance):
        representation = super(SubCategorySerializer, self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
        else:
            return representation

        for field in self.fields:
            if field in ['title', 'description', 'type']:
                representation[field] = instance.display_text(field, language)
        return representation

    class Meta:
        model = SubCategory
        exclude = ('category',)
 
class CategorySerializer(serializers.ModelSerializer):
    subcategories = serializers.SerializerMethodField()

    def get_subcategories(self, obj):
        return SubCategorySerializer(
            obj.subcategories.all(),
            many=True,
            context=self.context
        ).data

    def to_representation(self, instance):
        representation = super(CategorySerializer, self).to_representation(instance)
        request = self.context.get('request')
        if request and request.headers.get('language'):
            language = request.headers.get('language')
        else:
            return representation

        for field in self.fields:
            if field in ['title', 'description', 'type']:
                representation[field] = instance.display_text(field, language)
        return representation

    class Meta:
        model = Category
        fields = ['id', 'title', 'subcategories']
 
class CategoryListSerializer(CategorySerializer):
    subcategories = None

    class Meta:
        model = Category
        fields = ['id', 'title',]
 
class LevelSerializer(serializers.ModelSerializer):
    categories = serializers.SerializerMethodField()

    def get_categories(self, obj):
        return CategorySerializer(
           obj.categories.all(),
            many=True,
            context=self.context
        ).data

    def to_representation(self, instance):
        representation = super(LevelSerializer, self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
        else:
            return representation

        for field in self.fields:
            if field in [
                'title',
            ]:
                representation[field] = instance.display_text(field, language)
        return representation

    class Meta:
        model = Level
        fields = ['id', 'title', 'categories']
 
class LevelListSerializer(LevelSerializer):
    categories = None

    class Meta:
        model = Level
        fields = ['id', 'title',]
 
class HourRangeSerializer(serializers.Serializer):
    id = serializers.IntegerField(default=0)
    initial_hour = serializers.TimeField(format='%I:%M %p')
    end_hour = serializers.TimeField(format='%I:%M %p')

    class Meta:
        fields = ('id', 'initial_hour', 'end_hour')
 
class ScheduleSerializer(serializers.ModelSerializer):
    opening_hours = HourRangeSerializer(many=True, required=False)
    
    class Meta:
        model = Schedule
        fields = ['day', 'opening_hours']
        extra_kwargs = {
            'day': {'required': True},
            'opening_hours': {'required': False}
        }
 
class SpecialScheduleSerializer(serializers.ModelSerializer):
    # initial_hour = serializers.TimeField(format='%I:%M %p')
    # end_hour = serializers.TimeField(format='%I:%M %p')
    opening_hours = serializers.SerializerMethodField()
    #opening_hours = HourRangeSerializer(many=True)

    def get_opening_hours(self, obj):
        days = dict(zip(range(0, 6), Day.choices))
        try:
            if obj.opening_hours.exists():
                return HourRangeSerializer(obj.opening_hours.all(), many=True).data
            if obj.site.schedules.filter(day=days[obj.day.weekday()][0]).exists():
                return HourRangeSerializer(obj.site.schedules.filter(day=days[obj.day.weekday()][0]).first().opening_hours.all(), many=True).data
        except Exception as e:
            print(e)
            return []
        return []

    class Meta:
        model = SpecialSchedule
        exclude = ('site',)
 
class SiteGeolocationSerializer(serializers.ModelSerializer):
    address = AddressPointSerializer()

    class Meta:
        model = Site
        fields = ('id', 'address')

class SiteListSerializer(serializers.ModelSerializer):
    title = serializers.CharField()
    description = serializers.CharField()
    city = BaseCitySerializer()
    images = ThumbnailJSONSerializer(alias='', read_only=True, many=True)
    rating = serializers.SerializerMethodField()
    tags = TagOutputWithoutSelectionSerializer(many=True)
    is_fav = serializers.BooleanField(required=False, default=False)
    address = AddressSerializer()
    next_schedule = serializers.SerializerMethodField(required=False, allow_null=True)
    date_range = serializers.SerializerMethodField(required=False, allow_null=True)
    levels = LevelListSerializer(many=True)
    comments_count = serializers.SerializerMethodField(allow_null=True)

    def get_next_schedule(self, obj):
        next_schedule = obj.next_schedule()
        # Handle the schedule structure
        if obj.type == 'place':
            return ScheduleSerializer(next_schedule, required=False, allow_null=True).data
        if not next_schedule:
            return None
        if isinstance(next_schedule, Schedule):
            return ScheduleSerializer(next_schedule, required=False, allow_null=True).data
        elif isinstance(next_schedule, SpecialSchedule):
            return SpecialScheduleSerializer(next_schedule, required=False, allow_null=True).data
        return None

    def get_date_range(self, obj):
        if obj.type == 'event':
            today = localtime().date()
            dates = [
                obj.special_schedules.annotate_day_distance(today).order_by('day_distance').first(),
                obj.special_schedules.annotate_day_distance(today).order_by('day_distance').last()
            ]
            if len(dates) == 0 or not obj.special_schedules.exists():
                return []
            if obj.frequency == 'year' and dates[0].day < today:
                dates[0].day = dates[0].day.replace(year=today.year + 1)
            return SpecialScheduleSerializer(dates, many=True).data
        return []

    def get_rating(self, obj):
        average_rating = obj.comments.aggregate(average_rating=Avg('rating'))['average_rating']
        return round(average_rating, 1) if average_rating else 0

    def get_comments_count(self, obj):
        return obj.comments.count()

    def to_representation(self, instance):
        representation = super().to_representation(instance)
        request = self.context.get('request')
        language = request.headers.get('Language', 'es') if request else 'es'

        for field in ['title', 'description']:
            representation[field] = instance.display_text(field, language)
        return representation

    class Meta:
        model = Site
        exclude = ('users', 'categories', 'subcategories')

class SiteVisibilitySerializer(serializers.ModelSerializer):
    class Meta:
        model = Site
        fields = '__all__'
 
class VisibilityUpdateSerializer(serializers.Serializer):
    visibility = serializers.ChoiceField(
    choices=SiteVisibilityChoices.choices
    )
    reason = serializers.CharField(required=False)
    metadata = serializers.JSONField(required=False)
 
class VisibilityEventSerializer(serializers.ModelSerializer):
    class Meta:
        model = SystemEvent
        fields = ['created_at', 'event_type', 'payload']
 
class VisibilityStatsSerializer(serializers.Serializer):
    total_sites = serializers.IntegerField()
    visibility_counts = serializers.DictField()
    pending_updates = serializers.IntegerField()
    failed_attempts = serializers.IntegerField()
 
class SiteRandomListSerializer(serializers.ModelSerializer):
    images = serializers.SerializerMethodField()
    city = serializers.SerializerMethodField()
    rating = serializers.SerializerMethodField()
    comments_count = serializers.SerializerMethodField()
    is_fav = serializers.BooleanField(required=False, default=False)
    
    class Meta:
        model = Site
        fields = ('id', 'title', 'images', 'city', 'rating', 'comments_count', 'is_fav')

    def get_images(self, obj):
        if obj.images.exists():
            return ThumbnailJSONSerializer(
                instance=obj.images.first(),
                context=self.context,
                alias='',
                read_only=True
            ).data
        return None

    def get_city(self, obj):
        if obj.city:
            return {
                'id': obj.city.id,
                'name': obj.city.name
            }
        return None
    
    def get_rating(self, obj):
        average_rating = obj.comments.aggregate(average_rating=Avg('rating'))['average_rating']
        return round(average_rating, 1) if average_rating else 0

    def get_comments_count(self, obj):
        return obj.comments.count()

    def to_representation(self, instance):
        representation = super(SiteRandomListSerializer, self).to_representation(instance)
        request = self.context.get('request')
        
        if request and request.headers.get('language'):
            language = request.headers.get('language')
            representation['title'] = instance.display_text('title', language)
            if instance.city:
                representation['city']['name'] = instance.city.display_text('name', language)
        
        return representation
 
class SiteLiteListSerializer(SiteRandomListSerializer):
    city = None
    #rating = None
    tags = None
    address = None
    levels = LevelListSerializer(many=True)
    date_range = serializers.SerializerMethodField(required=False, allow_null=True)
    rating = serializers.SerializerMethodField(allow_null=True)
    comments_count = serializers.SerializerMethodField(allow_null=True)
    city = CitySerializer(required=False, read_only=True)

    class Meta:
        model = Site
        fields = ('id', 'title', 'images', 'is_fav', 'next_schedule', 'levels', 'date_range', 'rating', 'comments_count', 'city')

    def get_rating(self, obj):
        average_rating = obj.comments.aggregate(average_rating=Avg('rating'))['average_rating']
        return round(average_rating, 1) if average_rating else 0

    def get_comments_count(self, obj):
        return obj.comments.count()
    
class SiteBaseSerializer(serializers.ModelSerializer):
    """Base serializer with common fields and methods"""
    type = serializers.ChoiceField(choices=SiteType.choices)
    status = serializers.ChoiceField(choices=SiteStatusChoices.choices, read_only=True)
    subscription_status = serializers.ChoiceField(
        choices=SiteSubscriptionStatus.choices, 
        read_only=True
    )
    visibility = serializers.ChoiceField(
        choices=SiteVisibilityChoices.choices, 
        read_only=True
    )
    images = ThumbnailJSONSerializer(alias='', read_only=True, many=True)

    class Meta:
        model = Site
        abstract = True
 
class SiteDetailSerializer(SiteBaseSerializer):
    """Full serializer for detailed view"""
    title = serializers.CharField()
    description = serializers.CharField()
    city = CitySerializer()
    levels = LevelListSerializer(many=True)
    categories = CategoryListSerializer(many=True)
    subcategories = SubCategorySerializer(many=True)
    schedules = ScheduleSerializer(many=True)
    special_schedules = SpecialScheduleSerializer(many=True)
    tags = TagOutputWithoutSelectionSerializer(many=True)
    vendor_details = serializers.SerializerMethodField()
    analytics = serializers.SerializerMethodField()
    rating = serializers.SerializerMethodField(allow_null=True)
    comments_count = serializers.SerializerMethodField(allow_null=True)
    is_fav = serializers.BooleanField(required=False, default=False)
    address = AddressSerializer()
    
    class Meta:
        model = Site
        exclude = ('users',)
        read_only_fields = (
            'created_at', 'updated_at', 'published_at', 
            'unpublished_at', 'view_count', 'reviewed_by',
            'reviewed_at'
        )

    def get_vendor_details(self, obj):
        if obj.vendor:
            return {
                'id': obj.vendor.id,
                'name': obj.vendor.company_name,
                'contact_info': obj.vendor_contact_info,
                'is_featured': obj.is_vendor_featured
            }
        return None

    def get_analytics(self, obj):
        return obj.get_vendor_analytics()

    def get_rating(self, obj):
        average_rating = obj.comments.aggregate(average_rating=Avg('rating'))['average_rating']
        return round(average_rating, 1) if average_rating else 0

    def get_comments_count(self, obj):
        return obj.comments.count()
    
    def to_representation(self, instance):
        data = super().to_representation(instance)

        # Handle translation for title and description
        request = self.context.get('request')
        language = request.headers.get('language') if request else None
        if language:
            for field in ['title', 'description']:
                data[field] = instance.display_text(field, language)

        if "schedules" in data:
            data["schedules"] = sorted(
                data["schedules"], 
                key=lambda x: const.WEEK_DAY_ORDER.index(x["day"].lower())
            )
        return data

class SiteCreateUpdateSerializer(SiteBaseSerializer):
    """Serializer for create/update operations"""
    
    class Meta:
        model = Site
        exclude = (
            'users', 'created_at', 'updated_at', 'published_at',
            'unpublished_at', 'reviewed_by', 'reviewed_at'
        )

    def validate(self, data):
        """Validate the complete data set."""
        if 'type' in data:
            data['type'] = data['type'].lower()

        # Check for unique constraint
        title = data.get('title')
        site_type = data.get('type')
        
        if self.instance:  # Update case
            exists = Site.objects.exclude(id=self.instance.id).filter(
                title=title, 
                type=site_type
            ).exists()
        else:  # Create case
            exists = Site.objects.filter(
                title=title, 
                type=site_type
            ).exists()

        if exists:
            raise serializers.ValidationError(
                f"A {site_type} with this title already exists"
            )

        # Validate required fields for review submission
        if data.get('status') == SiteStatusChoices.PENDING_REVIEW:
            required_fields = ['title', 'description', 'type', 'address']
            missing_fields = [
                field for field in required_fields 
                if not data.get(field)
            ]
            if missing_fields:
                raise serializers.ValidationError(
                    f"Missing required fields for review: {', '.join(missing_fields)}"
                )

        return data

    @transaction.atomic
    def create(self, validated_data):
        schedules_data = validated_data.pop('schedules', [])
        special_schedules_data = validated_data.pop('special_schedules', [])
        address_data = validated_data.pop('address', None)
        tags_data = validated_data.pop('tags', [])
        
        site = Site.objects.create(**validated_data)
        
        if address_data:
            site.create_address(address_data)
        
        if schedules_data:
            site.add_schedules(schedules_data)
            
        if special_schedules_data:
            site.add_special_schedules(special_schedules_data)
            
        if tags_data:
            site.tags.set(tags_data)
            
        return site

    @transaction.atomic
    def update(self, instance, validated_data):
        # Similar to create but handles updates
        schedules_data = validated_data.pop('schedules', None)
        special_schedules_data = validated_data.pop('special_schedules', None)
        address_data = validated_data.pop('address', None)
        tags_data = validated_data.pop('tags', None)
        
        # Update main fields
        for attr, value in validated_data.items():
            setattr(instance, attr, value)
        
        if address_data:
            instance.create_address(address_data)
            
        if schedules_data is not None:
            instance.schedules.all().delete()
            instance.add_schedules(schedules_data)
            
        if special_schedules_data is not None:
            instance.special_schedules.all().delete()
            instance.add_special_schedules(special_schedules_data)
            
        if tags_data is not None:
            instance.tags.set(tags_data)
            
        instance.save()
        return instance

class PromotionSerializer(serializers.ModelSerializer):
    current_price = serializers.DecimalField(
        max_digits=10, 
        decimal_places=2, 
        read_only=True
    )
    discounted_price = serializers.DecimalField(
        max_digits=10, 
        decimal_places=2, 
        read_only=True
    )
    vendor_name = serializers.CharField(source='vendor.company_name', read_only=True)
    site_title = serializers.CharField(source='site.title', read_only=True)
    
    class Meta:
        model = Promotion
        fields = [
            'id',
            'name',
            'description',
            'discount_type',
            'discount_value',
            'target_type',
            'site',
            'site_title',
            'vendor',
            'vendor_name',
            'start_date',
            'end_date',
            'is_active',
            'current_price',
            'discounted_price',
            'created_at',
            'updated_at'
        ]
        read_only_fields = [
            'id', 
            'vendor', 
            'created_at', 
            'updated_at', 
            'current_price', 
            'discounted_price'
        ]
        
    def validate(self, data):
        # Validate dates
        start_date = data.get('start_date')
        end_date = data.get('end_date')
        
        if start_date and end_date and start_date >= end_date:
            raise serializers.ValidationError(
                _("End date must be after start date")
            )
            
        if start_date and start_date < timezone.now():
            raise serializers.ValidationError(
                _("Start date cannot be in the past")
            )
            
        # Validate discount value based on type
        discount_type = data.get('discount_type')
        discount_value = data.get('discount_value')
        
        if discount_type == 'percentage' and (discount_value <= 0 or discount_value > 100):
            raise serializers.ValidationError(
                _("Percentage discount must be between 0 and 100")
            )
            
        if discount_type == 'fixed' and discount_value <= 0:
            raise serializers.ValidationError(
                _("Fixed discount must be greater than 0")
            )
            
        return data
    
    def validate_site(self, site):
        request = self.context.get('request')
        if not site.vendor == request.user.vendor:
            raise serializers.ValidationError(
                _("You can only create promotions for your own sites")
            )
        
        if not site.can_have_promotion():
            raise serializers.ValidationError(
                _("This site cannot have promotions. Check subscription status and site visibility")
            )
        
        return site
        
    def to_representation(self, instance):
        data = super().to_representation(instance)
        if instance.site:
            data['current_price'] = instance.site.price
            data['discounted_price'] = instance.site.discounted_price
        return data
 
class SitePromotionListSerializer(serializers.ModelSerializer):
    """Serializer for listing promotions in site detail"""
    class Meta:
        model = Promotion
        fields = [
            'id',
            'name',
            'discount_type',
            'discount_value',
            'start_date',
            'end_date',
            'is_active'
        ]

class SiteSerializer(SiteListSerializer):
    type = serializers.ChoiceField(choices=SiteType.choices)
    city = CitySerializer(read_only=True)
    levels = LevelListSerializer(many=True)
    categories = CategoryListSerializer(many=True)
    subcategories = SubCategorySerializer(many=True)
    tags = TagOutputWithoutSelectionSerializer(many=True)
    address = AddressSerializer()
    schedules = ScheduleSerializer(many=True)
    special_schedules = SpecialScheduleSerializer(many=True)
    is_open = serializers.SerializerMethodField()
    schedule_summary = serializers.SerializerMethodField()
    current_status = serializers.SerializerMethodField()
    vendor_info = serializers.SerializerMethodField()
    analytics = serializers.SerializerMethodField()
    images = ThumbnailJSONSerializer(alias='', read_only=True, many=True)
    active_promotion = PromotionSerializer(read_only=True)
    current_price = serializers.DecimalField(max_digits=10, decimal_places=2, source='price', read_only=True)
    discounted_price = serializers.DecimalField(max_digits=10, decimal_places=2, read_only=True)
    average_rating = serializers.SerializerMethodField()
    review_count = serializers.SerializerMethodField()
    title = serializers.CharField(read_only=True)
    description = serializers.CharField(read_only=True)

    def get_is_open(self, obj):
        is_open = obj.is_open_by_schedule()
        return obj.check_frequency(is_open) if not is_open else is_open

    def get_schedule_summary(self, obj):
        if obj.always_open:
            return {"status": "always_open"}
        return {
            "is_open_now": self.get_is_open(obj),
            "next_schedule": str(obj.next_schedule())
        }

    def get_average_rating(self, obj):
        average_rating = obj.comments.aggregate(average_rating=Avg('rating'))['average_rating']
        return round(average_rating, 1) if average_rating else 0

    def get_review_count(self, obj):
        return obj.comments.count()

    def get_current_status(self, obj):
        return {
            "status": obj.status,
            "visibility": obj.visibility,
            "subscription_status": obj.subscription_status,
            "can_publish": obj.can_be_published(),
            "needs_review": obj.status == SiteStatusChoices.PENDING_REVIEW,
            "is_suggested": obj.is_suggested,
            "has_been_accepted": obj.has_been_accepted
        }

    def get_vendor_info(self, obj):
        if not obj.vendor:
            return None
        return {
            "id": obj.vendor.id,
            "name": obj.vendor.company_name,
            "contact_info": obj.vendor_contact_info,
            "is_featured": obj.is_vendor_featured,
            "notes": obj.vendor_notes
        }

    def get_analytics(self, obj):
        return {
            "view_count": obj.view_count,
            "favorite_count": obj.users.count(),
            "comments_count": getattr(obj, 'comments', []).count() if hasattr(obj, 'comments') else 0
        }

    def validate(self, data):
        """
        Enhanced validation including both business and data integrity rules
        """
        # Type validation
        if 'type' in data:
            data['type'] = data['type'].lower()
            if data['type'] not in dict(SiteType.choices):
                raise serializers.ValidationError("Invalid site type")

        # Unique constraint validation
        existing_query = Site.objects.filter(
            title=data.get('title'),
            type=data.get('type')
        )
        if self.instance:
            existing_query = existing_query.exclude(pk=self.instance.pk)
        
        if existing_query.exists():
            raise serializers.ValidationError(
                f"A {data.get('type')} with this title already exists"
            )

        # Status transition validation
        if 'status' in data and self.instance:
            try:
                self.instance.validate_transition(data['status'])
            except StateTransitionError as e:
                raise serializers.ValidationError(str(e))

        # Subscription validation for publishing
        if data.get('status') == SiteStatusChoices.PUBLISHED and self.instance:
            if not self.instance.has_valid_subscription():
                raise serializers.ValidationError(
                    "Cannot publish site without active subscription"
                )

        return data

    def create(self, validated_data):
        """
        Enhanced create method with proper transaction handling
        """
        schedules_data = validated_data.pop('schedules', [])
        special_schedules_data = validated_data.pop('special_schedules', [])
        address_data = validated_data.pop('address', None)
        tags_data = validated_data.pop('tags', [])
        
        with transaction.atomic():
            site = Site.objects.create(**validated_data)
            
            if address_data:
                Address.objects.create(site=site, **address_data)
            
            # Handle schedules
            for schedule_data in schedules_data:
                schedule = Schedule.objects.create(site=site, **schedule_data)
                
            # Handle special schedules
            for special_schedule_data in special_schedules_data:
                special_schedule = SpecialSchedule.objects.create(
                    site=site, 
                    **special_schedule_data
                )
            
            # Handle tags
            if tags_data:
                site.tags.set(tags_data)
                
        return site

    def to_representation(self, instance):
        representation = super().to_representation(instance)
        request = self.context.get('request')
        language = request.headers.get('Language', 'es') if request else 'es'

        for field in ['title', 'description']:
            representation[field] = instance.display_text(field, language)
        return representation

    class Meta:
        model = Site
        fields = (
            'id', 'title', 'description', 'type' ,
            'levels', 'categories', 'subcategories', 'tags', 'address', 'city', 'schedules',
            'special_schedules', 'is_open', 'schedule_summary', 'current_status', 'vendor_info',
            'analytics', 'is_suggested', 'has_been_accepted', 'created_by', 'always_open',
            'is_top_10', 'activated', 'view_count', 'price', 'current_price', 'discounted_price',
            'images', 'active_promotion', 'average_rating', 'review_count', 'created_at', 'updated_at'
        )
        read_only_fields = ('view_count', 'created_at', 'updated_at', 'is_top_10', 'activated')
        
class SiteWithoutCitySerializer(SiteSerializer):
    city = None
    levels = LevelListSerializer(many=True)
    categories = CategoryListSerializer(many=True)
    subcategories = SubCategorySerializer(many=True)

    class Meta:
        model = Site
        exclude = ('users', 'city')
 
class SiteForTravelSerializer(SiteWithoutCitySerializer):
    levels = None
    categories = None
    subcategories = None
    tags = None
    is_open = None
    special_schedules = None
    schedules = None
    next_schedule = None
    date_range = None
    city = CitySerializer()

    class Meta:
        model = Site
        exclude = (
            'users',
            'levels',
            'categories',
            'subcategories',
            'tags',
            'created_by',
            'has_been_accepted',
        )
 
class SiteCreationSerializer(serializers.ModelSerializer):
    schedules = serializers.ListField(child=serializers.JSONField(), required=False)
    special_schedules = serializers.ListField(child=serializers.JSONField(), required=False)
    levels = serializers.ListField(child=serializers.IntegerField(), required=True)
    categories = serializers.ListField(child=serializers.IntegerField(), required=True)
    subcategories = serializers.ListField(child=serializers.IntegerField(), required=True)
    location = serializers.JSONField()
    frequency = serializers.CharField(required=False)

    class Meta:
        model = Site
        fields = (
            'title',
            'type',
            'levels',
            'categories',
            'subcategories',
            'description',
            'schedules',
            'special_schedules',
            'location',
            'url',
            'phone',
            'frequency',
        )

    def is_valid(self, raise_exception=False):
        is_valid = super(SiteCreationSerializer, self).is_valid(raise_exception)

        levels = self.initial_data.get('levels')
        if Level.objects.filter(id__in=levels).count() != len(levels):
            if raise_exception:
                raise ValidationError(detail='Some of the selected levels does not exist', code=400)
            return False

        categories = self.initial_data.get('categories')
        if Category.objects.filter(id__in=categories).count() != len(categories):
            if raise_exception:
                raise ValidationError(detail='The selected category does not exist', code=400)
            return False

        subcategories = self.initial_data.get('subcategories')
        if SubCategory.objects.filter(id__in=subcategories).count() != len(subcategories):
            if raise_exception:
                raise ValidationError(detail='The selected subcategory does not exist', code=400)
            return False

        return is_valid
 
class FavoriteSitesFilterSerializer(serializers.Serializer):
    type = serializers.CharField(required=False)
    levels = serializers.ListField(required=False)
    categories = serializers.ListField(required=False)
    subcategories = serializers.ListField(required=False)
    keyword = serializers.CharField(required=False)
    city = serializers.CharField(required=False)
 
class CommentInputSerializer(serializers.Serializer):
    body = serializers.CharField(allow_blank=True)
    rating = serializers.DecimalField(max_digits=3, decimal_places=1)

    def is_valid(self, raise_exception=False):
        is_valid = super(CommentInputSerializer, self).is_valid(raise_exception)
        if is_valid and self.validated_data.get('rating') >= 1.0:
            return True
        if raise_exception:
            raise ValidationError(_('The review must have at least 1 star'), code=400)
        return False
 
class CommentSerializer(serializers.ModelSerializer):
    user = UserCommentOutputSerializer()
    body = serializers.CharField(allow_blank=True)

    class Meta:
        model = Comment
        exclude = [
            'site',
        ]
 
class DefaultImageSerializer(serializers.ModelSerializer):
    image = ThumbnailJSONSerializer(alias='', read_only=True)

    class Meta:
        model = DefaultImage
        fields = "__all__"
 
class SiteExistsSerializer(serializers.Serializer):
    id = serializers.CharField()
    order = serializers.IntegerField(required=False)

    def is_valid(self, raise_exception=False):
        is_valid = super().is_valid(raise_exception)
        try:
            Site.objects.get(id=self.validated_data.get('id'))
        except Site.DoesNotExist:
            if raise_exception:
                raise serializers.ValidationError(detail='The site does not exist', code=400)
            return False
        return is_valid
 
class CityExistsSerializer(serializers.Serializer):
    id = serializers.CharField()

    def is_valid(self, raise_exception=False):
        is_valid = super().is_valid(raise_exception)
        try:
            City.objects.get(id=self.validated_data.get('id'))
        except City.DoesNotExist:
            if raise_exception:
                raise serializers.ValidationError(detail='The city does not exist', code=400)
            return False
        return is_valid
 
class SitestoDiscoverSerializer(serializers.Serializer):
    id = serializers.IntegerField(read_only=True)
    display_text = serializers.SerializerMethodField(read_only=True)
    images = ThumbnailJSONSerializer(alias='', read_only=True, many=True)
    type = serializers.SerializerMethodField(read_only=True)
    slogan = serializers.CharField(required=False, read_only=True)
    num_of_routes = serializers.SerializerMethodField(read_only=True)
    city = CitySerializer(required=False, read_only=True)
    date_range = serializers.SerializerMethodField(required=False, read_only=True)
    address = AddressSerializer(required=False, read_only=True)
    categories = CategoryListSerializer(many=True, required=False, read_only=True)
    always_open = serializers.BooleanField(required=False, read_only=True)
    rating = serializers.SerializerMethodField(read_only=True)
    comments_count = serializers.SerializerMethodField(read_only=True)
    is_fav = serializers.BooleanField(required=False, default=False)
 
    def get_rating(self, obj):
        if type(obj) == Site:
            average_rating = obj.comments.aggregate(average_rating=Avg('rating'))['average_rating']
            return round(average_rating, 1) if average_rating else 0
        return 0

    def get_comments_count(self, obj):
        if type(obj) == Site:
            return obj.comments.count()
        else:
            return 0

    def get_type(self, obj):
        if type(obj) == City:
            return 'city'
        else:
            return obj.type

    def get_num_of_routes(self, obj):
        if type(obj) == City:
            return obj.routes.count()
        return 0

    def get_date_range(self, obj):
        if type(obj) == Site:
            if obj.type == 'event':
                dates = []
                dates.append(obj.special_schedules.first())
                dates.append(obj.special_schedules.last())
                return SpecialScheduleSerializer(dates, many=True).data
        return []

    def get_display_text(self, obj):
        try:
            getattr(obj, 'title')
            return obj.display_text('title', language=self.context.get('request').headers.get('Language'))
        except Exception:
            pass

        try:
            getattr(obj, 'name')
            return obj.display_text('name', language=self.context.get('request').headers.get('Language'))
        except Exception:
            pass
        return None
 
class ReviewChecklistItemSerializer(serializers.ModelSerializer):
    class Meta:
        model = ReviewChecklistItem
        fields = ['id', 'name', 'description', 'is_required', 'order']
 
class ReviewChecklistResponseSerializer(serializers.ModelSerializer):
    class Meta:
        model = ReviewChecklistResponse
        fields = ['id', 'checklist_item', 'is_satisfied', 'notes']
 
class SiteReviewSerializer(serializers.ModelSerializer):
    checklist_responses = ReviewChecklistResponseSerializer(many=True, read_only=True)
    
    class Meta:
        model = SiteReview
        fields = ['id', 'site', 'reviewer', 'status', 'feedback', 
                 'checklist_responses', 'created_at', 'updated_at']
        read_only_fields = ['created_at', 'updated_at', 'reviewer']

    def validate(self, data):
        if 'status' in data and data['status'] == SiteStatusChoices.APPROVED:
            # Check if all required checklist items are satisfied
            review = self.instance
            unsatisfied_required = review.checklist_responses.filter(
                checklist_item__is_required=True,
                is_satisfied=False
            ).exists()
            
            if unsatisfied_required:
                raise serializers.ValidationError(
                    _("Cannot approve site until all required checklist items are satisfied")
                )
        return data
 
class SitePublicationSerializer(serializers.ModelSerializer):
    subscription_status = serializers.SerializerMethodField()
    can_be_published = serializers.SerializerMethodField()
    publication_errors = serializers.SerializerMethodField()

    class Meta:
        model = Site
        fields = [
            'id', 'title', 'status', 'subscription_status',
            'can_be_published', 'publication_errors',
            'published_at', 'unpublished_at'
        ]
        read_only_fields = [
            'status', 'subscription_status', 'can_be_published',
            'publication_errors', 'published_at', 'unpublished_at'
        ]

    def get_subscription_status(self, obj):
        if not obj.active_subscription:
            return {
                'status': 'inactive',
                'message': _('No active subscription')
            }
        
        subscription = obj.active_subscription
        return {
            'status': 'active' if subscription.is_active() else 'expired',
            'plan': subscription.plan.name,
            'expires_at': subscription.end_date,
            'sites_remaining': subscription.remaining_site_quota
        }

    def get_can_be_published(self, obj):
        try:
            SitePublicationValidator.validate_publication_requirements(obj)
            return True
        except serializers.ValidationError:
            return False

    def get_publication_errors(self, obj):
        try:
            SitePublicationValidator.validate_publication_requirements(obj)
            return []
        except serializers.ValidationError as e:
            return e.detail

    def validate_publish(self, data):
        """
        Validate site can be published when publish action is called
        """
        try:
            SitePublicationValidator.validate_publication_requirements(self.instance)
        except serializers.ValidationError as e:
            raise serializers.ValidationError({
                'publish': e.detail
            })
        return data
 
class SiteReviewActionSerializer(serializers.ModelSerializer):
    action = serializers.ChoiceField(
        choices=['approve', 'reject'],
        write_only=True
    )
    feedback = serializers.CharField(
        required=False,
        allow_blank=True,
        style={'base_template': 'textarea.html'}
    )
    checklist_responses = serializers.ListField(
        child=serializers.DictField(),
        write_only=True,
        required=False
    )

    class Meta:
        model = SiteReview
        fields = [
            'id', 'site', 'status', 'feedback',
            'action', 'checklist_responses',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'site', 'status', 'created_at', 'updated_at']

    def validate(self, data):
        action = data.get('action')
        feedback = data.get('feedback')
        
        if action == 'reject' and not feedback:
            raise serializers.ValidationError({
                'feedback': _('Feedback is required when rejecting a site')
            })

        if action == 'approve':
            site = self.instance.site
            try:
                # Validate all checklist items are satisfied
                for response in self.instance.checklist_responses.all():
                    if response.checklist_item.is_required and not response.is_satisfied:
                        raise serializers.ValidationError(
                            _('All required checklist items must be satisfied')
                        )
            except Exception as e:
                raise serializers.ValidationError(str(e))

        return data

    def update(self, instance, validated_data):
        action = validated_data.pop('action')
        feedback = validated_data.get('feedback', '')
        checklist_responses = validated_data.pop('checklist_responses', [])

        # Update checklist responses if provided
        if checklist_responses:
            for response_data in checklist_responses:
                response = instance.checklist_responses.get(
                    id=response_data['id']
                )
                response.is_satisfied = response_data.get('is_satisfied', False)
                response.notes = response_data.get('notes', '')
                response.save()

        # Process the review action
        workflow = ReviewWorkflowManager(instance.site)
        if action == 'approve':
            workflow.approve_review(
                reviewer=self.context['request'].user,
                feedback=feedback
            )
        else:  # reject
            workflow.reject_review(
                reviewer=self.context['request'].user,
                feedback=feedback
            )

        return super().update(instance, validated_data)

class VendorSiteCreateSerializer(serializers.ModelSerializer):
    address = AddressCreateSerializer()
    schedules = ScheduleSerializer(many=True, required=False)
    type = serializers.ChoiceField(
        choices=['place', 'event'],
        required=True,
        error_messages={'invalid_choice': 'Type must be either "place" or "event".'}
    )
    frequency = serializers.ChoiceField(
        choices=['never', 'day', 'week', 'month', 'year', 'workday'],
        required=False,
        default='never'
    )

    class Meta:
        model = Site
        fields = [
            'title',
            'description',
            'type',
            'frequency',   
            'address',
            'schedules',
            'phone',
            'url',
            'categories',
            'subcategories',
            'tags',
            'levels',
            'phone',
            'url',
            'always_open',
            'vendor_notes',
            'vendor_contact_info'
        ]

    def to_internal_value(self, data):
        """
        Override to make schedules optional based on type
        """
        type_ = data.get('type')
        if type_ == 'event':
            self.fields['schedules'].required = False
            if 'special_schedules' in data:
                data['schedules'] = data.pop('special_schedules')
        return super().to_internal_value(data)

    def validate_categories(self, value):
        if not value:
            raise serializers.ValidationError("At least one category is required.")
        return value

    def validate_subcategories(self, value):
        if value:
            # Validate that subcategories belong to selected categories
            categories = self.initial_data.get('categories', [])
            invalid_subcategories = [
                sub.id for sub in value 
                if not sub.category_id in categories
            ]
            if invalid_subcategories:
                raise serializers.ValidationError(
                    "Subcategories must belong to selected categories."
                )
        return value
    
    def validate(self, data):
        data = super().validate(data)
        site_type = data.get('type')

        # Validate schedules based on type
        if site_type == 'event':
            if 'frequency' not in data:
                data['frequency'] = 'never'
            
            # For events, make schedules optional
            schedules = data.get('schedules', [])
            if schedules:
                for schedule in schedules:
                    try:
                        datetime.strptime(schedule['day'], '%Y-%m-%d')
                    except (ValueError, TypeError):
                        raise serializers.ValidationError({
                            'schedules': 'Invalid date format. Use YYYY-MM-DD for event schedules.'
                        })
        else:
            # For places, schedules might be required unless always_open is True
            if not data.get('always_open'):
                schedules = data.get('schedules', [])
                if not schedules:
                    raise serializers.ValidationError({
                        'schedules': 'Schedules are required for places unless always_open is True.'
                    })
                
                valid_days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday']
                for schedule in schedules:
                    if schedule['day'].lower() not in valid_days:
                        raise serializers.ValidationError({
                            'schedules': f'Invalid day format. Use one of: {", ".join(valid_days)}'
                        })

        return data

    @transaction.atomic
    def create(self, validated_data):
        address_data = validated_data.pop('address')
        schedules_data = validated_data.pop('schedules', [])
        categories = validated_data.pop('categories', [])
        subcategories = validated_data.pop('subcategories', [])
        tags = validated_data.pop('tags', [])
        levels = validated_data.pop('levels', [])

        # Create address
        address = Address.objects.create(
            street=address_data['street'],
            number=address_data['number'],
            city=address_data['city'],
            cp=address_data['cp'],
            latitude=address_data.get('latitude', Decimal('0.0')),
            longitude=address_data.get('longitude', Decimal('0.0')),
            point=Point(
                float(address_data.get('longitude', 0.0)),
                float(address_data.get('latitude', 0.0)),
                srid=4326
            )
        )

        # Create site/event based on type
        site_type = validated_data.get('type')
        if site_type == 'event':
            instance = Event.objects.create(
                **validated_data,
                address=address,
                city=address_data['city'],
                status=SiteStatusChoices.DRAFT
            )
            
            # Create special schedules for event if provided
            if schedules_data:
                for schedule_data in schedules_data:
                    special_schedule = SpecialSchedule.objects.create(
                        site=instance,
                        day=datetime.strptime(schedule_data['day'], '%Y-%m-%d').date()
                    )
                    
                    for hour_range in schedule_data.get('opening_hours', []):
                        SpecialHourRange.objects.create(
                            schedule=special_schedule,
                            initial_hour=hour_range['initial_hour'],
                            end_hour=hour_range['end_hour']
                        )
        else:
            instance = Site.objects.create(
                **validated_data,
                address=address,
                city=address_data['city'],
                status=SiteStatusChoices.DRAFT
            )
            
            # Create regular schedules for place if provided
            if schedules_data:
                for schedule_data in schedules_data:
                    schedule = Schedule.objects.create(
                        site=instance,
                        day=schedule_data['day'].lower()
                    )
                    
                    for hour_range in schedule_data.get('opening_hours', []):
                        HourRange.objects.create(
                            schedule=schedule,
                            initial_hour=hour_range['initial_hour'],
                            end_hour=hour_range['end_hour']
                        )

        # Set many-to-many relationships
        instance.categories.set(categories)
        instance.subcategories.set(subcategories)
        instance.tags.set(tags)
        instance.levels.set(levels)

        return instance

    def to_representation(self, instance):
        if instance.type == 'event':
            return VendorEventRetrieveSerializer(instance, context=self.context).data
        return VendorSiteRetrieveSerializer(instance, context=self.context).data
 
class VendorSiteRetrieveSerializer(serializers.ModelSerializer):
    address = AddressRetrieveSerializer()
    schedules = ScheduleSerializer(many=True, read_only=True)
    categories = CategoryListSerializer(many=True, read_only=True)
    subcategories = SubCategorySerializer(many=True, read_only=True)
    tags = TagOutputWithoutSelectionSerializer(many=True, read_only=True)
    levels = LevelListSerializer(many=True, read_only=True)
    
    class Meta:
        model = Site
        fields = [
            'id',
            'title',
            'description',
            'type',
            'address',
            'schedules',
            'phone',
            'url',
            'categories',
            'subcategories',
            'tags',
            'levels',
            'always_open',
            'vendor_notes',
            'vendor_contact_info',
            'created_at',
            'updated_at',
            'status'
        ]

class VendorEventRetrieveSerializer(serializers.ModelSerializer):
    address = AddressRetrieveSerializer()
    special_schedules = SpecialScheduleSerializer(many=True, read_only=True)
    schedules = ScheduleSerializer(many=True, read_only=True)
    categories = CategoryListSerializer(many=True, read_only=True)
    subcategories = SubCategorySerializer(many=True, read_only=True)
    tags = TagOutputWithoutSelectionSerializer(many=True, read_only=True)
    levels = LevelListSerializer(many=True, read_only=True)
    
    class Meta:
        model = Event
        fields = [
            'id',
            'title',
            'description',
            'type',
            'address',
            'special_schedules',
            'schedules',
            'phone',
            'url',
            'categories',
            'subcategories',
            'tags',
            'levels',
            'always_open',
            'vendor_notes',
            'vendor_contact_info',
            'created_at',
            'updated_at',
            'status',
            'frequency'  # Additional field specific to events
        ]

    def to_representation(self, instance):
        representation = super().to_representation(instance)
        
        # Add event-specific fields or modify existing ones
        if instance.frequency != 'never':
            next_schedule = instance.next_schedule()
            if next_schedule:
                representation['next_occurrence'] = next_schedule.day.isoformat() if hasattr(next_schedule, 'day') else None

        # Add opening days for events with schedules
        if instance.special_schedules.exists():
            first_day, last_day = instance.open_days()
            representation['first_day'] = first_day.day.isoformat() if first_day else None
            representation['last_day'] = last_day.day.isoformat() if last_day else None

        return representation
 
class LocationPreferenceSerializer(serializers.ModelSerializer):
    class Meta:
        model = LocationPreference
        fields = ['geolocation_enabled', 'default_radius', 'preferred_city']
    
class LocationInputSerializer(serializers.ModelSerializer):
    latitude = serializers.DecimalField(max_digits=9, decimal_places=6)
    longitude = serializers.DecimalField(max_digits=9, decimal_places=6)
    radius = serializers.IntegerField(default=5000, min_value=100, max_value=50000)   
    type = serializers.ChoiceField(choices=['all', 'place', 'event'], default='all')

    class Meta:
        model = LocationPreference
        fields = '__all__'