# sites/services/review_workflow.py

from django.db import transaction
from django.utils import timezone
from django.utils.translation import gettext as _

from local_secrets.sites.choices import SiteStatusChoices
from ..models import ReviewChecklistItem, Site, SiteReview, ReviewChecklistResponse
from django.core.exceptions import ValidationError
from ..notifications import SiteNotificationManager

class ReviewWorkflowManager:
    def __init__(self, site):
        self.site = site
        self.notification_manager = SiteNotificationManager()

    @transaction.atomic
    def submit_for_review(self):
        """Submit site for review"""
        if not self.site.can_submit_for_review():
            raise ValidationError(_("Site cannot be submitted for review"))
        
        self.site.status = SiteStatusChoices.PENDING_REVIEW
        self.site.submitted_for_review_at = timezone.now()
        self.site.save()

        # Create initial review
        review = SiteReview.objects.create(
            site=self.site,
            status=SiteStatusChoices.PENDING_REVIEW
        )

        # Create empty responses for all checklist items
        for item in ReviewChecklistItem.objects.all():
            ReviewChecklistResponse.objects.create(
                review=review,
                checklist_item=item
            )

        # Send notification
        self.notification_manager.create_status_notification(
            self.site,
            SiteStatusChoices.PENDING_REVIEW
        )

        return review

    @transaction.atomic
    def approve_review(self, reviewer, feedback=None):
        """Approve the site review"""
        if not self.site.can_be_approved():
            raise ValidationError(_("Site cannot be approved"))

        current_review = self.site.latest_review
        current_review.status = SiteStatusChoices.APPROVED
        current_review.reviewer = reviewer
        current_review.feedback = feedback
        current_review.save()

        self.site.status = SiteStatusChoices.APPROVED
        self.site.approved_at = timezone.now()
        self.site.save()

        # Send notification
        self.notification_manager.create_status_notification(
            self.site,
            SiteStatusChoices.APPROVED
        )

    @transaction.atomic
    def reject_review(self, reviewer, feedback):
        """Reject the site review"""
        if not feedback:
            raise ValidationError(_("Feedback is required for rejection"))

        current_review = self.site.latest_review
        current_review.status = SiteStatusChoices.REJECTED
        current_review.reviewer = reviewer
        current_review.feedback = feedback
        current_review.save()

        self.site.status = SiteStatusChoices.REJECTED
        self.site.save()

        # Send notification
        self.notification_manager.create_status_notification(
            self.site,
            SiteStatusChoices.REJECTED
        )
