# local_secrets/sites/services/site_visibility.py

from django.utils import timezone
from django.db import transaction
from typing import List, Optional, Dict, Any
import logging
from local_secrets.users.models import VendorSubscription
from local_secrets.sites.models import Site, SiteVisibilityChoices
from local_secrets.payment.models import VendorPayment
from django.core.exceptions import ValidationError
from django.conf import settings
 
from local_secrets.core.events import EventManager

logger = logging.getLogger(__name__)

class VisibilityUpdateEvent:
    VISIBILITY_CHANGED = 'site.visibility.changed'
    VISIBILITY_UPDATE_FAILED = 'site.visibility.update_failed'
    VISIBILITY_UPDATE_RETRIED = 'site.visibility.update_retried'
    VISIBILITY_UPDATE_SUCCEEDED = 'site.visibility.update_succeeded'

class VisibilityService:
    def __init__(self):
        self.event_manager = EventManager()

    @transaction.atomic
    def update_site_visibility(
        self,
        site: Site,
        new_visibility: str,
        reason: str = None,
        metadata: Dict[str, Any] = None
    ) -> bool:
        """
        Update site visibility with event tracking
        """
        if new_visibility not in SiteVisibilityChoices.values:
            raise ValidationError(f"Invalid visibility status: {new_visibility}")

        try:
            old_visibility = site.visibility
            site.visibility = new_visibility
            site.last_status_change = timezone.now()
            
            # Update retry tracking fields
            site.visibility_update_pending = False
            site.last_update_attempt = timezone.now()
            
            site.save()

            # Track visibility change event
            self.event_manager.emit(
                VisibilityUpdateEvent.VISIBILITY_CHANGED,
                {
                    'site_id': site.id,
                    'old_visibility': old_visibility,
                    'new_visibility': new_visibility,
                    'reason': reason,
                    'metadata': metadata or {}
                }
            )

            return True

        except Exception as e:
            logger.error(f"Failed to update site visibility: {str(e)}")
            self._mark_for_retry(site, new_visibility, reason, metadata)
            return False

    def _mark_for_retry(
        self,
        site: Site,
        target_visibility: str,
        reason: str = None,
        metadata: Dict[str, Any] = None
    ) -> None:
        """
        Mark site for retry in next cron run
        """
        try:
            site.visibility_update_pending = True
            site.update_attempts = (site.update_attempts or 0) + 1
            site.last_update_attempt = timezone.now()
            site.save()

            # Emit retry event
            self.event_manager.emit(
                VisibilityUpdateEvent.VISIBILITY_UPDATE_RETRIED,
                {
                    'site_id': site.id,
                    'target_visibility': target_visibility,
                    'retry_count': site.update_attempts,
                    'reason': reason,
                    'metadata': metadata or {}
                }
            )

            # If max retries reached, emit failure event
            if site.update_attempts >= settings.VISIBILITY_MAX_RETRIES:
                self.event_manager.emit(
                    VisibilityUpdateEvent.VISIBILITY_UPDATE_FAILED,
                    {
                        'site_id': site.id,
                        'target_visibility': target_visibility,
                        'final_retry_count': site.update_attempts,
                        'reason': reason,
                        'metadata': metadata or {}
                    }
                )

        except Exception as e:
            logger.error(f"Failed to mark site for retry: {str(e)}")

    def sync_visibility_with_payment(self, payment: VendorPayment) -> None:
        """
        Synchronize site visibility with payment status
        """
        vendor = payment.vendor
        sites = Site.objects.filter(vendor=vendor)

        visibility_mapping = {
            'processed': SiteVisibilityChoices.VISIBLE,
            'failed': SiteVisibilityChoices.PAYMENT_PENDING,
            'pending': SiteVisibilityChoices.PAYMENT_PENDING,
            'refunded': SiteVisibilityChoices.PAYMENT_PENDING
        }

        target_visibility = visibility_mapping.get(
            payment.status,
            SiteVisibilityChoices.PAYMENT_PENDING
        )

        metadata = {
            'payment_id': payment.id,
            'payment_status': payment.status,
            'payment_gateway': payment.payment_gateway
        }

        for site in sites:
            self.update_site_visibility(
                site=site,
                new_visibility=target_visibility,
                reason=f"Payment status: {payment.status}",
                metadata=metadata
            )


class SiteVisibilityManager:
    @staticmethod
    @transaction.atomic
    def update_site_visibility(subscription: VendorSubscription) -> None:
        """
        Update site visibility based on subscription status
        """
        sites = Site.objects.filter(vendor=subscription.vendor)
        current_time = timezone.now()

        if subscription.status == 'active' and subscription.end_date > current_time:
            sites.filter(
                visibility__in=[
                    SiteVisibilityChoices.SUBSCRIPTION_EXPIRED,
                    SiteVisibilityChoices.PAYMENT_PENDING
                ]
            ).update(
                visibility=SiteVisibilityChoices.VISIBLE,
                last_status_change=current_time,
                status_change_reason='subscription_activated',
                visibility_update_pending=False,
                update_attempts=0
            )
        elif subscription.status == 'expired' or subscription.end_date <= current_time:
            sites.filter(
                visibility=SiteVisibilityChoices.VISIBLE
            ).update(
                visibility=SiteVisibilityChoices.SUBSCRIPTION_EXPIRED,
                last_status_change=current_time,
                status_change_reason='subscription_expired',
                visibility_update_pending=False,
                update_attempts=0
            )
        elif subscription.payment_status == 'pending':
            sites.filter(
                visibility=SiteVisibilityChoices.VISIBLE
            ).update(
                visibility=SiteVisibilityChoices.PAYMENT_PENDING,
                last_status_change=current_time,
                status_change_reason='payment_pending',
                visibility_update_pending=False,
                update_attempts=0
            )

    @staticmethod
    @transaction.atomic
    def handle_payment_status_change(payment: VendorPayment) -> None:
        """
        Update site visibility based on payment status
        """
        try:
            subscription = VendorSubscription.objects.get(vendor=payment.vendor)
            
            if payment.status == 'processed':
                Site.objects.filter(
                    vendor=payment.vendor,
                    visibility=SiteVisibilityChoices.PAYMENT_PENDING
                ).update(
                    visibility=SiteVisibilityChoices.VISIBLE,
                    last_status_change=timezone.now(),
                    status_change_reason='payment_processed',
                    visibility_update_pending=False,
                    update_attempts=0
                )
            elif payment.status == 'failed':
                Site.objects.filter(
                    vendor=payment.vendor,
                    visibility=SiteVisibilityChoices.VISIBLE
                ).update(
                    visibility=SiteVisibilityChoices.PAYMENT_PENDING,
                    last_status_change=timezone.now(),
                    status_change_reason='payment_failed',
                    visibility_update_pending=False,
                    update_attempts=0
                )
                
        except Exception as e:
            logger.error(f"Error handling payment status change: {str(e)}")
            # Mark sites for retry via cron
            Site.objects.filter(vendor=payment.vendor).update(
                visibility_update_pending=True,
                last_update_attempt=timezone.now()
            )
