# sites/state_machine.py
from django.core.exceptions import ValidationError
from django.utils import timezone
from django.utils.translation import gettext_lazy as _

from local_secrets.sites.choices import SiteStatusChoices

class StateTransitionError(ValidationError):
    pass

class SiteStateMachine:
    """
    Handles site state transitions and validations
    """
    VALID_TRANSITIONS = {
        SiteStatusChoices.DRAFT: {
            'allowed_transitions': [SiteStatusChoices.PENDING_REVIEW],
            'required_fields': ['title', 'description', 'type'],
            'validation_message': _('Site must have title, description and type before submission')
        },
        SiteStatusChoices.PENDING_REVIEW: {
            'allowed_transitions': [SiteStatusChoices.APPROVED, SiteStatusChoices.REJECTED],
            'required_fields': ['reviewed_by'],
            'validation_message': _('Review must have an admin reviewer')
        },
        SiteStatusChoices.APPROVED: {
            'allowed_transitions': [SiteStatusChoices.PUBLISHED],
            'required_fields': ['active_subscription'],
            'validation_message': _('Site requires active subscription to be published')
        },
        SiteStatusChoices.PUBLISHED: {
            'allowed_transitions': [SiteStatusChoices.UNPUBLISHED],
            'required_fields': ['published_at'],
            'validation_message': _('Published site must have publication date')
        },
        SiteStatusChoices.REJECTED: {
            'allowed_transitions': [SiteStatusChoices.DRAFT],
            'required_fields': ['review_feedback'],
            'validation_message': _('Rejected sites must include feedback')
        },
        SiteStatusChoices.UNPUBLISHED: {
            'allowed_transitions': [SiteStatusChoices.PUBLISHED],
            'required_fields': ['active_subscription'],
            'validation_message': _('Site requires active subscription to be republished')
        }
    }

    @classmethod
    def validate_transition(cls, site, new_status):
        current_state = site.status
        state_config = cls.VALID_TRANSITIONS.get(current_state)
        
        if not state_config or new_status not in state_config['allowed_transitions']:
            raise StateTransitionError(
                _('Invalid status transition from %(current)s to %(new)s') % {
                    'current': current_state,
                    'new': new_status
                }
            )
 
        new_state_config = cls.VALID_TRANSITIONS.get(new_status)
        for field in new_state_config['required_fields']:
            if not getattr(site, field, None):
                raise StateTransitionError(new_state_config['validation_message'])

