# sites/validators.py
from django.core.exceptions import ValidationError
from local_secrets.sites.models import VideoSize
from django.utils import timezone
from django.utils.translation import gettext_lazy as _

class SubscriptionValidator:
    @staticmethod
    def validate_site_subscription(site):
        """Validate site subscription status"""
        if not site.active_subscription:
            raise ValidationError(_("No active subscription found"))

        subscription = site.active_subscription
        
        # Check subscription expiry
        if subscription.end_date and subscription.end_date < timezone.now().date():
            raise ValidationError(_("Subscription has expired"))
            
        # Check site limits
        active_sites = subscription.active_sites.filter(
            status=SiteStatusChoices.PUBLISHED
        ).count()
        
        if active_sites >= subscription.plan.max_sites:
            raise ValidationError(
                _("Maximum number of active sites ({}) reached for current subscription")
                .format(subscription.plan.max_sites)
            )

class SitePublicationValidator:
    @staticmethod
    def validate_publication_requirements(site):
        """Validate all requirements for site publication"""
        errors = []
        
        # Check review status
        if site.status != SiteStatusChoices.APPROVED:
            errors.append(_("Site must be approved before publication"))
            
        # Check subscription
        try:
            SubscriptionValidator.validate_site_subscription(site)
        except ValidationError as e:
            errors.extend(e.messages)
            
        # Check required content
        if not site.has_required_fields():
            errors.append(_("All required fields must be filled"))
            
        if errors:
            raise ValidationError(errors)
 
class SiteValidator:
    @staticmethod
    def validate_subscription(site):
        """Validate subscription status and limits"""
        if not site.active_subscription:
            return
        
        subscription = site.active_subscription
    
        if subscription.end_date and subscription.end_date < timezone.now().date():
            raise ValidationError(_('Subscription has expired'))
 
        active_sites_count = subscription.active_sites.count()
        if active_sites_count >= subscription.max_sites:
            raise ValidationError(_('Maximum number of sites reached for current subscription'))
    
    @staticmethod
    def validate_vendor_status(site):
        """Validate vendor status and permissions"""
        if not site.vendor:
            return
            
        if not site.vendor.is_active:
            raise ValidationError(_('Vendor account is not active'))
            
        if site.vendor.is_suspended:
            raise ValidationError(_('Vendor account is suspended'))
    
    @staticmethod
    def validate_content(site):
        """Validate site content requirements"""
        if site.status == SiteStatusChoices.PENDING_REVIEW:
            required_fields = ['title', 'description', 'type', 'address']
            missing_fields = [field for field in required_fields if not getattr(site, field, None)]
            
            if missing_fields:
                raise ValidationError(
                    _('Missing required fields: %(fields)s') % {
                        'fields': ', '.join(missing_fields)
                    }
                )
 
class SubscriptionValidator:
    @staticmethod
    def validate_site_subscription(site):
        """Validate site subscription status"""
        if not site.active_subscription:
            raise ValidationError(_("No active subscription found"))

        subscription = site.active_subscription
        
        # Check subscription expiry
        if subscription.end_date and subscription.end_date < timezone.now().date():
            raise ValidationError(_("Subscription has expired"))
            
        # Check site limits
        active_sites = subscription.active_sites.filter(
            status=SiteStatusChoices.PUBLISHED
        ).count()
        
        if active_sites >= subscription.plan.max_sites:
            raise ValidationError(
                _("Maximum number of active sites ({}) reached for current subscription")
                .format(subscription.plan.max_sites)
            )

class SitePublicationValidator:
    @staticmethod
    def validate_publication_requirements(site):
        """Validate all requirements for site publication"""
        errors = []
        
        # Check review status
        if site.status != SiteStatusChoices.APPROVED:
            errors.append(_("Site must be approved before publication"))
            
        # Check subscription
        try:
            SubscriptionValidator.validate_site_subscription(site)
        except ValidationError as e:
            errors.extend(e.messages)
            
        # Check required content
        if not site.has_required_fields():
            errors.append(_("All required fields must be filled"))
            
        if errors:
            raise ValidationError(errors)

def video_size(value):
    db_video_size = VideoSize.objects.first()
    print(value.size)
    size = value.size / 1024 / 1024
    if size > db_video_size.max_size:
        raise ValidationError(f'File too large. Size should not exceed {db_video_size.max_size} MB.')
    if size < db_video_size.min_size:
        raise ValidationError(f'File too small. Size should be greater than {db_video_size.min_size} MB.')
