# local_secrets/sites/views/dashboard.py

from django.conf import settings
from django.views.generic import TemplateView
from django.contrib.admin.views.decorators import staff_member_required
from django.utils.decorators import method_decorator
from django.db.models import Count, Q
from django.utils import timezone
from datetime import timedelta
from local_secrets.sites.models import Site, SiteVisibilityChoices
from local_secrets.core.events import SystemEvent


@method_decorator(staff_member_required, name='dispatch')
class VisibilityDashboardView(TemplateView):
    template_name = 'sites/dashboard/visibility_dashboard.html'
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        
        # Time ranges for analytics
        now = timezone.now()
        last_24h = now - timedelta(hours=24)
        last_week = now - timedelta(days=7)


        # Current status counts
        status_counts = Site.objects.values('visibility').annotate(
            count=Count('id')
        )


        # Pending updates
        pending_updates = Site.objects.filter(
            visibility_update_pending=True
        ).count()


        # Failed attempts
        failed_attempts = Site.objects.filter(
            update_attempts__gte=settings.VISIBILITY_MAX_RETRIES
        ).count()


        # Recent events
        recent_events = SystemEvent.objects.filter(
            event_type__startswith='site.visibility.'
        ).order_by('-created_at')[:50]


        # Status changes in last 24h
        recent_changes = SystemEvent.objects.filter(
            event_type='site.visibility.changed',
            created_at__gte=last_24h
        ).count()


        context.update({
            'status_counts': status_counts,
            'pending_updates': pending_updates,
            'failed_attempts': failed_attempts,
            'recent_events': recent_events,
            'recent_changes': recent_changes,
            'retry_threshold': settings.VISIBILITY_MAX_RETRIES,
        })
        
        return context


