import asyncio
from concurrent.futures import ThreadPoolExecutor
from datetime import datetime, timedelta, timezone
import json
from itertools import chain
from operator import attrgetter
import re
from typing import Any, Dict, List, Optional, Tuple
import uuid
from dal import autocomplete
from django.conf import settings
from django.contrib.gis.db.models.functions import Distance
from django.core.cache import cache
from django.core.files.images import get_image_dimensions
from django.db.models import Case, When, Max, Count, IntegerField, Prefetch, Q
from django.db.models.functions import Random
from django.shortcuts import redirect
import httpx
import openai
from openai import OpenAI
import requests
from requests.adapters import HTTPAdapter
from rest_framework import mixins, viewsets, status 
from rest_framework.decorators import action
from rest_framework.exceptions import PermissionDenied, ValidationError
from rest_framework.pagination import PageNumberPagination
from rest_framework.permissions import AllowAny, IsAuthenticated
from rest_framework.response import Response
from urllib3 import Retry
from local_secrets.cities.models import City
from local_secrets.core.permissions import IsAuthenticatedForPost
from local_secrets.core.utils.text import TextHelper as text
from local_secrets.sites.analytics import SiteAnalytics
from local_secrets.sites.models import DefaultImage, ImageSize, Level, Site, Category, SiteReview, SubCategory, VideoSize, SiteStatusChoices, ReviewChecklistResponse, ReviewChecklistItem
from local_secrets.sites.serializers import (
    CommentInputSerializer,
    CommentSerializer,
    DefaultImageSerializer,
    FavoriteSitesFilterSerializer,
    LevelSerializer,
    SiteCreationSerializer,
    SiteListSerializer,
    SiteSerializer,
    SitestoDiscoverSerializer, 
    SiteGeolocationSerializer, 
    SiteRandomListSerializer, 
    SiteLiteListSerializer,
    SiteReviewSerializer,
    ReviewChecklistItemSerializer,
    ReviewChecklistResponseSerializer,
    SitePublicationSerializer,
    VendorSiteCreateSerializer, 
    VendorSiteRetrieveSerializer, Promotion
)
from django.views.generic import ListView, DetailView, UpdateView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from django.shortcuts import get_object_or_404
from django.http import JsonResponse
from django.utils.translation import gettext_lazy as _
from rest_framework import viewsets, status, serializers
from rest_framework.views import APIView
from django.contrib.postgres.search import SearchVector, SearchRank, SearchQuery
from django.db.models import Q, F, Value, FloatField, CharField
from django.db.models.functions import Concat
from rest_framework.throttling import AnonRateThrottle, UserRateThrottle
from rest_framework.exceptions import Throttled
from local_secrets.sites.utils import AnonSearchRateThrottle, CacheManager, SearchRateThrottle
from local_secrets.users.models import LocationPreference
from local_secrets.users.permissions import HasActiveSubscription, IsVendorOwner
from .permissions import IsReviewerPermission, IsVendorOrAdmin
from .serializers import LocationInputSerializer, LocationPreferenceSerializer, SiteCreateUpdateSerializer, SiteDetailSerializer, SitePromotionListSerializer, SiteSerializer, PromotionSerializer 
from django.db import IntegrityError, transaction, models
from django.contrib.gis.geos import Point
from django.contrib.gis.measure import D
from django.views.decorators.csrf import csrf_exempt
from local_secrets.sites.models import Site, SiteVisibilityChoices
from local_secrets.core.events import SystemEvent
from .serializers import (
SiteVisibilitySerializer,
VisibilityUpdateSerializer,
VisibilityEventSerializer,
VisibilityStatsSerializer
)
from django.db.models.functions import Cast
from django.views.generic import TemplateView
from django.contrib.admin.views.decorators import staff_member_required
import logging
logger = logging.getLogger(__name__)

from django.utils.decorators import method_decorator

from httpx import RequestError as APIConnectionError

class CategoryViewSet(viewsets.GenericViewSet, mixins.ListModelMixin):
    def get_queryset(self):
        levels = Level.objects_for_api.all()
        level_type = self.request.query_params.get('type')
        if level_type:
            levels = levels.filter(type=level_type)
        city = self.request.query_params.get('city')
        if city:
            levels = levels.filter(sites__city__id=city)

            # Prefetch related subcategories that have sites in the specified city
            filtered_subcategories = SubCategory.objects_for_api.filter(
                sites__city=city,
                sites__type=level_type
            ).distinct().annotate(
                site_count=Count('sites', filter=Q(sites__city__id=city), output_field=IntegerField())
            ).filter(site_count__gt=0).order_by('order')

            # Prefetch related categories that have sites in the specified city
            filtered_categories = Category.objects_for_api.prefetch_related(
                Prefetch('subcategories', queryset=filtered_subcategories),
            ).filter(
                sites__city=city,
                sites__type=level_type
            ).distinct().annotate(
                site_count=Count('sites', filter=Q(sites__city__id=city), output_field=IntegerField())
            ).filter(site_count__gt=0).order_by('order', '-site_count')

            # Prefetch filtered categories to avoid fetching unfiltered categories
            levels = levels.prefetch_related(
                Prefetch('categories', queryset=filtered_categories),
            )

        levels = levels.annotate(site_count=Count('sites', filter=Q(sites__city__id=city), output_field=IntegerField())).filter(site_count__gt=0)
        return levels

    def get_serializer_class(self):
        return LevelSerializer

    def get_serializer_context(self):
        return {'request': self.request}

    @property
    def paginator(self):
        self._paginator = super(CategoryViewSet, self).paginator
        if not self.request.query_params.get('page'):
            self._paginator = None
        return self._paginator

# This view is used only on the admin page to filter the categories. It should not be used on the api
class CategoryAutocomplete(autocomplete.Select2QuerySetView):
    def get_queryset(self):
        name = self.request.GET.get('q')
        levels = self.forwarded.get('levels')
        if levels:
            if type(levels) == list:
                categories = Category.objects_for_admin.filter(level__in=levels)
            else:
                levels_as_list = levels.split(',')
                categories = Category.objects_for_admin.filter(level__in=levels_as_list)
        else:
            level = self.forwarded.get('level')
            if level:
                categories = Category.objects_for_admin.filter(level=level)
            else:
                categories = Category.objects_for_admin.all()
        if name:
            categories = categories.filter(title__icontains=name)
        return categories

class SubCategoryAutocomplete(autocomplete.Select2QuerySetView):
    def get_queryset(self):
        categories = self.forwarded.get('categories')
        name = self.request.GET.get('q')
        if categories:
            if type(categories) == list:
                subcategories = SubCategory.objects_for_admin.filter(category__in=categories)
            else:
                categories_as_list = categories.split(',')
                subcategories = SubCategory.objects_for_admin.filter(category__in=categories_as_list)
        else:
            subcategories = SubCategory.objects_for_admin.all()
        if name:
            subcategories = subcategories.filter(title__icontains=name)
        return subcategories

class LevelAutocomplete(autocomplete.Select2QuerySetView):
    def get_queryset(self):
        levels = Level.objects_for_admin.all()
        name = self.request.GET.get('q')
        if name:
            levels = levels.filter(title__icontains=name)
        return levels

class PlaceLevelAutocomplete(autocomplete.Select2QuerySetView):
    def get_queryset(self):
        levels = Level.objects_for_admin.filter(type='place')
        name = self.request.GET.get('q')
        if name:
            levels = levels.filter(title__icontains=name)
        return levels

class EventLevelAutocomplete(autocomplete.Select2QuerySetView):
    def get_queryset(self):
        levels = Level.objects_for_admin.filter(type='event')
        name = self.request.GET.get('q')
        if name:
            levels = levels.filter(title__icontains=name)
        return levels
 
class SiteViewSet(viewsets.GenericViewSet, mixins.ListModelMixin, mixins.RetrieveModelMixin, mixins.CreateModelMixin):    
    import logging
    logger = logging.getLogger(__name__)
    @property
    def paginator(self):
        self._paginator = super().paginator
        if not self.request.query_params.get('page'):
            self._paginator = None
        return self._paginator
 
    paginate_by = 2

    def get_permissions(self):
        if self.action == 'create':
            return [IsAuthenticated()]
        return [AllowAny()]

    def get_queryset(self): 
        queries = self.request.query_params
        language = self.request.headers.get('Language')

        queryset = Site.objects.all().annotate_num_of_favs()

        points = queries.getlist('points') or [
            queries.get('topLeft'),
            queries.get('topRight'),
            queries.get('bottomRight'),
            queries.get('bottomLeft')
        ]

        queryset = (
            queryset.filter_type(queries.get('type'))
            .filter_keyword(queries.get('keyword'), language)
            .filter_categories(queries.get('categories'))
            .filter_levels(queries.get('levels'))
            .filter_subcategories(queries.get('subcategories'))
            .filter_city(queries.get('city'), language)
            .filter_city_id(queries.get('city_id'))
            .filter_tag(queries.get('tag'))
            .filter_country(queries.get('country'))
            .filter_hours(queries.get('day'), queries.get('hour'))
            .filter_suggested(queries.get('is_suggested'))
            .filter_special_schedule(
                queries.get('first_date'), 
                queries.get('last_date')
            )
            .filter_polygon(points)
            .filter(has_been_accepted=True, city__activated=True)
            .annotate_is_fav(self.request.user)
            .distinct()
        )

        if queries.get('preview'):
            queryset = queryset.only('id', 'title')

        if queries.get('type') == 'event':
            return queryset
        return queryset.order_by('-is_top_10')

    def get_serializer_context(self):
        return {'request': self.request}

    @action(detail=True, methods=['POST'])
    def update_state(self, request, pk=None):
        site = self.get_object()
        new_state = request.data.get('state')
        
        try:
            result = site.change_state(new_state, request.user)
            return Response(result)
        except PermissionDenied as e:
            return Response(
                {'error': str(e)}, 
                status=status.HTTP_403_FORBIDDEN
            )
        except ValidationError as e:
            return Response(
                {'error': str(e)}, 
                status=status.HTTP_400_BAD_REQUEST
            )
        
    @transaction.atomic
    def create(self, request, *args, **kwargs):
        if not hasattr(request.user, 'vendor_profile'):
            return Response(
                {"detail": "You must be a vendor to create sites"}, 
                status=status.HTTP_403_FORBIDDEN
            )

        if request.user.vendor_profile.verification_status != 'verified':
            return Response(
                {"detail": "Your vendor account must be verified to create sites"}, 
                status=status.HTTP_403_FORBIDDEN
            )

        site_type = request.data.get('type', '').lower()
        if site_type not in ['place', 'event']:
            return Response(
                {"detail": "Invalid site type. Must be either 'place' or 'event'."}, 
                status=status.HTTP_400_BAD_REQUEST
            )

        try:
            serializer = self.get_serializer(data=request.data)
            serializer.is_valid(raise_exception=True)
            site = serializer.save(
                vendor=request.user.vendor_profile,
                created_by=request.user
            )
            return Response(
                self.get_serializer(site).data, 
                status=status.HTTP_201_CREATED
            )
        except IntegrityError:
            return Response(
                {"detail": "Site creation failed. Please try again."}, 
                status=status.HTTP_400_BAD_REQUEST
            ) 
 
    def list(self, request, *args, **kwargs):
        response = super().list(request, *args, **kwargs)
        if request.query_params.get('type') == 'event':
            new_results = sorted(
                response.data.get('results'), 
                key=lambda k: k['next_schedule']['day'] if k['next_schedule'] else 'z'
            )
            response.data['results'] = new_results
        return response

    def get_serializer_class(self):
        if self.action == 'list':
            return SiteListSerializer
        elif self.action in ['create', 'update', 'partial_update']:
            return SiteCreateUpdateSerializer
        return SiteDetailSerializer
    
    @action(detail=False, methods=['get'])
    def featured(self, request):
        """Get featured sites"""
        queryset = self.get_queryset().filter(is_vendor_featured=True)
        serializer = self.get_serializer(queryset, many=True)
        return Response(serializer.data)

    @action(detail=True, methods=['post'])
    def toggle_favorite(self, request, pk=None):
        """Toggle favorite status for a site"""
        site = self.get_object()
        created = site.mark_as_fav(request.user)
        return Response({'is_favorite': created})

    @action(detail=False, methods=['get'], permission_classes=(IsAuthenticated,))
    def favorites(self, request):
        paginator = PageNumberPagination()
        paginator.page_size = 25
        paginate = request.query_params.get('page')
        
        output_serializer = SiteListSerializer
        if self.request.query_params.get('preview'):
            output_serializer = SiteLiteListSerializer

        serializer = FavoriteSitesFilterSerializer(data=request.query_params)
        serializer.is_valid(raise_exception=True)

        language = request.headers.get('language')

        fav_sites = (
            request.user.fav_sites
            #.filter_type(serializer.validated_data.get('type'))
            .filter_keyword(serializer.validated_data.get('keyword'), language=language)
            .filter_levels(serializer.validated_data.get('levels'))
            .filter_categories(request.query_params.get('categories'))
            .filter_subcategories(request.query_params.get('subcategories'))
            .filter_city(serializer.validated_data.get('city'), language=language)
            .annotate_is_fav(self.request.user)
            .annotate_translated_fields(language)
            .order_by('-is_top_10')
            .distinct()
        )

        if self.request.query_params.get('preview'):
            fav_sites = fav_sites.only('id', 'title')

        if paginate:
            queryset = paginator.paginate_queryset(fav_sites, request)
            output = output_serializer(queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        
        output = output_serializer(fav_sites, context={'request': request}, many=True).data
        return Response(output)

    @action(detail=True, methods=['post'], permission_classes=(IsAuthenticated,))
    def check_fav(self, request, pk):
        created = self.get_object().mark_as_fav(request.user)
        detail = 'Marked as favorite' if created else 'Unmarked as favorite'
        return Response({'detail': detail})

    @action(detail=True, methods=['post'], permission_classes=(IsAuthenticated,))
    def comment(self, request, pk):
        serializer = CommentInputSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        if self.get_object().comments.filter(user__id=self.request.user.id).exists():
            raise ValidationError(detail='You have already posted a review', code=400)
        
        comment = self.get_object().add_comment(
            user=request.user,
            body=serializer.validated_data.get('body'),
            rating=serializer.validated_data.get('rating'),
        )
        return Response(CommentSerializer(comment, context={'request': request}).data, status=201)

    @action(detail=False, methods=['get'], permission_classes=(AllowAny,))
    def by_map(self, request):
        paginator = PageNumberPagination()
        paginator.page_size = 25
        paginate = request.query_params.get('page')

        sites = self.get_queryset().exclude(address__isnull=True)

        if paginate:
            queryset = paginator.paginate_queryset(sites, request)
            output = SiteListSerializer(queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        
        output = SiteListSerializer(sites, context={'request': request}, many=True).data
        return Response(output)

    @action(detail=False, methods=['get'], permission_classes=(AllowAny,))
    def geolocations(self, request):
        paginator = PageNumberPagination()
        paginator.page_size = 25
        paginate = request.query_params.get('page')

        sites = self.get_queryset().exclude(address__isnull=True)

        if paginate:
            queryset = paginator.paginate_queryset(sites, request)
            output = SiteGeolocationSerializer(queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        
        output = SiteGeolocationSerializer(sites, context={'request': request}, many=True).data
        return Response(output)

    def get_user_location(self, request) -> Tuple[Optional[Point], int]:
        """
        Get user location from request or preferences.
        Returns tuple of (location_point, radius)
        """
        location_prefs = None
        if request.user.is_authenticated:
            location_prefs = LocationPreference.objects.filter(user=request.user).first()
 
            location_serializer = LocationInputSerializer(data=request.query_params)
            if location_serializer.is_valid():
                data = location_serializer.validated_data
                return (
                    Point(float(data['longitude']), float(data['latitude']), srid=4326),
                    min(int(data.get('radius', 5000)), 50000)
                ) 
            if location_prefs:
                if not location_prefs.geolocation_enabled:
                    if location_prefs.preferred_city:
                        city_point = location_prefs.preferred_city.point
                        return (
                            Point(city_point.x, city_point.y, srid=4326),
                            location_prefs.default_radius
                        )
                    raise ValidationError(_("Please enable geolocation or set a preferred city"))
                
                current_location = location_prefs.get_current_location()
                if current_location:
                    return (
                        Point(
                            current_location['longitude'],
                            current_location['latitude'],
                            srid=4326
                        ),
                        location_prefs.default_radius
                    ) 
            raise ValidationError(_("Location information not available"))
 
    def get_queryset_with_location(self, user_location: Point, radius: int, site_type: str = 'all'):
        """Get queryset filtered by location and type"""
        queryset = self.get_queryset()
        
        if site_type != 'all':
            queryset = queryset.filter_type(site_type) 
        return (
            queryset
            .exclude(address__isnull=True)
            .annotate(
                distance=Distance('address__point', user_location),
                is_open=models.Case(
                    models.When(always_open=True, then=True),
                    models.When(
                        opening_hours__day=timezone.now().strftime('%A').lower(),
                        opening_hours__from_hour__lte=timezone.now().time(),
                        opening_hours__to_hour__gte=timezone.now().time(),
                        then=True
                    ),
                    default=False,
                    output_field=models.BooleanField(),
                )
            )
            .filter(address__point__distance_lte=(user_location, D(m=radius)))
            .order_by('distance')
        )
 

    @action(detail=False, methods=['get'], url_path='near-me', url_name='near-me')
    def near_me(self, request):
        """
        Find sites near the user's current location with advanced filtering options.
        
        Query Parameters:
        - latitude: float (required)
        - longitude: float (required)
        - radius: int (optional, default=5000, max=50000)
        - type: str (optional, choices: all, place, event)
        - category: int (optional, category ID)
        - is_open: bool (optional)
        - min_rating: float (optional, 1-5)
        - price_range: str (optional, format: min-max)
        - features: str (optional, comma-separated feature IDs)
        - sort_by: str (optional, choices: distance, rating, popularity)
        - time_range: str (optional, format: YYYY-MM-DD,YYYY-MM-DD)
        """
        try:
            # Basic location validation
            latitude = request.query_params.get('latitude')
            longitude = request.query_params.get('longitude')
            if not all([latitude, longitude]):
                return Response({'detail': 'Latitude and longitude are required'}, status=400)

            try:
                lat = float(latitude)
                lon = float(longitude)
                radius = min(float(request.query_params.get('radius', 5000)), 50000)
            except ValueError:
                return Response({'detail': 'Invalid coordinate values'}, status=400)

            # Generate cache key based on all query parameters
            cache_key = self._generate_cache_key(request.query_params)
            cached_results = cache.get(cache_key)
            
            if cached_results:
                logger.info('[website] Returning cached results for near_me query')
                return Response(cached_results)

            # Create location point
            user_location = Point(lon, lat, srid=4326)

            # Start with base queryset
            queryset = self.get_queryset().exclude(address__isnull=True)

            # Apply location-based filtering
            queryset = queryset.annotate(
                distance=Distance('address__point', user_location)
            ).filter(
                address__point__distance_lte=(user_location, D(m=radius))
            )

            # Apply additional filters
            queryset = self._apply_filters(queryset, request.query_params)

            # Apply sorting
            queryset = self._apply_sorting(queryset, request.query_params.get('sort_by', 'distance'))

            # Apply pagination
            page = self.paginate_queryset(queryset)
            if page is not None:
                serializer = self.get_serializer(page, many=True)
                response_data = self.get_paginated_response(serializer.data)
            else:
                serializer = self.get_serializer(queryset, many=True)
                response_data = serializer.data

            # Add metadata
            response_data = {
                'results': response_data.data if hasattr(response_data, 'data') else response_data,
                'metadata': {
                    'total_count': queryset.count(),
                    'radius': radius,
                    'location': {
                        'latitude': lat,
                        'longitude': lon
                    },
                    'filters_applied': self._get_applied_filters(request.query_params)
                }
            }

            # Cache the results
            cache.set(cache_key, response_data, timeout=300)  # Cache for 5 minutes
            return Response(response_data)

        except Exception as e:
            logger.error(f'[website] Error in near_me: {str(e)}')
            return Response(
                {'detail': 'An error occurred while processing your request'},
                status=500
            )

    def _generate_cache_key(self, query_params):
        """Generate a unique cache key based on query parameters"""
        params = sorted([f"{k}:{v}" for k, v in query_params.items()])
        return f"near_me_{'_'.join(params)}"

    def _apply_filters(self, queryset, params):
        """Apply all filters to the queryset"""
        # Filter by type
        site_type = params.get('type')
        if site_type and site_type != 'all':
            queryset = queryset.filter(type=site_type)

        # Filter by category
        category = params.get('category')
        if category:
            queryset = queryset.filter(categories__id=category)

        # Filter by opening status
        is_open = params.get('is_open')
        if is_open:
            now = timezone.now()
            queryset = queryset.annotate(
                is_currently_open=models.Case(
                    models.When(always_open=True, then=True),
                    models.When(
                        opening_hours__day=now.strftime('%A').lower(),
                        opening_hours__from_hour__lte=now.time(),
                        opening_hours__to_hour__gte=now.time(),
                        then=True
                    ),
                    default=False,
                    output_field=models.BooleanField(),
                )
            ).filter(is_currently_open=True)

        # Filter by rating
        min_rating = params.get('min_rating')
        if min_rating:
            queryset = queryset.filter(average_rating__gte=float(min_rating))

        # Filter by price range
        price_range = params.get('price_range')
        if price_range:
            min_price, max_price = map(float, price_range.split('-'))
            queryset = queryset.filter(
                price_range__gte=min_price,
                price_range__lte=max_price
            )

        # Filter by features
        features = params.get('features')
        if features:
            feature_ids = features.split(',')
            for feature_id in feature_ids:
                queryset = queryset.filter(features__id=feature_id)

        # Filter by time range
        time_range = params.get('time_range')
        if time_range:
            start_date, end_date = map(
                lambda x: timezone.datetime.strptime(x, '%Y-%m-%d').date(),
                time_range.split(',')
            )
            queryset = queryset.filter(
                models.Q(event_date__range=(start_date, end_date)) |
                models.Q(event_date__isnull=True)
            )

        return queryset

    def _apply_sorting(self, queryset, sort_by):
        """Apply sorting to the queryset"""
        if sort_by == 'rating':
            return queryset.order_by('-average_rating', 'distance')
        elif sort_by == 'popularity':
            return queryset.order_by('-visit_count', 'distance')
        return queryset.order_by('distance')  # Default sorting by distance

    def _get_applied_filters(self, params):
        """Get a list of filters that were applied to the query"""
        applied_filters = {}
        filter_fields = [
            'type', 'category', 'is_open', 'min_rating',
            'price_range', 'features', 'time_range', 'sort_by'
        ]
        for field in filter_fields:
            if params.get(field):
                applied_filters[field] = params[field]
        return applied_filters


    @action(detail=False, methods=['get'])
    def discover_around_me(self, request):
        """Enhanced discovery endpoint with location awareness"""
        try:
            user_location, radius = self.get_user_location(request)

            cache_key = f"discover_{user_location.y}_{user_location.x}_{radius}"
            cached_results = cache.get(cache_key)
            if cached_results:
                return Response(cached_results)

            places = self.get_queryset_with_location(
                user_location, radius, 'place'
            )[:10]


            events = self.get_queryset_with_location(
                user_location, radius, 'event'
            ).filter(
                event_date__gte=timezone.now().date()
            )[:10]

            location_prefs = None
            if request.user.is_authenticated:
                location_prefs = LocationPreference.objects.filter(user=request.user).first()

            combined_results = {
                'places': SiteListSerializer(
                    places, 
                    many=True,
                    context={'request': request}
                ).data,
                'events': SiteListSerializer(
                    events, 
                    many=True,
                    context={'request': request}
                ).data,
                'user_location': {
                    'latitude': user_location.y,
                    'longitude': user_location.x
                },
                'preferences': {
                    'geolocation_enabled': location_prefs.geolocation_enabled if location_prefs else True,
                    'default_radius': location_prefs.default_radius if location_prefs else 5000
                } if request.user.is_authenticated else None
            }
            cache.set(cache_key, combined_results, timeout=300)
            return Response(combined_results)

        except ValidationError as e:
            return Response({'detail': str(e)}, status=400)
        except Exception as e:
            logger.error(f'[website] Error in discover_around_me: {str(e)}')
            return Response(
                {'detail': 'An error occurred while processing your request'},
                status=500
            )

    def get_discovery_by_preferred_location(self, request, location_prefs):
        """
        Get discovery results based on user's preferred city
        """
        if not location_prefs.preferred_city:
            raise ValidationError(_("Please set a preferred city in your profile"))
            
        city_location = location_prefs.preferred_city.point
        return self.discover_around_me(request._replace(
            query_params={
                'latitude': city_location.y,
                'longitude': city_location.x,
                'radius': location_prefs.default_radius
            }
        ))

    @action(detail=True, methods=['get'])
    def nearby(self, request, pk):
        paginator = PageNumberPagination()
        paginator.page_size = 25
        paginate = request.query_params.get('page')

        is_cache_applicable = len(request.query_params) <= 1
        serializer = SiteListSerializer
        if request.query_params.get('preview'):
            serializer = SiteRandomListSerializer

        sites = None
        if is_cache_applicable:
            sites = cache.get(f'nearby_{pk}')
        
        if sites is None:
            site = Site.objects.get(id=pk)
            sites = self.get_queryset().filter_country_id(site.city.country.id)
            
            point = site.address.point if site.address else site.city.point
            
            if sites.exists():
                sites = (
                    sites.exclude(id=pk)
                    .annotate(distance=Distance('address__point', point))
                    .filter(distance__lte=50000)  # 50 Km
                    .order_by('distance')
                    .distinct()
                )
                
                if sites.count() > 50:
                    sites = sites[:50]
                
                if is_cache_applicable:
                    cache.set(f'nearby_{pk}', sites)

        if paginate:
            queryset = paginator.paginate_queryset(sites, request)
            output = serializer(queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        
        output = serializer(sites, many=True, context={'request': request}).data
        return Response(output)
    
    @action(detail=True, methods=['get'])
    def comments(self, request, pk):
        comments = (
            self.get_object()
            .comments.all()
            .annotate(
                is_from_user=Case(
                    When(user__id=request.user.id, then=True),
                    default=False
                )
            )
        ).order_by('-created_at')

        paginator = PageNumberPagination()
        paginator.page_size = 25
        paginate = request.query_params.get('page')

        if paginate:
            paginated_queryset = paginator.paginate_queryset(comments, request)
            output = CommentSerializer(paginated_queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        
        return Response(CommentSerializer(comments, context={'request': request}, many=True).data)

    @action(detail=True, methods=['get'])
    def similar(self, request, pk):
        tags = self.get_object().tags.all()
        sites = self.get_queryset().filter(tags__id__in=tags).exclude(id=self.get_object().id)
        return Response(SiteListSerializer(sites, context={'request': request}, many=True).data)

    @action(detail=False, methods=['post'])
    def suggest_site(self, request):
        serializer = SiteCreationSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        location = serializer.validated_data.pop('location')

        schedules = serializer.validated_data.pop('schedules', [])
        special_schedules = serializer.validated_data.pop('special_schedules', [])

        site = Site.objects.create(
            has_been_accepted=False,
            created_by=request.user,
            **serializer.validated_data
        )

        site.create_address(location)
        site.add_schedules(schedules)
        site.add_special_schedules(special_schedules)

        return Response(SiteSerializer(site, context={'request': request}).data, status=201)

    @action(detail=True, methods=['post'])
    def add_images(self, request, pk):
        site = Site.objects.get(id=pk)
        if site.created_by.id != request.user.id:
            raise PermissionDenied(detail='You can only add images to your created sites', code=400)
        
        files = request.FILES.getlist('images')
        img_sizes = ImageSize.objects.first()

        for file in files:
            width, height = get_image_dimensions(file)
            if not (img_sizes.max_width > width > img_sizes.min_width and 
                   img_sizes.max_height > height > img_sizes.min_height):
                raise ValidationError(
                    detail=f'The given images must size between {img_sizes.min_width} - '
                    f'{img_sizes.min_height} and {img_sizes.max_width} - {img_sizes.max_height}',
                    code=400,
                )
            site.images.create(image=file)
        
        return Response({'detail': 'Images were added successfully'}, status=200)

    @action(detail=True, methods=['get'])
    def generate_link(self, request, pk):
        link = f'https://{request.get_host()}/sites/redirect_detail?id={pk}'
        return Response({'detail': link}, status=200)

    @action(detail=False, methods=['get'])
    def redirect_detail(self, request):
        user_agent = request.META['HTTP_USER_AGENT'].lower()
        
        if user_agent not in ['local-secrets']:
            if 'android' in user_agent:
                return redirect('https://play.google.com/store')
            elif any(platform in user_agent for platform in ['ios', 'iphone', 'apple', 'darwin', 'mac']):
                return redirect('https://apps.apple.com/us/app/local-secrets/id6448513168')
            else:
                return redirect('https://play.google.com/store')

        site_id = request.query_params.get('id')
        site = self.get_queryset().filter(id=site_id).first()
        
        if not site:
            return Response({'detail': f'There is no site with id: {site_id}'}, status=404)
        
        return Response(SiteSerializer(site, context={'request': request}).data, status=200)
    
    @action(detail=False, methods=['get'])
    def sites_to_discover(self, request):
        cache_key = 'sites_to_discover'
        paginator = PageNumberPagination()
        paginator.page_size = 10
        paginate = request.query_params.get('page')
        
        # Select serializer based on preview parameter
        serializer = SitestoDiscoverSerializer
        if request.query_params.get('preview'):
            serializer = SiteRandomListSerializer

        # Try to get from cache
        result_list = cache.get(cache_key)
        
        if result_list is None:
            # Get sites and cities with random ordering
            sites = self.get_queryset().filter(type='place').annotate(order=Random() * 100)
            cities = City.objects.filter(activated=True).annotate(order=Random() * 100)

            # Apply city filter if present
            language = self.request.headers.get('language')
            if 'city' in self.request.query_params:
                city = text.remove_all_accent_marks(self.request.query_params.get('city').lower())
                if language:
                    if language == 'es':
                        cities = cities.filter(name__icontains=city)
                    else:
                        cities = cities.filter(translations__name__icontains=city)
                else:
                    cities = cities.filter(name__icontains=city)

            # Combine and sort results
            result_list = sorted(chain(sites, cities), key=attrgetter('order'))[:100]
            cache.set(cache_key, result_list)

        # Handle pagination
        if paginate:
            paginated_queryset = paginator.paginate_queryset(result_list, request)
            output = serializer(paginated_queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        
        output = serializer(result_list, context={'request': request}, many=True).data
        return Response(output)

    @action(detail=False, methods=['get'])
    def random(self, request):
        paginator = PageNumberPagination()
        paginator.page_size = 25
        paginate = request.query_params.get('page')

        if paginate:
            # Handle paginated response
            sites = cache.get('random_cache')
            if not sites:
                sites = (
                    Site.objects.only('id', 'title', 'city')
                    .select_related('city')
                    .prefetch_related('images')
                    .order_by('?', 'id')[:100]
                )
                cache.set('random_cache', sites)

            paginated_queryset = paginator.paginate_queryset(sites, request)
            serializer = SiteRandomListSerializer(
                paginated_queryset,
                many=True,
                context={'request': request}
            )
            return paginator.get_paginated_response(serializer.data)
        else:
            # Handle non-paginated response
            cached_data = cache.get('random_basic')
            if cached_data:
                return Response(cached_data)

            sites = (
                Site.objects.only('id', 'title', 'city')
                .select_related('city')
                .prefetch_related('images')
                .order_by('?', 'id')[:50]
            )
            
            serializer = SiteRandomListSerializer(
                sites,
                many=True,
                context={'request': request}
            )
            output = serializer.data
            cache.set('random_basic', output)
            return Response(output, status=200)

    @action(detail=False, methods=['get'])
    def suggested(self, request):
        paginator = PageNumberPagination()
        paginator.page_size = 25
        paginate = request.query_params.get('page')

        sites = self.get_queryset().filter_suggested(True)
        sites = sites.order_by('?')

        if paginate:
            queryset = paginator.paginate_queryset(sites[:100], request)
            output = SiteListSerializer(queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        
        output = SiteListSerializer(sites[:100], many=True, context={'request': request}).data
        return Response(output, status=200)

    def retrieve(self, request, *args, **kwargs):
        """
        Override retrieve to count views
        """
        instance = self.get_object()
        # View will be counted by middleware
        serializer = self.get_serializer(instance)
        return Response(serializer.data)
 
    @action(detail=True, methods=['post'])
    def record_view(self, request, pk=None):
        """
        Explicit endpoint for recording views (e.g., for SPA applications)
        """
        instance = self.get_object()
        instance.increment_view_count()
        return Response({'view_count': instance.total_views})


class PromotionPagination(PageNumberPagination):
    page_size = 10
    page_size_query_param = 'page_size'
    max_page_size = 100


class SitePromotionViewSet(viewsets.ModelViewSet):
    pagination_class = PromotionPagination
    serializer_class = PromotionSerializer
    permission_classes = [IsAuthenticated, IsVendorOwner, HasActiveSubscription]

    def get_queryset(self):
        """Filter promotions based on status and dates"""
        queryset = Promotion.objects.filter(vendor=self.request.user.vendor)
        
        # Filter by active status
        if self.action == 'active':
            queryset = queryset.filter(
                is_active=True,
                start_date__lte=timezone.now(),
                end_date__gte=timezone.now()
            )
            
        # Filter by promotion type if specified
        promotion_type = self.request.query_params.get('type')
        if promotion_type:
            queryset = queryset.filter(target_type=promotion_type)
            
        return queryset
 
    def get_serializer_class(self):
        if self.action == 'list':
            return SitePromotionListSerializer
        return PromotionSerializer
 
    @action(detail=False, methods=['get'])
    def active(self, request):
        """Get only active promotions"""
        active_promotions = self.get_queryset()
        serializer = self.get_serializer(active_promotions, many=True)
        return Response(serializer.data)
 
    @action(detail=True, methods=['post'])
    def activate(self, request, pk=None):
        """Activate a specific promotion"""
        promotion = self.get_object()
        if promotion.start_date > timezone.now():
            return Response(
                {'error': _("Cannot activate future promotion")},
                status=status.HTTP_400_BAD_REQUEST
            )
        promotion.is_active = True
        promotion.save()
        return Response({'status': 'promotion activated'})
 
    @action(detail=True, methods=['post'])
    def deactivate(self, request, pk=None):
        """Deactivate a specific promotion"""
        promotion = self.get_object()
        promotion.is_active = False
        promotion.save()
        return Response({'status': 'promotion deactivated'})
 
    def perform_create(self, serializer):
        """Create a new promotion"""
        try:
            site = Site.objects.get(
                pk=self.request.data.get('site'),
                vendor=self.request.user.vendor
            )
            if not site.can_have_promotion():
                raise ValidationError(_("Site cannot have promotions"))
                
            serializer.save(
                vendor=self.request.user.vendor,
                site=site
            )
        except Site.DoesNotExist:
            raise ValidationError(_("Invalid site selected"))
        except ValidationError as e:
            raise ValidationError(detail=str(e))
 
    @action(detail=True, methods=['post'])
    def cancel(self, request, pk=None):
        """Cancel an existing promotion"""
        promotion = self.get_object()
        try:
            promotion.is_active = False
            promotion.end_date = timezone.now()
            promotion.save()
            return Response(status=status.HTTP_204_NO_CONTENT)
        except ValidationError as e:
            return Response(
                {'error': str(e)}, 
                status=status.HTTP_400_BAD_REQUEST
            )
 
    @action(detail=True, methods=['patch'])
    def update_promotion(self, request, pk=None):
        """Update an existing promotion"""
        promotion = self.get_object()
        serializer = self.get_serializer(
            promotion, 
            data=request.data, 
            partial=True
        )
        serializer.is_valid(raise_exception=True)
        try:
            serializer.save()
            return Response(serializer.data)
        except ValidationError as e:
            return Response(
                {'error': str(e)}, 
                status=status.HTTP_400_BAD_REQUEST
            )


class SiteVisibilityViewSet(viewsets.ModelViewSet):
    permission_classes = [IsAuthenticated]
    serializer_class = SiteVisibilitySerializer
    def get_queryset(self):
        if self.request.user.is_staff:
            return Site.objects.all()
        return Site.objects.filter(vendor=self.request.user.vendor)


    @action(detail=True, methods=['post'])
    def update_visibility(self, request, pk=None):
        site = self.get_object()
        serializer = VisibilityUpdateSerializer(data=request.data)
        
        if serializer.is_valid():
            from local_secrets.sites.services.site_visibility import VisibilityService
            service = VisibilityService()
            
            result = service.update_site_visibility(
                site=site,
                new_visibility=serializer.validated_data['visibility'],
                reason=serializer.validated_data.get('reason'),
                metadata=serializer.validated_data.get('metadata')
            )
            
            if result:
                return Response({'status': 'visibility updated'})
            return Response(
                {'error': 'Failed to update visibility'},
                status=status.HTTP_400_BAD_REQUEST
            )
            
        return Response(
            serializer.errors,
            status=status.HTTP_400_BAD_REQUEST
        )


    @action(detail=True, methods=['post'])
    def reset_visibility(self, request, pk=None):
        site = self.get_object()
        site.visibility_update_pending = False
        site.update_attempts = 0
        site.save()
        
        return Response({'status': 'visibility reset'})


    @action(detail=False, methods=['get'])
    def stats(self, request):
        queryset = self.get_queryset()
        
        stats = {
            'total_sites': queryset.count(),
            'visibility_counts': dict(
                queryset.values('visibility')
                .annotate(count=Count('id'))
                .values_list('visibility', 'count')
            ),
            'pending_updates': queryset.filter(
                visibility_update_pending=True
            ).count(),
            'failed_attempts': queryset.filter(
                update_attempts__gte=settings.VISIBILITY_MAX_RETRIES
            ).count()
        }
        
        serializer = VisibilityStatsSerializer(stats)
        return Response(serializer.data)


    @action(detail=True, methods=['get'])
    def events(self, request, pk=None):
        site = self.get_object()
        events = SystemEvent.objects.filter(
            event_type__startswith='site.visibility.',
            payload__site_id=site.id
        ).order_by('-created_at')
        
        serializer = VisibilityEventSerializer(events, many=True)
        return Response(serializer.data)
 
class DefaultImageViewSet(viewsets.GenericViewSet, mixins.ListModelMixin):
    def get_queryset(self):
        return DefaultImage.objects.all()

    def get_serializer_class(self):
        return DefaultImageSerializer

class CommentViewSet(
    viewsets.GenericViewSet, mixins.RetrieveModelMixin, mixins.UpdateModelMixin, mixins.DestroyModelMixin
):
    permission_classes = (IsAuthenticatedForPost,)
    paginate_by = 35

    @property
    def paginator(self):
        self._paginator = super(CommentViewSet, self).paginator
        if not self.request.query_params.get('page'):
            self._paginator = None
        return self._paginator

    def list(self, request, *args, **kwargs):
        queryset = self.get_queryset()

        paginator = PageNumberPagination()
        paginator.page_size = 25
        paginate = request.query_params.get('page')

        if paginate:
            paginated_queryset = paginator.paginate_queryset(queryset, request)
            output = self.get_serializer_class()(paginated_queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        else:
            return Response(self.get_serializer_class()(queryset).data)

    def update(self, request, *args, **kwargs):
        if self.get_object().user != self.request.user:
            return Response({'detail': 'You do not have permissions to modify this comment'}, status=402)
        return super().update(request, *args, **kwargs)

    def destroy(self, request, *args, **kwargs):
        if self.get_object().user != self.request.user:
            return Response({'detail': 'You do not have permissions to delete this comment'}, status=402)
        super().destroy(request, *args, **kwargs)
        return Response({'detail': 'The comment was deleted'}, status=200)

    def get_queryset(self):
        return self.request.user.comments.all()

    def get_serializer_class(self):
        return CommentSerializer

    def get_serializer_context(self):
        return {'request': self.request}

class VendorSiteViewSet(viewsets.ModelViewSet):
    permission_classes = [IsAuthenticated]
    
    def get_serializer_class(self):
        if self.action in ['create', 'update', 'partial_update']:
            return VendorSiteCreateSerializer
        return VendorSiteRetrieveSerializer

    @csrf_exempt
    def dispatch(self, *args, **kwargs):
        return super().dispatch(*args, **kwargs)

    def get_queryset(self):
        return Site.objects.filter(vendor=self.request.user.vendor_profile)

    def perform_create(self, serializer):
        serializer.save(vendor=self.request.user.vendor_profile)
 
class SiteReviewViewSet(viewsets.ModelViewSet):
    serializer_class = SiteReviewSerializer
    permission_classes = [IsAuthenticated, IsReviewerPermission]

    def get_queryset(self):
        if self.request.user.is_staff:
            return SiteReview.objects.all()
        # Ambassadors can only see their reviews
        return SiteReview.objects.filter(reviewer=self.request.user)

    def perform_create(self, serializer):
        serializer.save(reviewer=self.request.user)

    @action(detail=True, methods=['post'])
    def submit_review(self, request, pk=None):
        review = self.get_object()
        
        # Check if user has permission to modify this review
        if not request.user.is_staff and review.reviewer != request.user:
            raise PermissionDenied("You don't have permission to modify this review")

        serializer = self.get_serializer(review, data=request.data, partial=True)
        
        if serializer.is_valid():
            review = serializer.save()
            # Update site status based on review
            review.site.status = review.status
            review.site.save()
            return Response(serializer.data)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

class ReviewChecklistItemViewSet(viewsets.ReadOnlyModelViewSet):
    queryset = ReviewChecklistItem.objects.all()
    serializer_class = ReviewChecklistItemSerializer
    permission_classes = [IsAuthenticated]
 
class SitePublicationViewSet(viewsets.ModelViewSet):
    permission_classes = [IsAuthenticated, IsVendorOrAdmin]
    serializer_class = SitePublicationSerializer

    def get_queryset(self):
        if self.request.user.is_staff:
            return Site.objects.all()
        return Site.objects.filter(vendor__user=self.request.user)

    @action(detail=True, methods=['post'])
    def submit_for_review(self, request, pk=None):
        site = self.get_object()
        workflow = ReviewWorkflowManager(site)
        
        try:
            review = workflow.submit_for_review()
            return Response({
                'status': 'success',
                'review_id': review.id
            })
        except ValidationError as e:
            return Response({
                'status': 'error',
                'errors': e.messages
            }, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=['post'])
    def publish(self, request, pk=None):
        site = self.get_object()
        try:
            SitePublicationValidator.validate_publication_requirements(site)
            site.publish_site()
            return Response({'status': 'success'})
        except ValidationError as e:
            return Response({
                'status': 'error',
                'errors': e.messages
            }, status=status.HTTP_400_BAD_REQUEST)

class VendorSiteListView(LoginRequiredMixin, ListView):
    template_name = 'sites/vendor/site_list.html'
    context_object_name = 'sites'

    def get_queryset(self):
        queryset = Site.objects.filter(
            vendor__user=self.request.user
        ).select_related('active_subscription')
        logger.info(queryset)
        # If this is a location-based request, filter by location
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            try:
                latitude = float(self.request.GET.get('latitude'))
                logger.info(latitude)
                longitude = float(self.request.GET.get('longitude'))
                logger.info(longitude)
                radius = float(self.request.GET.get('radius', 5000))
                logger.info(radius)
                site_type = self.request.GET.get('type', 'all')

                user_location = Point(longitude, latitude, srid=4326)
                
                queryset = queryset.annotate(
                    distance=Distance('address__point', user_location)
                ).filter(
                    distance__lte=radius  # Filter by radius (in meters)
                )

                if site_type != 'all':
                    queryset = queryset.filter(type=site_type)

                queryset = queryset.order_by('distance')

            except (ValueError, TypeError):
                pass  # If coordinates are invalid, return unfiltered queryset

        return queryset

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        queryset = self.get_queryset()
        
        context.update({
            'pending_reviews': queryset.filter(
                status=SiteStatusChoices.PENDING_REVIEW
            ).count(),
            'published_sites': queryset.filter(
                status=SiteStatusChoices.PUBLISHED
            ).count()
        }) 
        return context

    def render_to_response(self, context, **response_kwargs):
        if self.request.headers.get('X-Requested-With') == 'XMLHttpRequest':
            sites_data = []
            for site in context['sites']:
                sites_data.append({
                    'id': site.id,
                    'title': site.title,
                    'type': site.type,
                    'city_name': site.city.name if site.city else 'N/A',
                    'status': site.status,
                    'is_open': site.is_open if hasattr(site, 'is_open') else None,
                    'avg_rating': float(site.avg_rating) if hasattr(site, 'avg_rating') else None,
                    'distance': float(site.distance.m) if hasattr(site, 'distance') else None,
                    'address': str(site.address) if site.address else None,
                    'image': site.images.first().image.url if site.images.exists() else None,
                    'created_at': site.created_at.strftime('%Y-%m-%d %H:%M:%S'),
                    'latitude': site.address.latitude if site.address else None,
                    'subscription_status': site.active_subscription.status if site.active_subscription else None
                })
            logger.info(sites_data)
            return JsonResponse({
                'results': sites_data,
                'count': len(sites_data),
                'pending_reviews': context['pending_reviews'],
                'published_sites': context['published_sites']
            })
        
        return super().render_to_response(context, **response_kwargs)

class VendorSiteDetailView(LoginRequiredMixin, DetailView):
    template_name = 'sites/vendor/site_detail.html'
    context_object_name = 'site'
    
    def get_queryset(self):
        return Site.objects.filter(vendor__user=self.request.user)
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        site = self.get_object()
        context.update({
            'reviews': site.reviews.order_by('-created_at'),
            #'latest_review': site.latest_review,
            'subscription_valid': site.has_valid_subscription(),
            'can_publish': site.can_be_published(),

        })
        return context

class SearchIntent:
    def __init__(self, query, location=None, category=None, filters=None):
        self.query = query
        self.location = location
        self.category = category
        self.filters = filters or {}


class SmartSiteSearch(APIView):
    LOCATION_KEYWORDS = [
        'near me', 'around me', 'nearby', 'close to me', 
        'in my area', 'my location', 'around my location',
        'near my position', 'around here', 'close by'
    ]
    SEARCH_PATTERNS = {
    'question_starters': [
        'what', 'where', 'when', 'which', 'how',
        'can you', 'could you', 'would you'
    ],
    'location_patterns': [
        'in', 'at', 'near', 'around', 'close to'
    ],
    'type_indicators': [
        'restaurant', 'cafe', 'bar', 'shop', 'store',
        'venue', 'place', 'location'
    ],
    'attribute_patterns': [
        'top rated', 'best rated', 'popular', 'new',
        'open', 'available', 'premium'
    ],
    'time_patterns': [
        'now', 'today', 'tonight', 'tomorrow',
        'this week', 'this weekend'
    ]
}
 
    QUERY_ANALYSIS_PROMPT = """
    Analyze the search query and extract search parameters. Pay special attention to location context.
    Return a JSON with:
    {
        "understood": boolean,
        "search_params": {
            "search_terms": list of search terms,
            "location_type": one of ["user_location", "specified_location", "no_location"],
            "location_text": specified location or null,
            "radius_preference": "near" or "around" or null,
            "type": "place" or "event" or null,
            "time_context": time reference or null,
            "attributes": list of attributes
        },
        "needs_clarification": boolean,
        "guidance": {
            "understood_parts": what was understood from the query,
            "missing_info": what information would help clarify the search,
            "suggestions": list of suggested query reformulations,
            "examples": list of example queries that would work well
        },
        "error_type": null or one of ["ambiguous", "incomplete", "invalid_format", "unknown"]
    }
    """

    def __init__(self):
        super().__init__()
        self.client = OpenAI(api_key=settings.OPENAI_API_KEY)
 
    def get_user_location(self, request):
        """Get user's location from various sources"""
        try:
            # First check request data for coordinates
            lat = request.data.get('latitude')
            lng = request.data.get('longitude')
            
            if lat and lng:
                return {
                    'latitude': float(lat),
                    'longitude': float(lng),
                    'source': 'request',
                    'radius': float(request.data.get('radius', 5000))  # Default 5km
                }
            
            # Check user preferences if authenticated
            if request.user.is_authenticated:
                location_pref = LocationPreference.objects.filter(
                    user=request.user,
                    geolocation_enabled=True
                ).first()
                
                if location_pref and location_pref.latitude and location_pref.longitude:
                    return {
                        'latitude': float(location_pref.latitude),
                        'longitude': float(location_pref.longitude),
                        'source': 'preferences',
                        'radius': float(location_pref.radius or 5000)
                    }
            
            return None
        except Exception as e:
            logger.error(f"Error getting user location: {e}")
            return None
    
    def parse_natural_query(self, query):
        """Enhanced natural language query parser"""
        # Normalize query
        original_query = query.lower().strip()
        query_parts = {
            'search_terms': [],
            'location': None,
            'type': None,
            'attributes': [],
            'time_context': None,
            'is_question': False
        }

        # Remove question marks and normalize punctuation
        query = re.sub(r'[?!.]', '', original_query)
        
        # Check if it's a question
        query_parts['is_question'] = any(q in query for q in self.SEARCH_PATTERNS['question_starters'])
        
        # Remove common patterns and clean up the query
        cleaned_query = query
        for pattern_type, patterns in self.SEARCH_PATTERNS.items():
            for pattern in patterns:
                if pattern in cleaned_query:
                    if pattern_type == 'location_patterns':
                        # Extract location
                        location_match = re.search(f"{pattern}\s+([^,\.]+)", cleaned_query)
                        if location_match:
                            query_parts['location'] = location_match.group(1).strip()
                    
                    elif pattern_type == 'type_indicators':
                        # Extract type
                        if pattern in cleaned_query:
                            query_parts['type'] = pattern.rstrip('s')  # Remove plural 's'
                    
                    elif pattern_type == 'attribute_patterns':
                        # Extract attributes
                        query_parts['attributes'].append(pattern)
                    
                    elif pattern_type == 'time_patterns':
                        # Extract time context
                        query_parts['time_context'] = pattern
                    
                    # Remove the pattern from the query
                    cleaned_query = cleaned_query.replace(pattern, '').strip()

        # Clean up remaining text for search terms
        cleaned_terms = [
            term.strip() for term in cleaned_query.split()
            if term.strip() and len(term.strip()) > 1
            and term not in sum(self.SEARCH_PATTERNS.values(), [])
        ]
        query_parts['search_terms'] = cleaned_terms

        return query_parts

    def build_enhanced_search_query(self, query_parts):
        """Build enhanced search query with context"""
        queries = Q()
        
        # Build search terms query
        for term in query_parts['search_terms']:
            term_query = Q()
            # Exact match (higher priority)
            term_query |= (
                Q(title__iexact=term) |
                Q(tags__title__iexact=term) |
                Q(categories__title__iexact=term)
            )
            # Contains match
            term_query |= (
                Q(title__icontains=term) |
                Q(description__icontains=term) |
                Q(tags__title__icontains=term) |
                Q(categories__title__icontains=term)
            )
            # Starts with match
            term_query |= (
                Q(title__istartswith=term) |
                Q(tags__title__istartswith=term) |
                Q(categories__title__istartswith=term)
            )
            queries &= term_query

        # Add type filter
        if query_parts['type']:
            queries &= Q(type=query_parts['type'])

        # Add location filter
        if query_parts['location']:
            queries &= (
                Q(city__name__icontains=query_parts['location']) |
                Q(address__city__name__icontains=query_parts['location'])
            )

        # Add attribute filters
        for attr in query_parts['attributes']:
            if attr in ['top rated', 'best rated']:
                queries &= Q(is_top_10=True)
            elif attr == 'open':
                queries &= Q(always_open=True)
            elif attr == 'premium':
                queries &= Q(subscription_state='PREMIUM')

        return queries

    def _build_search_params(self, analysis):
        """Build enhanced search parameters from analysis"""
        search_params = {
        'status': 'approved',  # Only show approved sites
        'page_size': 10
        }
        # Extract search terms
        if analysis['search_params']['search_terms']:
            search_params['search'] = ' '.join(analysis['search_params']['search_terms'])
        
        # Add type filter if specified
        if analysis['search_params']['type']:
            search_params['type'] = analysis['search_params']['type']
        else:
            # Check search terms for type hints
            search_terms = [term.lower() for term in analysis['search_params']['search_terms']]
            if 'event' in search_terms:
                search_params['type'] = 'event'
            elif 'place' in search_terms:
                search_params['type'] = 'place'
        
        return search_params
    
    def _handle_location_based_search(self, request, search_params):
        """Enhanced location-based search handler with better error handling and caching"""
        user_location = self.get_user_location(request)
        
        if not user_location:
            # Try to get location from request body if not in user preferences
            lat = request.data.get('latitude')
            lng = request.data.get('longitude')
            if lat and lng:
                user_location = {
                    'latitude': float(lat),
                    'longitude': float(lng),
                    'source': 'request',
                    'radius': float(request.data.get('radius', 5000))
                }
            else:
                raise ValidationError({
                    "error": "Location information required",
                    "code": "LOCATION_REQUIRED",
                    "details": "Please provide latitude and longitude or enable location services"
                })

        # Generate cache key based on location and search params
        cache_key = f"location_search_{user_location['latitude']}_{user_location['longitude']}_{hash(frozenset(search_params.items()))}"
        
        # Try to get cached results first
        cached_results = cache.get(cache_key)
        if cached_results:
            return cached_results

        try:
            # Try discover endpoint first
            results = self._call_discover_endpoint(request, user_location, search_params)
            cache.set(cache_key, results, timeout=300)  # Cache for 5 minutes
            return results
        except Exception as e:
            logger.warning(f"Discover endpoint failed, trying near_me: {e}")
            try:
                # Fallback to near_me endpoint
                results = self._call_near_me_endpoint(user_location, search_params)
                cache.set(cache_key, results, timeout=300)
                return results
            except Exception as ne:
                logger.error(f"Both search endpoints failed: {ne}")
                raise ValidationError({
                    "error": "Search service unavailable",
                    "code": "SERVICE_UNAVAILABLE",
                    "details": str(ne)
                })

    def _call_discover_endpoint(self, request, location, search_params):
        """Enhanced discover endpoint integration with timeout and retry logic"""
        discover_url = f"{settings.API_BASE_URL}/sites/discover_around_me"
        
        # Merge location data with search params
        params = {
            **search_params,
            'latitude': location['latitude'],
            'longitude': location['longitude'],
            'radius': location.get('radius', 5000),
            'user': request.user.id if request.user.is_authenticated else None
        }
        
        # Add additional context parameters
        if search_params.get('time_context'):
            params['time_range'] = search_params['time_context']
        if search_params.get('attributes'):
            params['features'] = ','.join(search_params['attributes'])
        
        # Setup retry strategy
        retry_strategy = Retry(
            total=3,
            backoff_factor=0.5,
            status_forcelist=[500, 502, 503, 504]
        )
        
        with requests.Session() as session:
            session.mount('http://', HTTPAdapter(max_retries=retry_strategy))
            session.mount('https://', HTTPAdapter(max_retries=retry_strategy))
            
            response = session.get(
                discover_url,
                params=params,
                headers=self._get_request_headers(request),
                timeout=(3.05, 10)  # (connect timeout, read timeout)
            )
            
            response.raise_for_status()
            return response.json()

    def _call_near_me_endpoint(self, location: Dict[str, float], search_params: Dict[str, Any]) -> Dict[str, Any]:
        """Enhanced near-me endpoint integration with improved error handling and validation"""
        try:
            # Validate location data
            if not all(isinstance(location.get(k), float) for k in ['latitude', 'longitude']):
                raise ValidationError({
                    "error": "Invalid location format",
                    "code": "INVALID_LOCATION",
                    "details": "Latitude and longitude must be valid numbers"
                })

            # Build enhanced parameters
            params = {
                'latitude': location['latitude'],
                'longitude': location['longitude'],
                'radius': location.get('radius', 5000),
                'type': search_params.get('type', 'all'),
                'category': search_params.get('category'),
                'is_open': search_params.get('is_open'),
                'min_rating': search_params.get('min_rating'),
                'price_range': search_params.get('price_range'),
                'features': ','.join(search_params.get('features', [])),
                'sort_by': search_params.get('sort_by', 'distance'),
                'time_range': search_params.get('time_range'),
                'page': search_params.get('page', 1),
                'page_size': min(int(search_params.get('page_size', 25)), 100)
            }

            # Remove None values
            params = {k: v for k, v in params.items() if v is not None}
            
            # Make request with enhanced error handling
            try:
                response = requests.get(
                    f"{settings.API_BASE_URL}/sites/near-me",
                    params=params,
                    headers=self._get_request_headers(),
                    timeout=10
                )
                response.raise_for_status()
                
                # Process and enhance results
                results = response.json()
                if results.get('results'):
                    results['results'] = self._enhance_results(results['results'])
                
                return results

            except requests.Timeout:
                raise ValidationError({
                    "error": "Search timeout",
                    "code": "TIMEOUT",
                    "details": "The search operation took too long to complete"
                })
            except requests.RequestException as e:
                logger.error(f'[website] Error calling near-me endpoint: {str(e)}')
                raise ValidationError({
                    "error": "Search failed",
                    "code": "REQUEST_FAILED",
                    "details": str(e)
                })

        except Exception as e:
            logger.error(f'[website] Unexpected error in near-me search: {str(e)}')
            raise

    def _enhance_results(self, results: List[Dict]) -> List[Dict]:
        """Enhance search results with additional data and formatting"""
        enhanced_results = []
        
        for result in results:
            # Add distance formatting
            if 'distance' in result:
                result['distance_formatted'] = {
                    'meters': round(result['distance'], 2),
                    'kilometers': round(result['distance'] / 1000, 2),
                    'readable': f"{round(result['distance'] / 1000, 1)} km"
                }
            
            # Add opening status if available
            if 'opening_hours' in result:
                result['is_open_now'] = self._check_if_open(result['opening_hours'])
            
            # Add rating summary if available
            if 'ratings' in result:
                result['rating_summary'] = self._calculate_rating_summary(result['ratings'])
            
            enhanced_results.append(result)
        
        return enhanced_results

    def _check_if_open(self, opening_hours: Dict) -> bool:
        """Check if a site is currently open"""
        try:
            if not opening_hours:
                return None
                
            current_time = datetime.now()
            day_of_week = current_time.strftime('%A').lower()
            
            if day_of_week not in opening_hours:
                return None
                
            today_hours = opening_hours[day_of_week]
            if not today_hours:
                return False
                
            current_minutes = current_time.hour * 60 + current_time.minute
            
            for period in today_hours:
                open_minutes = self._time_to_minutes(period['open'])
                close_minutes = self._time_to_minutes(period['close'])
                
                if open_minutes <= current_minutes <= close_minutes:
                    return True
                    
            return False
        except Exception:
            return None

    def _time_to_minutes(self, time_str: str) -> int:
        """Convert time string to minutes since midnight"""
        try:
            hours, minutes = map(int, time_str.split(':'))
            return hours * 60 + minutes
        except Exception:
            return 0

    def _get_geolocations(self, request):
        """Get geolocation data for sites"""
        geo_url = f"{settings.API_BASE_URL}/sites/geolocations"
        response = requests.get(
            geo_url,
            headers=self._get_request_headers(request)
        )
        response.raise_for_status()
        return response.json()

    def _get_request_headers(self, request):
        """Get headers for API requests"""
        headers = {
            'Accept': 'application/json',
            'X-Request-ID': request.headers.get('X-Request-ID', ''),
            'X-Client-Version': request.headers.get('X-Client-Version', '')
        }
        
        # Forward authorization if present
        if 'Authorization' in request.headers:
            headers['Authorization'] = request.headers['Authorization']
            
        return headers

    def _enhance_with_geo_data(self, results, geo_data):
        """Enhance results with geolocation data"""
        geo_map = {item['id']: item['address'] for item in geo_data}
        
        for result in results:
            if result['id'] in geo_map:
                result['address'] = geo_map[result['id']]
        
        return results

    def _format_response(self, results, analysis):
        """Format response with quantity limits and enhanced filtering"""
        # Split results by type and apply status filter
        places = [r for r in results if r['type'] == 'place' and r['status'] == 'approved']
        events = [r for r in results if r['type'] == 'event' and r['status'] == 'approved']
        
        # Apply quantity limits if specified
        quantity_limit = analysis['search_params'].get('quantity_limit')
        if quantity_limit is not None:
            places = places[:quantity_limit]
            events = events[:quantity_limit]
        
        # If exact quantity is required, ensure we only return that many results
        if analysis['search_params'].get('exact_quantity', False):
            total_needed = quantity_limit or 1
            if len(places) > total_needed:
                places = places[:total_needed]
            if len(events) > total_needed:
                events = events[:total_needed]
        
        # Prioritize exact address matches
        if analysis['search_params'].get('location_text'):
            places = self._prioritize_exact_matches(
                places, 
                analysis['search_params']['location_text']
            )
        
        return {
            'results': {
                'primary': places,
                'secondary': events,
                'total_found': len(places) + len(events)
            },
            'location_context': self._get_location_context(analysis),
            'guidance': analysis['guidance'] if analysis['needs_clarification'] else None,
            'metadata': {
                'total_results': len(results),
                'primary_count': len(places),
                'secondary_count': len(events),
                'search_terms_used': analysis['search_params']['search_terms']
            }
        }

    def _prioritize_exact_matches(self, results, location_text):
        """Prioritize exact location matches"""
        exact_matches = []
        partial_matches = []
        
        location_text = location_text.lower()
        location_parts = set(location_text.split())
        
        for result in results:
            address = result.get('address', {})
            street = address.get('street', '').lower()
            
            # Check for exact street match
            if street and location_text in street:
                exact_matches.append(result)
            # Check for partial matches
            elif street and any(part in street for part in location_parts):
                partial_matches.append(result)
        
        # Return exact matches first, then partial matches
        return exact_matches + partial_matches

    def _get_location_context(self, analysis):
        """Get location context for response"""
        return {
            'type': analysis['search_params']['location_type'],
            'specified_location': analysis['search_params'].get('location_text'),
            'radius_used': None
        }

    def _needs_geolocation(self, results):
        """Check if results need geolocation data"""
        return any('address' not in result for result in results)

    def analyze_query_with_gpt(self, query):
        """Enhanced query analysis with better location and context detection"""
        try:
            # Prepare a more specific prompt for location awareness
            location_aware_prompt = """
            Analyze the search query for a location-based search system. Extract:
            1. Main search terms
            2. Location information
            3. Any type specifications (restaurant, event, etc.)
            4. Time context if any
            5. Quantity requirements (e.g., 'only one', 'single', 'first')
            6. Location details (street names, neighborhoods, landmarks)
            7. Search terms and type specifications
            8. Any other constraints or preferences
            
            Return the analysis as a structured JSON with these fields explicitly marked.
            Format the response as a structured analysis.
            """
            
            response = self.client.chat.completions.create(
                model="gpt-3.5-turbo",
                messages=[
                    {"role": "system", "content": location_aware_prompt},
                    {"role": "user", "content": query}
                ],
                temperature=0.3,
                max_tokens=500
            )
            
            # Parse GPT response
            raw_analysis = json.loads(response.choices[0].message.content)
            
            # Extract location and search terms
            words = query.lower().split()
            location_phrases = self._extract_location_phrases(query)
            quantity_info = self._extract_quantity_info(query)
            
            enhanced_analysis = {
                'understood': True,
                'search_params': {
                    'search_terms': [w for w in words if w not in ['in', 'at', 'on', 'near'] and not any(w in phrase.lower() for phrase in location_phrases)],
                    'location_type': 'specified_location' if location_phrases else 'no_location',
                    'location_text': location_phrases[0] if location_phrases else None,
                    'radius_preference': None,
                    'quantity_limit': quantity_info.get('limit'),
                    'exact_quantity': quantity_info.get('exact'),
                    'type': raw_analysis.get('type'),
                    'time_context': raw_analysis.get('time_context'),
                    'attributes': []
                },
                'needs_clarification': False,
                'guidance': {
                    'understood_parts': [
                        f"Looking for {quantity_info.get('description', 'places')} in {location_phrases[0]}" if location_phrases else "General search"
                    ],
                    'missing_info': None,
                    'suggestions': [],
                    'examples': []
                },
                'error_type': None
            }
            
            logger.info(f"Enhanced query analysis: {enhanced_analysis}")
            return enhanced_analysis
            
        except Exception as e:
            logger.error(f"GPT analysis error: {str(e)}")
            return self.fallback_analysis(query)

    def _extract_quantity_info(self, query):
        """Extract quantity requirements from query"""
        query_lower = query.lower()
        
        # Define quantity patterns
        quantity_patterns = {
            'single': {
                'patterns': ['only one', 'single', 'just one', 'one place', '1 place'],
                'limit': 1,
                'exact': True,
                'description': 'a single place'
            },
            'first': {
                'patterns': ['first', 'top result'],
                'limit': 1,
                'exact': True,
                'description': 'the first matching place'
            },
            'few': {
                'patterns': ['few', 'some'],
                'limit': 3,
                'exact': False,
                'description': 'a few places'
            }
        }
        
        # Check for quantity patterns
        for qty_type, info in quantity_patterns.items():
            if any(pattern in query_lower for pattern in info['patterns']):
                return info
        
        return {'limit': None, 'exact': False, 'description': 'places'}
 
    def _extract_location_phrases(self, query):
        """Enhanced location phrase extraction"""
        query_lower = query.lower()
        locations = []
        
        # Extract complete address or street patterns
        address_patterns = [
            r'(?:in|at|on|near)\s+((?:\w+\s+){1,5}(?:street|st|avenue|ave|road|rd|boulevard|blvd|plaza|paseo|torre|carrer|calle)(?:\s+\w+)*)',
            r'(?:in|at|on|near)\s+((?:\w+\s+){1,3}(?:\d+(?:[a-zA-Z])?)?)',
            r'((?:\w+\s+){1,5}(?:street|st|avenue|ave|road|rd|boulevard|blvd|plaza|paseo|torre|carrer|calle)(?:\s+\w+)*)'
        ]
        
        for pattern in address_patterns:
            matches = re.finditer(pattern, query_lower, re.IGNORECASE)
            for match in matches:
                location_phrase = match.group(1).strip()
                if location_phrase:
                    locations.append(location_phrase)
        
        return list(set(locations))

    def fallback_analysis(self, query):
        """Fallback analysis when GPT fails"""
        return {
            'understood': True,
            'search_params': {
                'search_terms': [term for term in query.split() if term],
                'location_type': 'no_location',
                'location_text': None,
                'radius_preference': None,
                'type': None,
                'time_context': None,
                'attributes': []
            },
            'needs_clarification': False,
            'guidance': {
                'understood_parts': f"Basic search for: {query}",
                'missing_info': None,
                'suggestions': [],
                'examples': []
            },
            'error_type': None
        }

    def _filter_results(self, results, analysis):
        """Apply advanced filtering based on query analysis"""
        search_terms = [term.lower() for term in analysis['search_params']['search_terms']]
        
        filtered_results = []
        for item in results:
            score = self._calculate_relevance_score(item, search_terms)
            if score > 0:
                item['relevance_score'] = score
                filtered_results.append(item)
        
        # Sort by relevance score
        filtered_results.sort(key=lambda x: x['relevance_score'], reverse=True)
        return filtered_results

    async def make_async_api_call(self, url, params=None, headers=None):
        """Make async API call using httpx"""
        async with httpx.AsyncClient() as client:
            response = await client.get(
                url,
                params=params,
                headers=headers,
                timeout=10.0
            )
            return response.json()

    def get_auth_headers(self, request):
        """Generate headers for internal API calls"""
        return {
            'Authorization': f"Bearer {settings.INTERNAL_API_KEY}",
            'X-Request-ID': str(uuid.uuid4()),
            'X-Original-User': str(request.user.id) if request.user.is_authenticated else 'anonymous'
        }

    def handle_api_error(self, error):
        """Handle API connection errors"""
        return Response({
            'error': 'Search service temporarily unavailable',
            'code': 'SERVICE_UNAVAILABLE',
            'details': str(error)
        }, status=status.HTTP_503_SERVICE_UNAVAILABLE)

    def handle_timeout_error(self, error):
        """Handle API timeout errors"""
        return Response({
            'error': 'Search operation timed out',
            'code': 'TIMEOUT',
            'details': str(error)
        }, status=status.HTTP_504_GATEWAY_TIMEOUT)

    def _calculate_relevance_score(self, item, search_terms):
        """Calculate relevance score with proper type handling"""
        score = 0
        try:
            # Safely get item values
            def get_safe_value(obj, key):
                value = obj.get(key, '')
                if isinstance(value, (str, int, float)):
                    return str(value).lower()
                return ''
            
            # Get basic item details
            title = get_safe_value(item, 'title')
            description = get_safe_value(item, 'description')
            item_type = get_safe_value(item, 'type')
            status = get_safe_value(item, 'status')
            
            # Get location information
            city = None
            if isinstance(item.get('city'), str):
                city = item['city'].lower()
            elif isinstance(item.get('address'), dict):
                city = get_safe_value(item['address'], 'city')
            
            # Calculate scores
            if status == 'approved':
                score += 10
            
            # Title matching
            for term in search_terms:
                if term in title:
                    score += 5
                if term == title:
                    score += 10
            
            # Type matching
            if item_type and any(term in item_type for term in search_terms):
                score += 15
            
            # Location matching
            if city and any(term in city for term in search_terms):
                score += 8
            
            # Description matching
            if description:
                for term in search_terms:
                    if term in description:
                        score += 2
            
            # Category matching
            categories = item.get('categories', [])
            if isinstance(categories, list):
                for category in categories:
                    if isinstance(category, (str, dict)):
                        cat_text = category.get('title', category) if isinstance(category, dict) else category
                        if any(term in str(cat_text).lower() for term in search_terms):
                            score += 4
            
            # Tags matching
            tags = item.get('tags', [])
            if isinstance(tags, list):
                for tag in tags:
                    if isinstance(tag, (str, dict)):
                        tag_text = tag.get('title', tag) if isinstance(tag, dict) else tag
                        if any(term in str(tag_text).lower() for term in search_terms):
                            score += 3
            
            return score
            
        except Exception as e:
            logger.error(f"Error calculating relevance score: {str(e)}")
            return 0

    def execute_search(self, request, query, analysis):
        """Execute search with improved error handling and logging"""
        try:
            # Store analysis for use in other methods
            self.current_analysis = analysis
            
            # Build search parameters
            search_params = self._build_search_params(analysis)
            
            # Log search attempt
            logger.info(f"Executing search with params: {search_params}")
            
            try:
                # Execute search based on context
                if analysis['search_params']['location_type'] == 'user_location':
                    results = self._handle_location_based_search(request, search_params)
                else:
                    results = self._handle_basic_search(request, search_params)
                
                # Validate results structure
                if not isinstance(results, (list, dict)):
                    raise ValueError("Invalid results format received from search")
                
                # Convert to list if dict
                if isinstance(results, dict) and 'results' in results:
                    results = results['results']
                elif isinstance(results, dict):
                    results = [results]
                
                # Format and return results
                formatted_response = self._format_response(results, analysis)
                
                # Log success
                logger.info(f"Search completed successfully. Found {len(results)} results")
                
                return Response(formatted_response)
                
            except ValidationError as ve:
                logger.warning(f"Validation error during search: {str(ve)}")
                return Response({
                    'error': str(ve),
                    'code': 'VALIDATION_ERROR',
                    'details': ve.detail if hasattr(ve, 'detail') else None
                }, status=status.HTTP_400_BAD_REQUEST)
                
            except Exception as e:
                logger.error(f"Error during search execution: {str(e)}")
                return Response({
                    'error': 'Search operation failed',
                    'code': 'EXECUTION_ERROR',
                    'details': str(e)
                }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
                
        except Exception as e:
            logger.error(f"Unexpected error in execute_search: {str(e)}")
            return Response({
                'error': 'Unexpected error occurred',
                'code': 'UNEXPECTED_ERROR',
                'details': str(e)
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)


    def _prepare_search_params(self, params):
        """Prepare and clean search parameters"""
        clean_params = {}
        
        # Basic parameters
        clean_params['status'] = 'approved'
        clean_params['page_size'] = params.get('page_size', 10)
        
        # Search terms
        if 'search' in params:
            search_terms = [
                term for term in params['search'].split()
                if term.lower() not in ['find', 'show', 'get', 'give', 'only', 'one', 'place', 'in']
            ]
            if search_terms:
                clean_params['search'] = ' '.join(search_terms)
        
        return clean_params

    def _filter_by_address(self, results, address_text):
        """Filter results by address match"""
        address_text = address_text.lower()
        address_parts = set(address_text.split())
        
        exact_matches = []
        partial_matches = []
        
        for result in results:
            if not result.get('address'):
                continue
                
            result_address = result['address'].get('street', '').lower()
            if not result_address:
                continue
            
            # Check for exact match
            if address_text in result_address:
                exact_matches.append(result)
            # Check for partial match (all parts of the address are found)
            elif all(part in result_address for part in address_parts):
                partial_matches.append(result)
        
        # Return exact matches first, then partial matches
        filtered_results = exact_matches + partial_matches
        
        # If no matches found, return original results
        return filtered_results if filtered_results else results

    def _get_location_context(self, analysis):
        """Get enhanced location context for response"""
        location_text = analysis['search_params'].get('location_text')
        location_type = analysis['search_params'].get('location_type')
        
        return {
            'type': location_type or 'no_location',
            'specified_location': location_text,
            'radius_used': None,
            'address_context': {
                'is_street_address': bool(location_text and any(
                    word in location_text.lower() 
                    for word in ['street', 'st', 'avenue', 'ave', 'road', 'rd', 'boulevard', 'blvd', 'plaza', 'paseo', 'torre', 'carrer', 'calle']
                )),
                'full_text': location_text
            } if location_text else None
        }

    def _handle_basic_search(self, request, search_params):
        """Handle basic search with improved parameter handling"""
        try:
            sites_url = f"{settings.API_BASE_URL}/sites/"
            
            # Clean and prepare search parameters
            search_params = self._prepare_search_params(search_params)
            
            # Get location from analysis
            location_text = self.current_analysis['search_params'].get('location_text')
            
            # If location is a street address, use it in the search parameter instead of city
            if location_text and any(word in location_text.lower() for word in ['street', 'st', 'avenue', 'ave', 'road', 'rd', 'boulevard', 'blvd', 'plaza', 'paseo', 'torre', 'carrer', 'calle']):
                search_params['address'] = location_text
                # Remove city parameter if it exists
                search_params.pop('city', None)
            
            # Add search terms
            if self.current_analysis['search_params'].get('search_terms'):
                search_params['search'] = ' '.join(
                    term for term in self.current_analysis['search_params']['search_terms']
                    if term.lower() not in ['find', 'show', 'get', 'give', 'only', 'one', 'place', 'in']
                )
            
            # Default parameters
            search_params.update({
                'status': 'approved',
                'page_size': self.current_analysis['search_params'].get('quantity_limit', 10)
            })
            
            # Remove empty parameters
            search_params = {k: v for k, v in search_params.items() if v is not None and v != ''}
            
            logger.info(f"Making request to {sites_url} with params: {search_params}")
            
            response = requests.get(
                sites_url,
                params=search_params,
                headers=self._get_request_headers(request),
                timeout=10
            )
            
            response.raise_for_status()
            results = response.json()
            
            # Handle different response formats
            if isinstance(results, dict):
                results = results.get('results', [])
            
            # Filter results based on exact address match if searching by street
            if location_text and 'address' in search_params:
                results = self._filter_by_address(results, location_text)
            
            # Apply quantity limit if specified
            if self.current_analysis['search_params'].get('quantity_limit'):
                results = results[:self.current_analysis['search_params']['quantity_limit']]
            
            return results
            
        except requests.Timeout:
            logger.error("Search request timed out")
            raise ValidationError({
                "error": "Search timeout",
                "code": "TIMEOUT",
                "details": "The search operation took too long to complete"
            })
        except requests.RequestException as e:
            logger.error(f"Error in basic search: {str(e)}")
            raise ValidationError({
                "error": "Search failed",
                "code": "REQUEST_FAILED",
                "details": str(e)
            })
    
    def apply_location_filter(self, queryset, lat, lng, radius):
        """Apply location-based filtering with distance calculation"""
        try:
            user_location = Point(float(lng), float(lat), srid=4326)
            
            return queryset.annotate(
                distance=Distance('address__point', user_location)
            ).filter(
                address__point__distance_lte=(user_location, D(m=float(radius)))
            ).order_by('distance')
            
        except Exception as e:
            logger.error(f"Location filtering error: {e}")
            return queryset
 
    def format_results(self, queryset, include_distance=False):
        """Format results with distance information"""
        results = []
        for site in queryset:
            result = {
                'id': site.id,
                'title': site.title,
                'description': site.description[:200] if site.description else None,
                'type': site.type,
                'status': site.status,
                'visibility': site.visibility,
                'categories': [cat.title for cat in site.categories.all()],
                'tags': [tag.title for tag in site.tags.all()],
                'city': site.city.name if site.city else None,
                'is_top_10': site.is_top_10,
                'view_count': site.view_count,
                'created_at': site.created_at
            }
            
            # Add distance if available
            if include_distance and hasattr(site, 'distance'):
                result['distance'] = {
                    'meters': round(site.distance.m, 2),
                    'kilometers': round(site.distance.km, 2)
                }
            
            results.append(result)
        return results

    def post(self, request):
        try:
            query = request.data.get('query', '').strip()
            if not query:
                return Response({
                    'error': 'Query is required',
                    'code': 'QUERY_REQUIRED'
                }, status=status.HTTP_400_BAD_REQUEST)

            # Get query analysis from GPT
            analysis = self.analyze_query_with_gpt(query)
            
            # Execute search with analysis
            return self.execute_search(request, query, analysis)

        except Exception as e:
            logger.error(f"Search error: {str(e)}")
            return Response({
                'error': 'Search operation failed',
                'code': 'SEARCH_FAILED',
                'details': str(e)
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
 
    def generate_error_guidance(self, error_type=None):
        """Generate comprehensive error guidance"""
        generic_guidance = {
            'error': {
                'type': error_type or 'unknown',
                'message': 'We encountered an issue processing your search'
            },
            'guidance': {
                'understood_parts': [],
                'suggestions': [
                    'Try a simpler search query',
                    'Make sure your search is clear and specific',
                    'Check your internet connection'
                ],
                'examples': [
                    'restaurants in Barcelona',
                    'cafes near me',
                    'events this weekend'
                ]
            }
        }

        # Specific error type handling
        error_specific_guidance = {
            'location_error': {
                'message': 'Unable to process location-based search',
                'suggestions': [
                    'Enable location services in your browser',
                    'Try searching with a specific location instead',
                    'Update your location preferences'
                ],
                'examples': [
                    'restaurants in downtown',
                    'cafes in Barcelona',
                    'events in Central Park'
                ]
            },
            'query_too_complex': {
                'message': 'Your search query was too complex',
                'suggestions': [
                    'Break down your search into simpler terms',
                    'Remove unnecessary words',
                    'Focus on key search terms'
                ],
                'examples': [
                    'italian restaurants',
                    'coffee shops open now',
                    'weekend events'
                ]
            },
            'invalid_format': {
                'message': 'Search format not recognized',
                'suggestions': [
                    'Use simple, clear language',
                    'Specify what you\'re looking for',
                    'Include location if relevant'
                ],
                'examples': [
                    'pizza restaurants near me',
                    'art galleries in Madrid',
                    'live music tonight'
                ]
            },
            'no_results': {
                'message': 'No results found for your search',
                'suggestions': [
                    'Try different search terms',
                    'Expand your search area',
                    'Remove some filters'
                ],
                'examples': [
                    'restaurants any cuisine',
                    'entertainment venues',
                    'activities this week'
                ]
            }
        }

        if error_type in error_specific_guidance:
            generic_guidance['error']['message'] = error_specific_guidance[error_type]['message']
            generic_guidance['guidance']['suggestions'] = error_specific_guidance[error_type]['suggestions']
            generic_guidance['guidance']['examples'] = error_specific_guidance[error_type]['examples']

        return generic_guidance

    def generate_empty_query_guidance(self):
        """Generate guidance for empty queries"""
        return {
            'error': {
                'type': 'empty_query',
                'message': 'Please enter a search query'
            },
            'guidance': {
                'suggestions': [
                    'Try searching for places or events',
                    'You can search by location or category',
                    'Include specific details for better results'
                ],
                'examples': [
                    'restaurants near me',
                    'coffee shops in downtown',
                    'events this weekend',
                    'art galleries in Barcelona'
                ],
                'features': [
                    'Location-based search available',
                    'Category filtering',
                    'Time-based search for events'
                ]
            }
        }

    def generate_guidance_response(self, analysis):
        """Generate guidance based on query analysis"""
        return {
            'results': {
                'places': [],
                'events': []
            },
            'guidance': {
                'understood_parts': analysis['guidance']['understood_parts'],
                'missing_info': analysis['guidance']['missing_info'],
                'suggestions': [
                    *analysis['guidance']['suggestions'],
                    'Try using more specific terms',
                    'Include location information if relevant'
                ],
                'examples': [
                    *analysis['guidance']['examples'],
                    'restaurants near me open now',
                    'art exhibitions this weekend in city center'
                ],
                'features_available': [
                    'Location-based search',
                    'Category filtering',
                    'Time-based filtering',
                    'Rating-based sorting'
                ]
            },
            'error': {
                'type': analysis['error_type'],
                'needs_clarification': analysis['needs_clarification']
            }
        }

    def handle_location_error(self, analysis, user_location_available):
        """Handle location-related search errors"""
        guidance = {
            'error': {
                'type': 'location_error',
                'message': 'Unable to process location-based search'
            },
            'guidance': {
                'understood_parts': analysis['guidance']['understood_parts'],
                'current_status': {
                    'location_services': 'unavailable' if not user_location_available else 'available',
                    'location_type': analysis['search_params']['location_type']
                },
                'suggestions': [
                    'Enable location services for nearby search',
                    'Specify a location in your search',
                    'Try searching without location constraints'
                ],
                'alternatives': [
                    'Search by city name',
                    'Search by neighborhood',
                    'Search by landmark'
                ],
                'examples': [
                    'restaurants in downtown',
                    'cafes in Barcelona',
                    'events near Central Park'
                ]
            }
        }
        return guidance

    def log_search_metrics(self, query, analysis, results_count, error_type=None):
        """Log search metrics for monitoring and improvement"""
        try:
            metrics = {
                'timestamp': datetime.now().isoformat(),
                'query': query,
                'analysis': {
                    'understood': analysis['understood'],
                    'location_type': analysis['search_params']['location_type'],
                    'needs_clarification': analysis['needs_clarification']
                },
                'results_count': results_count,
                'error_type': error_type
            }
            logger.info(f"Search metrics: {json.dumps(metrics)}")
        except Exception as e:
            logger.error(f"Error logging search metrics: {e}")

 
@method_decorator(staff_member_required, name='dispatch')
class SiteViewLogView(TemplateView):
    template_name = 'admin/sites/analytics_changelist.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        from .analytics import SiteAnalytics
        
        # Get date range from request or default to 30 days
        date_range = int(self.request.GET.get('date_range', 30))
        analytics = SiteAnalytics(date_range=date_range)
        
        context.update({
            'date_range': date_range,
            'review_metrics': analytics.get_review_metrics(),
            'subscription_metrics': analytics.get_subscription_metrics(),
            'top_sites': analytics.get_top_viewed_sites(limit=10),
            'view_trends': analytics.get_view_trends(),
            # Add these for admin template
            'title': 'Site Analytics',
            'is_popup': False,
            'has_permission': True,
            'site_url': '/',
        })
        
        return context
