# local_secrets/users/admin.py
from django.contrib import admin
from django.contrib.auth.models import Group
from django.db.models import Count
from django.utils import timezone  
from django.contrib.admin import TabularInline
from django.contrib.auth.admin import UserAdmin
from django.utils.translation import gettext_lazy as _
from import_export.admin import ImportExportModelAdmin
from import_export.resources import ModelResource
from nested_admin import NestedModelAdmin, NestedTabularInline
from django.utils.html import format_html 
from django.utils.safestring import mark_safe
from .models import Ambassador, CustomUser, Notification, Tag, TranslatedTag, UserTags
from ..core.admin import admin_site
from ..core.utils.admin_actions import ExportCsvMixin
from .models import (
    SubscriptionPlan, VendorSubscription, SiteClaim,
    SubscriptionTransaction, PlanFeature, PlanFeatureValue, Vendor, VendorVerificationStatus,
    TranslatedBusinessCategory, TranslatedBusinessType, BusinessType, BusinessCategory 
)
from .forms import SiteClaimForm
from django.db import transaction
from django.core.exceptions import ValidationError
from django import forms
from django.utils.text import slugify

class NotificationAdmin(admin.ModelAdmin):
    list_display = ('title', 'created_at')
    search_fields = ('title',)

class CustomUserAdmin(UserAdmin, ExportCsvMixin):
    filter_horizontal = ('groups',)

class AmbassadorAdmin(CustomUserAdmin):
    filter_horizontal = ('cities', 'groups')
 
class TranslatedTagResource(ModelResource):
    class Meta:
        model = TranslatedTag
        fields = ('id', 'tag', 'language', 'title')

class TranslatedTagInLine(NestedTabularInline):
    """Inline to manage translations for each tag"""
    model = TranslatedTag
    extra = 0
    fields = ('language', 'title')
    autocomplete_fields = ('language',)

class TagResource(ModelResource):
    class Meta:
        model = Tag
        fields = ('id', 'title')

class TagAdmin(ImportExportModelAdmin, NestedModelAdmin):
    """Admin to manage Tags"""
    resource_class = TagResource

    def num_of_translations(self, obj):
        """Count the number of translations for a tag"""
        return obj.translations.count()

    num_of_translations.admin_order_field = 'num_of_translations'
    num_of_translations.short_description = 'Translations'

    def num_of_users(self, obj):
        """Count the number of users associated with a tag"""
        return obj.users.count()

    num_of_users.admin_order_field = 'num_of_users'
    num_of_users.short_description = 'Users'

    list_display = ('title', 'num_of_translations', 'num_of_users')
    search_fields = ('title', 'translations__title', 'translations__language__name')
    inlines = [TranslatedTagInLine]
    list_filter = ('translations__language',)
 
class TranslatedTagAdmin(ImportExportModelAdmin):
    """Admin to manage Translated Tags"""
    resource_class = TranslatedTagResource

    list_display = ('id', 'tag', 'language', 'title')
    search_fields = ('tag__title', 'title', 'language__name')
    list_filter = ('language', 'tag__title')
 
class TranslatedBusinessCategoryInline(NestedTabularInline):
    model = TranslatedBusinessCategory
    extra = 0
    fields = ('language', 'name')
 
class TranslatedBusinessTypeInline(NestedTabularInline):
    model = TranslatedBusinessType
    extra = 0
    fields = ('language', 'name')
 
class BusinessCategoryAdmin(admin.ModelAdmin):
    list_display = ['name', 'slug', 'order']
    search_fields = ['name', 'slug']
    prepopulated_fields = {'slug': ('name',)}
    inlines = [TranslatedBusinessCategoryInline]
 
class BusinessTypeAdmin(admin.ModelAdmin):
    list_display = ['name', 'category', 'slug', 'order']
    list_filter = ['category', 'service_level', 'target_audience', 'operating_season']
    search_fields = ['name', 'slug', 'category__name']
    prepopulated_fields = {'slug': ('name',)}
    inlines = [TranslatedBusinessTypeInline]

class VendorAdminForm(forms.ModelForm):
    class Meta:
        model = Vendor
        fields = '__all__'
        widgets = {
            'description': forms.Textarea(attrs={'rows': 4}),
            'short_description': forms.Textarea(attrs={'rows': 2}),
        }

    def clean_user(self):
        user = self.cleaned_data.get('user')
        if user:
            # Verify user exists
            if not CustomUser.objects.filter(id=user.id).exists():
                raise ValidationError(f"User {user.id} does not exist in database")
            # Check if user already has a vendor profile
            if hasattr(user, 'vendor_profile'):
                raise ValidationError(f"User {user.id} already has a vendor profile")
        return user

    def clean_slug(self):
        slug = self.cleaned_data.get('slug')
        if not slug:
            company_name = self.cleaned_data.get('company_name')
            if company_name:
                slug = slugify(company_name)
        return slug


class VendorAdmin(admin.ModelAdmin):
    list_display = [
        'company_name',
        'user',
        'verification_badge',
        'subscription_type',
        'subscription_status',
        'is_featured',
        'created_at',
        'preview_logo'
    ]   

    list_filter = [
        'verification_status',
        'subscription_type',
        'is_featured',
        'cities',
        'categories',
        'created_at'
    ]     

    readonly_fields = [
        'uuid',
        'created_at',
        'updated_at',
        'verified_at',
        'subscription_status',
        'preview_logo',
        'preview_header'
    ]    

    filter_horizontal = ['cities', 'categories', 'tags']
    prepopulated_fields = {'slug': ('company_name',)}    
    autocomplete_fields = ['business_type', 'secondary_business_types']
    search_fields = ['company_name', 'contact_email'] 
    
    fieldsets = (
        (None, {
            'fields': (
                'user',
                'uuid',
                'company_name',
                'slug',
                'business_type',
                'tax_id',
            )
        }),
        (_('Business Details'), {
            'fields': (
                'description',
                'short_description',
                'founded_year',
                'website',
                'categories',
                'tags',
            )
        }),
        (_('Verification & Subscription'), {
            'fields': (
                'verification_status',
                'verified_at',
                'subscription_type',
                'subscription_expires',
                'is_featured',
            )
        }),
        (_('Contact Information'), {
            'fields': (
                'contact_email',
                'contact_phone',
                'primary_address',
                'cities',
            )
        }),
        (_('Social Media'), {
            'fields': (
                'facebook_url',
                'instagram_url',
                'twitter_url',
            ),
            'classes': ('collapse',)
        }),
        (_('Media'), {
            'fields': (
                ('logo', 'preview_logo'),
                ('header_image', 'preview_header'),
            )
        }),
        (_('Timestamps'), {
            'fields': (
                'created_at',
                'updated_at',
            ),
            'classes': ('collapse',)
        }),
    )

    def get_queryset(self, request):
        return super().get_queryset(request).select_related(
            'business_type',
        ).prefetch_related('secondary_business_types')

    def verification_badge(self, obj):
        colors = {
            VendorVerificationStatus.VERIFIED: 'green',
            VendorVerificationStatus.PENDING: 'orange',
            VendorVerificationStatus.REJECTED: 'red'
        }
        return format_html(
            '<span style="color: {};">{}</span>',
            colors.get(obj.verification_status, 'black'),
            obj.get_verification_status_display()
        )
    verification_badge.short_description = _('Verification')
    
    def subscription_status(self, obj):
        if obj.get_active_subscription():
            if obj.subscription_type == 'FREE':
                return _('Free Plan')
            return format_html(
                '<span style="color: green;">Active until {}</span>',
                obj.subscription_expires.strftime('%Y-%m-%d')
            )
        return format_html(
            '<span style="color: red;">Expired</span>'
        )
    subscription_status.short_description = _('Subscription Status')

    def preview_logo(self, obj):
        if obj.logo:
            return format_html('<img src="{}" width="50" height="50" />', obj.logo.url)
        return format_html('No Logo')
    preview_logo.short_description = 'Logo Preview'

    def preview_header(self, obj):
        if obj.header_image:
            return format_html(
                '<img src="{}" style="max-height: 100px;"/>',
                obj.header_image.url
            )
        return _('No header image uploaded')
    preview_header.short_description = _('Header Preview')

    @transaction.atomic
    def save_model(self, request, obj, form, change):
        try:
            if not change:  # Creating new vendor
                if not obj.user_id:
                    raise ValidationError(_("User must be selected when creating a vendor"))
                
                user = CustomUser.objects.get(id=obj.user_id)
                
                if hasattr(user, 'vendor_profile') and user.vendor_profile:
                    raise ValidationError(_("Selected user already has a vendor profile"))
                
                # Set the user before saving
                obj.user = user
                
                # Save the main object first
                super().save_model(request, obj, form, change)
                
                # Assign vendor group
                vendor_group, _ = Group.objects.get_or_create(name='Vendor')
                user.groups.add(vendor_group)
                
                # Set verification date if status is verified
                if obj.verification_status == VendorVerificationStatus.VERIFIED:
                    obj.verified_at = timezone.now()
                    obj.save()
                
            else:  # Updating existing vendor
                # Update verification date if status changed to verified
                if obj.verification_status == VendorVerificationStatus.VERIFIED and not obj.verified_at:
                    obj.verified_at = timezone.now()
                super().save_model(request, obj, form, change)
        
        except ValidationError as ve:
            raise ve
        except Exception as e:
            raise ValidationError(f"Error creating/updating vendor: {str(e)}")

    def save_related(self, request, form, formsets, change):
        """
        Override save_related to ensure proper handling of M2M relationships
        """
        try:
            super().save_related(request, form, formsets, change)
        except Exception as e:
            raise ValidationError(f"Error saving related data: {str(e)}")

    def get_readonly_fields(self, request, obj=None):
        if obj:  # editing an existing object
            return self.readonly_fields + ['user']
        return self.readonly_fields

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == "user":
            kwargs["queryset"] = CustomUser.objects.filter(vendor_profile__isnull=True)
        return super().formfield_for_foreignkey(db_field, request, **kwargs)

    class Media:
        css = {
            'all': ('admin/css/vendor.css',)
        }
        
class VendorSubscriptionAdmin(admin.ModelAdmin):
    list_display = [
        'vendor', 'plan', 'status', 'payment_status',
        'start_date', 'end_date', 'sites_quota_used', 'events_quota_used'
    ]
    list_filter = ['status', 'payment_status', 'plan']
    search_fields = ['vendor__company_name', 'vendor__user__email']
    readonly_fields = ['sites_quota_used', 'events_quota_used']
    raw_id_fields = ['vendor']
    
    def get_queryset(self, request):
        return super().get_queryset(request).select_related('vendor', 'plan')
  
class SiteClaimAdmin(admin.ModelAdmin):
    form = SiteClaimForm
    list_display = ['site', 'vendor', 'status', 'created_at', 'processed_at', 'processed_by']
    list_filter = ['status', 'created_at', 'processed_at']
    search_fields = ['site__title', 'vendor__company_name', 'notes']
    raw_id_fields = ['site', 'vendor', 'processed_by']
    readonly_fields = ['created_at']
    
    fieldsets = (
        (None, {
            'fields': ('site', 'vendor', 'status', 'notes')
        }),
        (_('Processing Information'), {
            'fields': ('processed_by', 'processed_at', 'created_at')
        }),
        (_('Documents'), {
            'fields': ('proof_documents',)
        }),
    )

    def save_model(self, request, obj, form, change):
        if not change:  # New claim
            obj.processed_by = request.user
        elif 'status' in form.changed_data:  # Status changed
            obj.processed_by = request.user
            obj.processed_at = timezone.now()
            
            if obj.status == SiteClaim.APPROVED:
                obj.site.vendor = obj.vendor
                obj.site.save(update_fields=['vendor'])
                
        super().save_model(request, obj, form, change)
  
class PlanFeatureValueInline(admin.TabularInline):
    model = PlanFeatureValue
    extra = 1 
    fields = ['feature', 'boolean_value', 'numeric_value', 'text_value']
    raw_id_fields = ['feature']
    autocomplete_fields = ['feature']
    show_change_link = True
    min_num = 0
    max_num = 10
    can_delete = True
    verbose_name = "Plan Feature Value"
    verbose_name_plural = "Plan Feature Values"
 
class SubscriptionPlanAdmin(admin.ModelAdmin):
    list_display = ['name', 'type', 'price', 'billing_period', 'is_active']
    list_filter = ['is_active', 'type', 'billing_period']
    search_fields = ['name']
    readonly_fields = ['created_at', 'updated_at']
    fieldsets = (
        (None, {
            'fields': ('name', 'type', 'price', 'billing_period')
        }),
        ('Limits', {
            'fields': ('max_sites', 'max_events', 'duration_days')
        }),
        ('Status', {
            'fields': ('is_active', 'created_at', 'updated_at')
        })
    )
    inlines = [PlanFeatureValueInline]

class PlanFeatureAdmin(admin.ModelAdmin):
    list_display = ['name', 'feature_type', 'description']
    search_fields = ['name', 'description']
    list_filter = ['feature_type']
    fieldsets = (
        (None, {
            'fields': ('name', 'feature_type', 'description')
        }),
    )

    def get_queryset(self, request):
        return super().get_queryset(request).prefetch_related('planfeaturevalue_set')

class PlanFeatureValueAdmin(admin.ModelAdmin):
    list_display = ['plan', 'feature', 'get_value']
    list_filter = ['plan', 'feature']
    search_fields = ['plan__name', 'feature__name']

    def get_queryset(self, request):
        return super().get_queryset(request).select_related('plan', 'feature')

# Register all models
admin_site.register(SubscriptionPlan, SubscriptionPlanAdmin)
admin_site.register(VendorSubscription, VendorSubscriptionAdmin)
admin_site.register(SiteClaim, SiteClaimAdmin)
admin_site.register(PlanFeature, PlanFeatureAdmin)
admin_site.register(PlanFeatureValue, PlanFeatureValueAdmin)
admin_site.register(Vendor, VendorAdmin) 

admin_site.register(BusinessCategory, BusinessCategoryAdmin)
admin_site.register(BusinessType, BusinessTypeAdmin) 
 
admin_site.register(CustomUser, CustomUserAdmin)
admin_site.register(Ambassador, AmbassadorAdmin)
admin_site.register(Tag, TagAdmin)
admin_site.register(TranslatedTag, TranslatedTagAdmin)
admin_site.register(UserTags)
admin_site.register(Notification, NotificationAdmin)
admin_site.register(SubscriptionTransaction)
 