# local_secrets/users/cron.py

import logging
from datetime import datetime, timedelta
from django.utils import timezone
from django.conf import settings
from django.db import transaction
from django.db.models import Count, Avg, Sum, Q
from decimal import Decimal
from .models import (
VendorSubscription,
SiteClaim,
SubscriptionUsageReport,
Vendor
)
from .services.subscription_services import SubscriptionService
from .emails import VendorEmailNotifications
from local_secrets.payment.utility import (
process_payment,
get_payment_status,
send_payment_notification
)
from local_secrets.payment.models import VendorPayment
from local_secrets.payment.exceptions import PaymentError


logger = logging.getLogger(__name__)

def cleanup_pending_claims():
    """Clean up old pending site claims"""
    try:
        days_threshold = 30
        cutoff_date = timezone.now() - timedelta(days=days_threshold)
        old_claims = SiteClaim.objects.filter(
                status='PENDING',
                created_at__lt=cutoff_date
            )
            
        count = old_claims.count()
        old_claims.update(
                status='REJECTED',
                notes='Automatically rejected due to inactivity',
                processed_at=timezone.now()
            )
            
        logger.info(f'Cleaned up {count} old pending claims')
            
    except Exception as e:
        logger.error(f'Error in cleanup_pending_claims: {str(e)}')
 
def check_expiring_subscriptions():
    """Check and notify vendors about expiring subscriptions"""
    try:
        # Check for multiple thresholds (7, 3, 1 days)
        thresholds = [7, 3, 1]
        now = timezone.now()
        for days in thresholds:
            expiration_date = now.date() + timedelta(days=days)
            
            expiring_subscriptions = VendorSubscription.objects.filter(
                status='active',
                end_date__date=expiration_date,
                auto_renew=False
            ).exclude(
                audit_logs__action='expiry_notice',
                audit_logs__created_at__date=now.date()
            ).select_related('vendor', 'vendor__user', 'plan')
            
            for subscription in expiring_subscriptions:
                try:
                    with transaction.atomic():
                        # Check if sites need to be deactivated
                        if days == 1:
                            sites_to_deactivate = subscription.vendor.vendor_sites.filter(
                                is_active=True
                            )
                            sites_to_deactivate.update(
                                is_active=False,
                                deactivation_reason='subscription_expiring'
                            )
                            
                        VendorEmailNotifications.send_subscription_expiring_soon(
                            subscription, 
                            days_remaining=days,
                            sites_affected=sites_to_deactivate.count() if days == 1 else None
                        )
                        from local_secrets.sites.services.site_visibility import SiteVisibilityManager
                        SiteVisibilityManager.update_site_visibility(subscription)

                        
                        subscription.audit_logs.create(
                            action='expiry_notice',
                            details={
                                'days_remaining': days,
                                'sites_affected': sites_to_deactivate.count() if days == 1 else 0
                            }
                        )
                        
                        logger.info(f'Sent {days}-day expiration notice to {subscription.vendor.company_name}')
                        
                except Exception as e:
                    logger.error(f'Failed to send notice to {subscription.vendor.company_name}: {str(e)}')
                 
    except Exception as e:
        logger.error(f'Error in check_expiring_subscriptions: {str(e)}')
 

def process_subscription_renewals():
    """Process automatic subscription renewals"""
    try:
        # Get subscriptions due for renewal in the next 24 hours
        renewal_window_end = timezone.now() + timedelta(days=1)
        due_subscriptions = VendorSubscription.objects.filter(
            Q(status='active') & 
            Q(auto_renew=True) & 
            Q(end_date__lte=renewal_window_end) &
            ~Q(payment_status='pending')
        ).select_related('vendor', 'plan', 'last_payment', 'last_payment__payment_method')
        
        for subscription in due_subscriptions:
            try:
                with transaction.atomic():
                    # Create payment record
                    payment = VendorPayment.objects.create(
                        vendor=subscription.vendor,
                        amount=subscription.plan.price,
                        currency=subscription.plan.currency,
                        payment_method=subscription.last_payment.payment_method,
                        description=f"Auto-renewal for {subscription.plan.name}",
                        metadata={
                            'subscription_id': str(subscription.id),
                            'renewal_period': f"{subscription.plan.duration_days} days"
                        }
                    )
                    
                    subscription.payment_status = 'pending'
                    subscription.save()
                    
                    try:
                        # Process payment
                        process_payment(payment)
                        
                        if payment.status == 'processed':
                            # Update subscription
                            old_end_date = subscription.end_date
                            subscription.start_date = old_end_date
                            subscription.end_date = old_end_date + timedelta(days=subscription.plan.duration_days)
                            subscription.last_payment = payment
                            subscription.payment_status = 'completed'
                            subscription.save()
                            
                            # Reactivate sites if needed
                            subscription.vendor.vendor_sites.filter(
                                is_active=False,
                                deactivation_reason='subscription_expired'
                            ).update(
                                is_active=True,
                                deactivation_reason=None
                            )
                            
                            # Send success notification
                            send_payment_notification(payment, success=True)
                            logger.info(f'Successfully renewed subscription for {subscription.vendor.company_name}')
                            
                        else:
                            raise PaymentError(f"Payment failed with status: {payment.status}")
                            
                    except Exception as e:
                        subscription.payment_status = 'failed'
                        subscription.save()
                        
                        # Send failed renewal notification
                        VendorEmailNotifications.send_renewal_failed(subscription)
                        logger.error(f'Renewal failed for {subscription.vendor.company_name}: {str(e)}')
                        
            except Exception as e:
                logger.error(f'Failed to process renewal for {subscription.vendor.company_name}: {str(e)}')
                
    except Exception as e:
        logger.error(f'Error in process_subscription_renewals: {str(e)}')
 

def generate_subscription_reports():
    """Generate daily subscription usage reports"""
    try:
        yesterday = timezone.now().date() - timedelta(days=1)        
        active_subscriptions = VendorSubscription.objects.filter(
            status='active'
        ).exclude(
            usage_reports__report_date=yesterday
        ).select_related('vendor', 'plan')
        
        for subscription in active_subscriptions:
            try:
                with transaction.atomic():
                    # Get usage metrics
                    sites_count = subscription.vendor.vendor_sites.count()
                    events_count = subscription.vendor.vendor_sites.filter(type='event').count()
                    active_sites = subscription.vendor.vendor_sites.filter(is_active=True).count()
                    
                    # Calculate quotas
                    sites_quota_percentage = (
                        (sites_count / subscription.plan.max_sites * 100)
                        if subscription.plan.max_sites > 0 else 0
                    )
                    events_quota_percentage = (
                        (events_count / subscription.plan.max_events * 100)
                        if subscription.plan.max_events > 0 else 0
                    )
                    
                    # Create usage report
                    report = SubscriptionUsageReport.objects.create(
                        subscription=subscription,
                        report_date=yesterday,
                        sites_count=sites_count,
                        active_sites_count=active_sites,
                        sites_quota_percentage=min(sites_quota_percentage, 100),
                        events_count=events_count,
                        events_quota_percentage=min(events_quota_percentage, 100),
                        extra_metrics={
                            'inactive_sites': sites_count - active_sites,
                            'quota_remaining': {
                                'sites': max(0, subscription.plan.max_sites - sites_count),
                                'events': max(0, subscription.plan.max_events - events_count)
                            }
                        }
                    )
                    
                    # Check quotas and send warnings
                    if sites_quota_percentage >= 90 or events_quota_percentage >= 90:
                        VendorEmailNotifications.send_quota_warning(
                            subscription, 
                            report,
                            sites_remaining=max(0, subscription.plan.max_sites - sites_count),
                            events_remaining=max(0, subscription.plan.max_events - events_count)
                        )
                    
                    logger.info(f'Generated usage report for {subscription.vendor.company_name}')
                    
            except Exception as e:
                logger.error(f'Failed to generate report for {subscription.vendor.company_name}: {str(e)}')
                
    except Exception as e:
        logger.error(f'Error in generate_subscription_reports: {str(e)}')
