import logging
import time
from django.utils.translation import gettext as _
from django.template.loader import render_to_string
from django.conf import settings
from datetime import datetime
from utils.encrypt_decrypt import encrypt
from utils.email_translation import EmailTranslationManager
from local_secrets.core import constants as const
from django.core.mail import EmailMultiAlternatives
from email.mime.image import MIMEImage
from django.template.loader import render_to_string

logger = logging.getLogger(__name__)


class EmailNotificationService:
    """
      A service class responsible for sending email notifications to users.

      This class provides methods to send various types of emails, including:
      - Account verification emails with an encrypted email link.
      - Account verification success emails.
    """

    def __init__(self, request):
        self.request = request
        self.language = request.headers.get('language')
        self.web_url = settings.WEBSITE_URL
        self.FROM_EMAIL = f"Local Secrets {settings.EMAIL_HOST_USER}"
        self.translation = EmailTranslationManager(self.language)

    def _get_logo(self, msg: EmailMultiAlternatives):
        """
        Load Local secret logo.
        Set a unique Content-ID for the image so it can be referenced in the HTML template

        """
        with open('media/default_images/App_Local_secrets_logo_1.png', 'rb') as img:
            image = MIMEImage(img.read())
            image.add_header('Content-ID', '<logo_image_id>')
            image.add_header('Content-Disposition',
                             'inline', filename='logo.png')
            msg.attach(image)
        return msg

    def send_verification_email(self, user):
        """
        Sends an email verification link to the user when their account is created.

        This function generates a verification link and sends it to the user's registered email 
        to confirm their account. The user must click the link to activate their account.

        The email is encrypted using a lightweight encryption method to prevent tampering 
        and ensure security.

        The email content is dynamically generated based on the language preference provided.
        """
        try:
            recipient = user.email
            template_type = const.USER_EMAIL_VERIFICATION_TEMPLATE_ID

            # Encrypt the email
            encrypted_string = encrypt(recipient, key=16)
            verification_url = f"{self.web_url}verify-user-email?email={encrypted_string}&timestamp={int(time.time())}"

            # Language based content manipulation
            translated_content = self.translation.localize(
                template_type, context={"recipient": recipient})
            context = {
                "language": self.language,
                "content": translated_content,
                "verification_url": verification_url,
                "logo_cid": "logo_image_id"
            }
            html_content = render_to_string(
                'user/email_verify.html', context)

            msg = EmailMultiAlternatives(
                subject=translated_content.get("subject"),
                from_email=self.FROM_EMAIL,
                to=[recipient],
            )

            # Get local secret logo
            msg = self._get_logo(msg)
            msg.attach_alternative(html_content, "text/html")
            msg.send()

            logger.info("Email send successfully !!!")
        except Exception as e:
            logger.error(f"Error sending verification email: {str(e)}")

    def send_verification_success_email(self, user):
        """
        Sends a verification email to the user after account creation.

        The email content is dynamically generated based on the language preference provided.
        """
        try:
            template_type = const.USER_EMAIL_VERIFICATION_SUCCESS_TEMPLATE_ID

            # Language based content manipulation
            translated_content = self.translation.localize(template_type)

            context = {
                "language": self.language,
                "content": translated_content,
                "website_url": settings.WEBSITE_URL,
                "logo_cid": "logo_image_id"
            }
            html_content = render_to_string(
                'user/email_verified.html', context)

            msg = EmailMultiAlternatives(
                subject=translated_content.get("subject"),
                from_email=self.FROM_EMAIL,
                to=[user.email],
            )

            # Get local secret logo
            msg = self._get_logo(msg)
            msg.attach_alternative(html_content, "text/html")
            msg.send()

        except Exception as e:
            logger.error(f"Error sending verification success email: {str(e)}")


    def send_forgot_password_email(self, user, password):
        """
        Sends a Password recovery email.

        The email content is dynamically generated based on the language preference provided.
        """
        try:
            template_type = const.USER_FORGOT_PASSWORD_TEMPLATE_ID

            # Language based content manipulation
            translated_content = self.translation.localize(
                template_type, context={"password": password}
            )

            context = {
                "language": self.language,
                "content": translated_content,
                "logo_cid": "logo_image_id"
            }
            html_content = render_to_string(
                'user/pasword_recovery.html', context)

            msg = EmailMultiAlternatives(
                subject=translated_content.get("subject"),
                from_email=self.FROM_EMAIL,
                to=[user.email],
            )

            # Get local secret logo
            msg = self._get_logo(msg)
            msg.attach_alternative(html_content, "text/html")
            msg.send()

        except Exception as e:
            logger.error(f"Error sending password recovery email: {str(e)}")


 