# local_secrets/users/management/commands/initialize_business_types.py

from django.core.management.base import BaseCommand
from django.utils.text import slugify
from local_secrets.users.models import BusinessCategory, BusinessType

BUSINESS_CATEGORIES = {
    'food_beverage': {
        'name': 'Food & Beverage',
        'types': [
            {'name': 'Restaurants (Full-Service)', 'service_level': 'standard'},
            {'name': 'Cafes & Coffee Shops', 'service_level': 'standard'},
            {'name': 'Bars & Nightclubs', 'service_level': 'standard'},
            {'name': 'Fast Food & Quick Service', 'service_level': 'economy'},
            {'name': 'Food Trucks', 'service_level': 'economy'},
            {'name': 'Specialty Food Shops', 'service_level': 'premium'},
            {'name': 'Wineries & Breweries', 'service_level': 'premium'},
        ]
    },
    'accommodation': {
        'name': 'Accommodation',
        'types': [
            {'name': 'Hotels & Resorts', 'service_level': 'luxury'},
            {'name': 'Boutique Hotels', 'service_level': 'premium'},
            {'name': 'Bed & Breakfasts', 'service_level': 'standard'},
            {'name': 'Hostels', 'service_level': 'economy'},
            {'name': 'Vacation Rentals', 'service_level': 'premium'},
            {'name': 'Camping & Glamping', 'service_level': 'standard'},
            {'name': 'Guest Houses', 'service_level': 'standard'},
        ]
    },
    'tourism_travel': {
        'name': 'Tourism & Travel Services',
        'types': [
            {'name': 'Tour Operators', 'service_level': 'standard'},
            {'name': 'Travel Agencies', 'service_level': 'standard'},
            {'name': 'Tourist Information Centers', 'service_level': 'standard'},
            {'name': 'Transportation Services', 'service_level': 'standard'},
            {'name': 'Car Rental Services', 'service_level': 'premium'},
            {'name': 'Bike Rental Services', 'service_level': 'standard'},
            {'name': 'Local Guides & Experiences', 'service_level': 'premium'},
        ]
    },
    'cultural_entertainment': {
        'name': 'Cultural & Entertainment',
        'types': [
            {'name': 'Museums', 'service_level': 'standard'},
            {'name': 'Art Galleries', 'service_level': 'premium'},
            {'name': 'Theaters & Performance Venues', 'service_level': 'premium'},
            {'name': 'Cinema & Movie Theaters', 'service_level': 'standard'},
            {'name': 'Cultural Centers', 'service_level': 'standard'},
            {'name': 'Historical Sites', 'service_level': 'standard'},
            {'name': 'Live Music Venues', 'service_level': 'premium'},
        ]
    },
    'sports_recreation': {
        'name': 'Sports & Recreation',
        'types': [
            {'name': 'Sports Facilities', 'service_level': 'standard'},
            {'name': 'Adventure Sports', 'service_level': 'premium'},
            {'name': 'Fitness Centers', 'service_level': 'standard'},
            {'name': 'Golf Courses', 'service_level': 'luxury'},
            {'name': 'Water Sports', 'service_level': 'premium'},
            {'name': 'Ski Facilities', 'service_level': 'premium'},
            {'name': 'Recreation Centers', 'service_level': 'standard'},
        ]
    },
    'wellness_health': {
        'name': 'Wellness & Health',
        'types': [
            {'name': 'Spas & Wellness Centers', 'service_level': 'luxury'},
            {'name': 'Massage Services', 'service_level': 'premium'},
            {'name': 'Yoga Studios', 'service_level': 'standard'},
            {'name': 'Health Retreats', 'service_level': 'luxury'},
            {'name': 'Beauty Services', 'service_level': 'premium'},
            {'name': 'Alternative Medicine', 'service_level': 'premium'},
            {'name': 'Fitness Services', 'service_level': 'standard'},
        ]
    },
    'educational': {
        'name': 'Educational & Workshops',
        'types': [
            {'name': 'Cooking Schools', 'service_level': 'premium'},
            {'name': 'Language Schools', 'service_level': 'standard'},
            {'name': 'Art & Craft Workshops', 'service_level': 'standard'},
            {'name': 'Dance Studios', 'service_level': 'standard'},
            {'name': 'Wine Tasting Classes', 'service_level': 'premium'},
            {'name': 'Cultural Workshops', 'service_level': 'standard'},
            {'name': 'Photography Tours', 'service_level': 'premium'},
        ]
    },
    'events_venues': {
        'name': 'Events & Venues',
        'types': [
            {'name': 'Event Spaces', 'service_level': 'premium'},
            {'name': 'Wedding Venues', 'service_level': 'luxury'},
            {'name': 'Conference Centers', 'service_level': 'premium'},
            {'name': 'Exhibition Halls', 'service_level': 'standard'},
            {'name': 'Festival Organizers', 'service_level': 'premium'},
            {'name': 'Party Planning Services', 'service_level': 'premium'},
            {'name': 'Corporate Event Services', 'service_level': 'premium'},
        ]
    },
    'specialty_experiences': {
        'name': 'Specialty Experiences',
        'types': [
            {'name': 'Food Tours', 'service_level': 'premium'},
            {'name': 'Heritage Tours', 'service_level': 'premium'},
            {'name': 'Photography Tours', 'service_level': 'premium'},
            {'name': 'Adventure Experiences', 'service_level': 'premium'},
            {'name': 'Local Crafts Workshops', 'service_level': 'standard'},
            {'name': 'Cultural Immersion', 'service_level': 'premium'},
            {'name': 'Historical Reenactments', 'service_level': 'premium'},
        ]
    },
}

class Command(BaseCommand):
    help = 'Initialize business categories and types'

    def handle(self, *args, **kwargs):
        for category_slug, category_data in BUSINESS_CATEGORIES.items():
            category, created = BusinessCategory.objects.get_or_create(
                slug=category_slug,
                defaults={
                    'name': category_data['name'],
                    'order': list(BUSINESS_CATEGORIES.keys()).index(category_slug)
                }
            )
            
            if created:
                self.stdout.write(
                    self.style.SUCCESS(f'Created category: {category.name}')
                )
            
            for type_data in category_data['types']:
                type_slug = slugify(type_data['name'])
                business_type, type_created = BusinessType.objects.get_or_create(
                    slug=type_slug,
                    category=category,
                    defaults={
                        'name': type_data['name'],
                        'service_level': type_data['service_level'],
                        'order': category_data['types'].index(type_data)
                    }
                )
                
                if type_created:
                    self.stdout.write(
                        self.style.SUCCESS(f'Created business type: {business_type.name}')
                    )

# python manage.py initialize_business_types
# python manage.py initialize_business_types --noinput
# python manage.py initialize_business_types --verbosity 2