# local_secrets/users/middleware.py

from django.utils import timezone
import logging
from local_secrets.users.exceptions import VendorRegistrationError
from local_secrets.users.services.subscription_services import SubscriptionService
from local_secrets.users.tasks import notify_admins 
logger = logging.getLevelName(__name__)


class SubscriptionMiddleware:
    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        if hasattr(request, 'user') and request.user.is_authenticated:
            if hasattr(request.user, 'vendor_profile'):
                service = SubscriptionService(request.user.vendor_profile)
                active_sub = service.active_subscription
                
                if active_sub and active_sub.is_active():
                    # Update usage periodically (e.g., once per day)
                    last_report = active_sub.usage_reports.first()
                    if not last_report or last_report.report_date < timezone.now().date():
                        service.update_usage()

        response = self.get_response(request)
        return response

class VendorErrorHandlerMiddleware:
    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        response = self.get_response(request)
        return response

    def process_exception(self, request, exception):
        if isinstance(exception, VendorRegistrationError):
            # Log the error
            logger.error(f"Vendor registration error: {str(exception)}")
            
            # Notify admins if needed
            notify_admins.delay(
                error_type='vendor_registration',
                error_details=str(exception),
                user_email=request.data.get('email')
            )