# local_secrets/users/models.py
from datetime import timezone
from typing import Any, Dict, Optional
import uuid
import threading
from django.contrib.auth import get_user_model
from django.contrib.auth.models import AbstractUser, Group
from django.core.mail import send_mail
from django.db import models
from django.utils.translation import gettext_lazy as _
from easy_thumbnails.fields import ThumbnailerImageField
from local_secrets.cities.models import Address
from local_secrets.cities.models import City
from local_secrets.languages.models import Language
from local_secrets.messaging.providers import FCMProvider
from local_secrets.sites.choices import SiteSubscriptionStatus
from local_secrets.users.managers import CustomUserManager, NotificationManager, TagManager
from django.conf import settings 
from django.utils.timezone import now
from django.utils.text import slugify
from django.db import transaction
from django.core.exceptions import ValidationError
from .choices import VendorVerificationStatus, VendorSubscriptionType
from django.db.models import Q, Count, Avg
from django.core.validators import MinValueValidator, MaxValueValidator
from local_secrets.users.helpers.email_helper import EmailNotificationService

class CustomUser(AbstractUser):
    tags = models.ManyToManyField('Tag', through='UserTags', verbose_name=_('Tags'))
    phone_prefix = models.CharField(max_length=10, verbose_name=_('Prefix'))
    phone = models.CharField(max_length=18, verbose_name=_('Phone'))
    profile_picture = ThumbnailerImageField(upload_to='pfp', verbose_name=_('Profile Picture'), null=True, blank=True)
    device_id = models.CharField(max_length=512, verbose_name=_('Device ID'), null=True, blank=True)
    language = models.ForeignKey(Language, on_delete=models.SET_NULL, null=True, blank=True)

    objects = CustomUserManager()

    def update_tags(self, tags):
        self.tags.set(tags)

    def update_pfp(self, file):
        if self.profile_picture:
            # Delete old file if exists
            try:
                self.profile_picture.delete()
            except:
                pass
        self.profile_picture = file[0]
        self.save()

    def send_notification(self, notification):
        return FCMProvider.send_notification_to_device(
            title=notification.title, body=notification.body, device_id=self.device_id
        )

    def restore_password(self, request=None):
        new_password = f'{uuid.uuid4().hex[:8].upper()}$'

        # Send password reset email
        notify = EmailNotificationService(request)
        thread = threading.Thread(
            target=notify.send_forgot_password_email,
            args=(self, new_password)
        )
        thread.start()
            
        self.set_password(new_password)
        self.save()

    def assign_vendor_role(self, **vendor_data):
        """
        Assign vendor role to user and create vendor profile
        """
        from .models import Vendor  # Import here to avoid circular imports
        from django.utils.timezone import now
        from django.utils.text import slugify
        from django.db import transaction
        from django.core.exceptions import ValidationError
        
        print(f"Assigning vendor role to user {self.id}")  # Debug print
        
        # Verify user exists and doesn't have a vendor profile
        if not CustomUser.objects.filter(id=self.id).exists():
            raise ValidationError(f"User {self.id} does not exist")
        
        if hasattr(self, 'vendor_profile'):
            raise ValidationError(f"User {self.id} already has a vendor profile")
        
        vendor_group, _ = Group.objects.get_or_create(name='Vendor')
        self.groups.add(vendor_group)
        
        company_name = vendor_data.get('company_name', f"{self.get_full_name()}'s Company")
        
        try:
            last_vendor = Vendor.objects.latest('id')
            next_id = last_vendor.id + 1
        except Vendor.DoesNotExist:
            next_id = 1
            
        vendor = Vendor(
            id=next_id,
            user=self,
            company_name=company_name,
            slug=vendor_data.get('slug', slugify(company_name)),
            contact_email=vendor_data.get('contact_email', self.email),
            business_type=vendor_data.get('business_type', 'Other'),
        )
        print(f"Created vendor {vendor.id} for user {self.id}")   
        vendor.save()  
        print(f"Saved vendor {vendor.id} for user {self.id}")   
     
        return vendor 

    def assign_ambassador_role(self, **ambassador_data):
        """
        Assign ambassador role to user and create ambassador profile
        """
        from .models import Ambassador 
        
        ambassador_group, _ = Group.objects.get_or_create(name='Ambassador')
        self.groups.add(ambassador_group)
        
        if not Ambassador.objects.filter(id=self.id).exists():
            ambassador = Ambassador.objects.create(
                user_ptr=self,
                **ambassador_data
            )
            return ambassador
        return Ambassador.objects.get(id=self.id)

    @property
    def is_vendor(self):
        """Check if user has a vendor profile"""
        return hasattr(self, 'vendor_profile')
    
    def get_vendor_profile(self):
        """Safely get vendor profile"""
        if self.is_vendor:
            return self.vendor_profile
        return None
        
    def save(self, *args, **kwargs):
        return super(CustomUser, self).save(*args, **kwargs)
 
class Tag(models.Model):
    title = models.CharField(max_length=120, verbose_name=_('Tag')) 
    objects = TagManager.as_manager() 
    class Meta:
        verbose_name = _('Tag')
        verbose_name_plural = _('Tags') 
    def __str__(self):
        return self.title

    def display_text(self, field, language='en'):
        try:
            return getattr(self.translations.get(language__code=language), field)
        except BaseException:
            return getattr(self, field)
 
class TranslatedTag(models.Model):
    tag = models.ForeignKey(Tag, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    title = models.CharField(max_length=120, verbose_name=_('Translated Tag'))
 
class UserTags(models.Model):
    user = models.ForeignKey(get_user_model(), on_delete=models.CASCADE, verbose_name=_('User'))
    tag = models.ForeignKey(Tag, on_delete=models.CASCADE, verbose_name=_('Tag'), related_name='users')

    class Meta:
        verbose_name = _('User Preference')
        verbose_name_plural = _('Users Preferences')

    def __str__(self):
        return f'{self.user.username} - {self.tag.title}'
 
class Notification(models.Model):
    from local_secrets.sites.models import Site

    created_at = models.DateTimeField(auto_now=True, verbose_name=_('Created at'))
    title = models.CharField(max_length=150, verbose_name=_('Title'))
    body = models.TextField(verbose_name=_('Body'))
    site = models.ForeignKey(Site, null=True, on_delete=models.SET_NULL, verbose_name=_('Site'))
    link = models.CharField(max_length=500, null=True, blank=True)

    objects = NotificationManager.as_manager()

    class Meta:
        verbose_name = _('Notification')
        verbose_name_plural = _('Notifications')

    # Mark notification as seen for user
    def mark_as_seen(self, user):
        # Get or create notification relation
        user_notification, created = UserNotification.objects.get_or_create(user=user, notification=self)
        #  Update State to seen/True
        user_notification.has_been_seen = True
        #  Save relation
        user_notification.save()

    def save(self, *args, **kwargs):
        users = CustomUser.objects.all()
        notification = super(Notification, self).save(*args, **kwargs)
        notification_list = []
        for user in users:
            if UserNotification.objects.filter(user=user, notification=self).exists():
                continue
            notification_list.append(UserNotification(user=user, notification=self))
            user.send_notification(self)
        UserNotification.objects.bulk_create(notification_list)
        return notification
 
class UserNotification(models.Model):
    user = models.ForeignKey(CustomUser, on_delete=models.CASCADE, related_name='notifications', verbose_name=_('User'))
    notification = models.ForeignKey(
        Notification, on_delete=models.CASCADE, related_name='users', verbose_name=_('Notification')
    )
    has_been_seen = models.BooleanField(default=False, verbose_name=_('Has been seen'))
    class Meta:
        verbose_name = _('Users Notification')
        verbose_name_plural = _('Users Notifications')


class UserNotificationPreference(models.Model):
    user = models.ForeignKey(CustomUser, on_delete=models.CASCADE, related_name='notification_preference', verbose_name=_('User'))
    is_promotion = models.BooleanField(default=False, verbose_name=_('Promotions and News'))
    is_recommendation = models.BooleanField(default=False, verbose_name=_('Recommend to a Friend'))
    is_weekly_discovery = models.BooleanField(default=False, verbose_name=_('Weekly discovery'))
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        verbose_name = _('Users Notification Preference')
        verbose_name_plural = _('Users Notification Preferences')
 
class Ambassador(CustomUser):
    cities = models.ManyToManyField(City)

    class Meta:
        verbose_name = _('Ambassador')
        verbose_name_plural = _('Ambassadors')

    def save(self, *args, **kwargs):
        created = False if self.id else True
        super(Ambassador, self).save(*args, **kwargs)
        if created:
            self.is_staff = True
            self.groups.add(Group.objects.get(id=2))
        res = super(Ambassador, self).save(*args, **kwargs)
        return res
 
class GroupDescription(models.Model):
    description = models.TextField(verbose_name=_('Description'))

    group = models.OneToOneField(Group, on_delete=models.CASCADE)

    def display_text(self, field, language='en'):
        try:
            return getattr(self.translations.get(language__code=language), field)
        except BaseException:
            if field == 'name':
                return self.group.name
            return getattr(self, field)
 
class TranslatedGroupDescription(models.Model):
    group_description = models.ForeignKey(GroupDescription, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    description = models.TextField(verbose_name=_('Translated Description'))
    name = models.CharField(max_length=150, verbose_name=_('Translated Name'))

# Vendor/Location/Business/SiteClaim/Subscription - Implementations

LOCATION_MODE_CHOICES = [
    ('current', 'Use Current Location'),
    ('manual', 'Set Specific Location'),
]

class LocationPreference(models.Model):
    user = models.OneToOneField(settings.AUTH_USER_MODEL,on_delete=models.CASCADE,related_name='location_preferences')
    geolocation_enabled = models.BooleanField(help_text=_("Enable/disable location services"))
    default_latitude = models.FloatField(null=True, blank=True, validators=[MinValueValidator(-90), MaxValueValidator(90)])
    default_longitude = models.FloatField(null=True,blank=True, validators=[MinValueValidator(-180),MaxValueValidator(180)])
    default_radius = models.FloatField(default=5000, validators=[MinValueValidator(100), MaxValueValidator(50000)], help_text=_("Default search radius in meters"))
    last_known_latitude = models.FloatField(null=True, blank=True, validators=[MinValueValidator(-90), MaxValueValidator(90)])
    last_known_longitude = models.FloatField(null=True, blank=True, validators=[MinValueValidator(-180), MaxValueValidator(180)])
    updated_at = models.DateTimeField(auto_now=True)
    location_mode = models.CharField(
        max_length=10,
        choices=LOCATION_MODE_CHOICES,
        null=True,
        blank=True,
        default=None,
        help_text="Choose how to set your location: 'current' or 'manual'."
    )

    class Meta:
        verbose_name = _("Location Preference")
        verbose_name_plural = _("Location Preferences")
        ordering = ['-updated_at']

    def __str__(self) -> str:
        return f"Location preferences for {self.user.username}"

    def get_current_location(self) -> Optional[Dict[str, Any]]:
        """
        Get user's current location (last known or default).
        
        Returns:
            dict: Location data or None if no location is set
        """
        if self.last_known_latitude and self.last_known_longitude:
            return {
                'latitude': self.last_known_latitude,
                'longitude': self.last_known_longitude,
                'is_current': True,
                'timestamp': self.updated_at
            }
        if self.default_latitude and self.default_longitude:
            return {
                'latitude': self.default_latitude,
                'longitude': self.default_longitude,
                'is_current': False,
                'timestamp': self.updated_at
            }
        return None

    def update_current_location(self, latitude: float, longitude: float) -> None:
        """
        Update user's current location.
        
        Args:
            latitude (float): New latitude
            longitude (float): New longitude
            
        Raises:
            ValidationError: If coordinates are invalid
        """
        if not (-90 <= latitude <= 90) or not (-180 <= longitude <= 180):
            raise ValidationError(_("Invalid coordinates provided"))
        
        self.last_known_latitude = latitude
        self.last_known_longitude = longitude
        self.geolocation_enabled = True
        self.location_mode = "current"
        self.save(update_fields=['last_known_latitude', 'last_known_longitude', 'geolocation_enabled', 'updated_at', 'location_mode'])

    def get_search_radius(self) -> float:
        """Get user's search radius in meters."""
        return self.default_radius

class VendorManager(models.Manager):
    """
    Custom manager for Vendor model providing additional query methods.
    """
    def get_verified(self):
        """Get only verified vendors"""
        return self.filter(verified=True)
        
    def get_by_category(self, category_id):
        """Get vendors by service category"""
        return self.filter(categories__id=category_id)
    
    def get_by_city(self, city_id):
        """Get vendors operating in a specific city"""
        return self.filter(cities__id=city_id)
    
    def search(self, query):
        """Search vendors by name, description, or tags"""
        return self.filter(
            Q(company_name__icontains=query) | 
            Q(description__icontains=query) |
            Q(tags__name__icontains=query)
        ).distinct()
    
    def top_rated(self, limit=10):
        """Get top rated vendors based on reviews"""
        return self.annotate(
            avg_rating=Avg('reviews__rating')
        ).filter(
            avg_rating__isnull=False
        ).order_by('-avg_rating')[:limit]
    
    def most_popular(self, limit=10):
        """Get most popular vendors based on site favorites"""
        return self.annotate(
            site_favorites=Count('vendor_sites__users')
        ).order_by('-site_favorites')[:limit]

class Vendor(models.Model):
    """
    Represents service providers or businesses that offer products/services 
    on the Local Secrets platform.
    """
    user = models.OneToOneField(
        #CustomUser, 
        settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE, 
        related_name='vendor_profile',
        verbose_name=_('User Account')
    )
    uuid = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    company_name = models.CharField(max_length=255, verbose_name=_('Company Name'))
    slug = models.SlugField(max_length=255, unique=True, verbose_name=_('URL Slug'))
    business_type = models.ForeignKey(
        'BusinessType',
        on_delete=models.PROTECT,
        related_name='vendors',
        verbose_name=_('Business Type')
    )
    secondary_business_types = models.ManyToManyField(
        'BusinessType',
        related_name='secondary_vendors',
        verbose_name=_('Additional Business Types'),
        blank=True
    )
    tax_id = models.CharField(max_length=50, blank=True, null=True, verbose_name=_('Tax ID'))
    website = models.URLField(blank=True, null=True, verbose_name=_('Website'))
    description = models.TextField(blank=True, verbose_name=_('Business Description'))
    short_description = models.CharField(max_length=280, blank=True, verbose_name=_('Short Description'))
    founded_year = models.PositiveIntegerField(blank=True, null=True, verbose_name=_('Year Founded'))
    
    # Verification details
    verification_status = models.CharField(
        max_length=20,
        choices=VendorVerificationStatus.choices,
        default=VendorVerificationStatus.PENDING,
        verbose_name=_('Verification Status')
    )
    verified_at = models.DateTimeField(null=True, blank=True, verbose_name=_('Verification Date'))
    
    # Subscription details
    subscription_type = models.CharField(
        max_length=20,
        choices=VendorSubscriptionType.choices,
        default=VendorSubscriptionType.FREE,
        verbose_name=_('Subscription Type')
    )
    subscription_expires = models.DateTimeField(null=True, blank=True, verbose_name=_('Subscription Expiry Date'))
    
    # Contact information
    contact_email = models.EmailField(verbose_name=_('Contact Email'))
    contact_phone = models.CharField(max_length=50, blank=True, verbose_name=_('Contact Phone'))
    address = models.CharField(max_length=255, blank=True, verbose_name=_('Address'))
    
    # Social media
    facebook_url = models.URLField(blank=True, null=True, verbose_name=_('Facebook URL'))
    instagram_url = models.URLField(blank=True, null=True, verbose_name=_('Instagram URL'))
    twitter_url = models.URLField(blank=True, null=True, verbose_name=_('Twitter URL'))
    
    # Logo and header image
    logo = models.ImageField(upload_to='vendor_logos', blank=True, null=True, verbose_name=_('Logo'))
    header_image = models.ImageField(upload_to='vendor_headers', blank=True, null=True, verbose_name=_('Header Image'))
    
    # Operating areas and categories        
    cities = models.ManyToManyField(
        'cities.City',
        related_name='vendors',
        verbose_name=_('Operating Cities'),
        blank=True
    )
    primary_address = models.ForeignKey(
        Address, 
        on_delete=models.SET_NULL, 
        null=True, 
        blank=True, 
        related_name='primary_for_vendors',
        verbose_name=_('Primary Address')
    )
    
    # Service categories (from sites app)
    categories = models.ManyToManyField(
        'sites.Category',
        related_name='vendors',
        verbose_name=_('Categories'),
        blank=True  # Allow empty
    )
  
    tags = models.ManyToManyField(
        'users.Tag',
        related_name='vendors',
        verbose_name=_('Tags'),
        blank=True
    )

    created_at = models.DateTimeField(auto_now_add=True, verbose_name=_('Created At'))
    updated_at = models.DateTimeField(auto_now=True, verbose_name=_('Updated At'))

    is_featured = models.BooleanField(default=False, verbose_name=_('Featured Vendor'))
    objects = VendorManager()
    
    class Meta:
        verbose_name = _('Vendor')
        verbose_name_plural = _('Vendors')
        ordering = ['-created_at']
        indexes = [
            models.Index(fields=['company_name']),
            models.Index(fields=['verification_status']),
            models.Index(fields=['is_featured']),
        ]

    def __str__(self):
        return self.company_name
    
    def save(self, *args, **kwargs):
        from django.utils.text import slugify
        from django.utils.timezone import now
        if not self.pk:    
            try:
                last_vendor = Vendor.objects.latest('id')
                self.id = last_vendor.id + 1
            except Vendor.DoesNotExist:
                self.id = 1
                
        if self.verification_status == VendorVerificationStatus.VERIFIED and not self.verified_at:
            self.verified_at = now()
        super().save(*args, **kwargs) 
 
        is_new = self.pk is None  # Check if this is a new instance
        
        if not self.slug:
            self.slug = self._generate_unique_slug()
        super().save(*args, **kwargs)

        if not self.slug:
            self.slug = slugify(self.company_name)
            base_slug = self.slug
            counter = 1
            while Vendor.objects.filter(slug=self.slug).exclude(id=self.id).exists():
                self.slug = f"{base_slug}-{counter}"
                counter += 1
        
        if self.verification_status == VendorVerificationStatus.VERIFIED and not self.verified_at:
            self.verified_at = timezone.now()
        
        super().save(*args, **kwargs)
        
        if is_new:  # Use is_new instead of created
            # Auto-assign the user to the vendor group
            vendor_group, _ = Group.objects.get_or_create(name='Vendor')
            self.user.groups.add(vendor_group)
    
    def display_text(self, field, language='en'):
        try:
            return getattr(self.translations.get(language__code=language), field)
        except:
            return getattr(self, field)
    
    def register_site(self, site_data):
        """Register a new site owned by this vendor"""
        from local_secrets.sites.models import Site
        
        site = Site(
            title=site_data.get('title', ''),
            description=site_data.get('description', ''),
            type=site_data.get('type', 'place'),
            created_by=self.user,
            vendor=self,
            # Other fields from site_data
        )
        site.save()
        
        # Add categories
        if 'categories' in site_data and site_data['categories']:
            site.categories.set(site_data['categories'])
        
        # Add address
        if 'address_data' in site_data and site_data['address_data']:
            # Implement address creation based on how addresses are handled
            from local_secrets.cities.models import Address
            address = Address.objects.create(**site_data['address_data'])
            site.address = address
            site.save(update_fields=['address'])
            
        return site
    
    def get_managed_sites(self):
        """Get all sites managed by this vendor"""
        return self.vendor_sites.all()
    
    def get_active_subscription(self):
        """Check if vendor has an active subscription"""
        if not self.subscription_expires:
            return self.subscription_type == VendorSubscriptionType.FREE
        return self.subscription_expires > now()
    
    def get_business_analytics(self, period='month'):
        """Get business analytics based on sites activity"""
        # Implementation will depend on the analytics tracking system
        sites = self.vendor_sites.all()
        
        # Example analytics calculation
        total_views = sum(site.view_count for site in sites if hasattr(site, 'view_count'))
        total_favorites = sum(site.users.count() for site in sites)
        
        return {
            'total_views': total_views,
            'total_favorites': total_favorites,
            'sites_count': sites.count(),
            # Additional metrics could be added here
        }
    
    def is_verified(self):
        """Check if vendor is verified"""
        return self.verification_status == VendorVerificationStatus.VERIFIED

    def can_manage_events(self):
        """Check if vendor can manage events"""
        return self.subscription_type in ['basic', 'premium', 'enterprise']

    def create_events(self, event_data):
        """Create events for this vendor"""
        from local_secrets.events.models import Event
        
        event = Event(
            title=event_data.get('title', ''),
            description=event_data.get('description', ''),
            start_time=event_data.get('start_time', None),
            end_time=event_data.get('end_time', None),
            vendor=self,
            # Other fields from event_data
        )
        event.save()
        
        # Add categories
        if 'categories' in event_data and event_data['categories']:
            event.categories.set(event_data['categories'])
        
        return event
 
    def _generate_unique_slug(self):
        """
        Generate a unique slug from company name
        """
        base_slug = slugify(self.company_name)
        slug = base_slug
        counter = 1
        
        while (Vendor.objects.filter(slug=slug)
               .exclude(id=self.id)
               .exists()):
            slug = f"{base_slug}-{counter}"
            counter += 1
            
        return slug

    def get_readonly_fields(self, request, obj=None):
        if obj:  # editing an existing object
            return self.readonly_fields + ('user',)
        return self.readonly_fields
         
class TranslatedVendorInfo(models.Model):
    """Store translated information about vendors"""
    vendor = models.ForeignKey(Vendor, on_delete=models.CASCADE, related_name='translations')
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    company_name = models.CharField(max_length=255, verbose_name=_('Translated Company Name'))
    description = models.TextField(verbose_name=_('Translated Business Description'))
    short_description = models.CharField(max_length=280, blank=True, verbose_name=_('Translated Short Description'))
    
    class Meta:
        unique_together = ('vendor', 'language')
        verbose_name = _('Translated Vendor Information')
        verbose_name_plural = _('Translated Vendor Information')
        
    def __str__(self):
        return f"{self.company_name} ({self.language.code})"
 
class BusinessCategory(models.Model):
    """Main business category like 'Food & Beverage', 'Accommodation'"""
    name = models.CharField(max_length=100)
    slug = models.SlugField(unique=True)
    description = models.TextField(blank=True)
    order = models.PositiveIntegerField(default=0)
    icon = models.CharField(max_length=50, blank=True)  

    class Meta:
        verbose_name = _('Business Category')
        verbose_name_plural = _('Business Categories')
        ordering = ['order', 'name']

    def __str__(self):
        return self.name

class BusinessType(models.Model):
    """Specific business type like 'Restaurants', 'Hotels'"""
    category = models.ForeignKey(
        BusinessCategory, 
        on_delete=models.PROTECT,
        related_name='business_types'
    )
    name = models.CharField(max_length=100)
    slug = models.SlugField(unique=True)
    description = models.TextField(blank=True)
    order = models.PositiveIntegerField(default=0)
 
    service_level = models.CharField(
        max_length=20,
        choices=[
            ('luxury', _('Luxury')),
            ('premium', _('Premium')),
            ('standard', _('Standard')),
            ('budget', _('Budget')),
            ('economy', _('Economy')),
        ],
        blank=True
    )
    
    target_audience = models.CharField(
        max_length=20,
        choices=[
            ('families', _('Families')),
            ('solo', _('Solo Travelers')),
            ('groups', _('Groups')),
            ('business', _('Business Travelers')),
            ('adventure', _('Adventure Seekers')),
            ('cultural', _('Cultural Enthusiasts')),
            ('luxury', _('Luxury Travelers')),
        ],
        blank=True
    )
    
    operating_season = models.CharField(
        max_length=20,
        choices=[
            ('year_round', _('Year-Round')),
            ('seasonal', _('Seasonal')),
            ('summer', _('Summer Only')),
            ('winter', _('Winter Only')),
        ],
        blank=True
    )

    class Meta:
        verbose_name = _('Business Type')
        verbose_name_plural = _('Business Types')
        ordering = ['category', 'order', 'name']
        unique_together = ['category', 'slug']

    def __str__(self):
        return f"{self.category.name} - {self.name}"

class TranslatedBusinessCategory(models.Model):
    business_category = models.ForeignKey(
        BusinessCategory, 
        on_delete=models.CASCADE,
        related_name='translations'
    )
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=100)
    description = models.TextField(blank=True)

class TranslatedBusinessType(models.Model):
    business_type = models.ForeignKey(
        BusinessType, 
        on_delete=models.CASCADE,
        related_name='translations'
    )
    language = models.ForeignKey(Language, on_delete=models.CASCADE)
    name = models.CharField(max_length=100)
    description = models.TextField(blank=True)
 
class VendorContact(models.Model):
    """Contact person for a vendor"""
    vendor = models.ForeignKey(Vendor, on_delete=models.CASCADE, related_name='contacts')
    name = models.CharField(max_length=255, verbose_name=_('Contact Name'))
    position = models.CharField(max_length=100, blank=True, verbose_name=_('Position'))
    email = models.EmailField(verbose_name=_('Email'))
    phone = models.CharField(max_length=50, blank=True, verbose_name=_('Phone'))
    is_primary = models.BooleanField(default=False, verbose_name=_('Primary Contact'))
    
    class Meta:
        verbose_name = _('Vendor Contact')
        verbose_name_plural = _('Vendor Contacts')
        
    def __str__(self):
        return f"{self.name} ({self.vendor.company_name})"
 
class VendorReview(models.Model):
    """Reviews for vendors"""
    vendor = models.ForeignKey(Vendor, on_delete=models.CASCADE, related_name='reviews')
    user = models.ForeignKey(CustomUser, on_delete=models.CASCADE, related_name='vendor_reviews')
    rating = models.PositiveSmallIntegerField(verbose_name=_('Rating'))
    title = models.CharField(max_length=255, blank=True, verbose_name=_('Review Title'))
    comment = models.TextField(verbose_name=_('Review Comment'))
    created_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        verbose_name = _('Vendor Review')
        verbose_name_plural = _('Vendor Reviews')
        unique_together = ('vendor', 'user')
        
    def __str__(self):
        return f"Review for {self.vendor.company_name} by {self.user.username}"
 
class VendorOpeningHours(models.Model):
    """Store vendor opening hours"""
    vendor = models.ForeignKey(Vendor, on_delete=models.CASCADE, related_name='opening_hours')
    day_of_week = models.PositiveSmallIntegerField(verbose_name=_('Day of Week'))
    from_hour = models.TimeField(verbose_name=_('From Hour'))
    to_hour = models.TimeField(verbose_name=_('To Hour'))
    
    class Meta:
        verbose_name = _('Vendor Opening Hours')
        verbose_name_plural = _('Vendor Opening Hours')
        ordering = ['day_of_week', 'from_hour']
        
    def __str__(self):
        days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday']
        return f"{days[self.day_of_week]} {self.from_hour}-{self.to_hour}"
 
class VendorDocument(models.Model):
    """Store vendor verification documents"""
    vendor = models.ForeignKey(Vendor, on_delete=models.CASCADE, related_name='documents')
    name = models.CharField(max_length=255, verbose_name=_('Document Name'))
    file = models.FileField(upload_to='vendor_documents', verbose_name=_('Document File'), blank=True, default=None)
    document_type = models.CharField(max_length=100, verbose_name=_('Document Type'))
    uploaded_at = models.DateTimeField(auto_now_add=True)
    verified = models.BooleanField(default=False)
    
    class Meta:
        verbose_name = _('Vendor Document')
        verbose_name_plural = _('Vendor Documents')
        
    def __str__(self):
        return f"{self.name} - {self.vendor.company_name}"
 
class SiteClaim(models.Model):
    site = models.ForeignKey('sites.Site', on_delete=models.CASCADE, related_name='vendor_claims')
    vendor = models.ForeignKey('users.Vendor', on_delete=models.CASCADE, related_name='claimed_sites')
    status = models.CharField(
        max_length=20,
        choices=[
            ('PENDING', _('Pending')),
            ('APPROVED', _('Approved')),
            ('REJECTED', _('Rejected'))
        ],
        default='PENDING'
    )
    proof_documents = models.ManyToManyField(
        'users.VendorDocument',  
        blank=True,
        related_name='site_claims',
        verbose_name=_('Proof documents')
    )

    created_at = models.DateTimeField(auto_now_add=True)
    processed_at = models.DateTimeField(null=True, blank=True)
    processed_by = models.ForeignKey(
        'users.CustomUser', 
        on_delete=models.SET_NULL, 
        null=True, 
        related_name='processed_site_claims'
    )
    notes = models.TextField(blank=True)

    class Meta:
        verbose_name = _('Site Claim')
        verbose_name_plural = _('Site Claims')
        unique_together = ['site', 'vendor']
        ordering = ['-created_at']
        
    def __str__(self):
        return f"{self.site.title} - {self.vendor.company_name} ({self.get_status_display()})"
 
class SubscriptionPlan(models.Model):
    name = models.CharField(max_length=100)
    type = models.CharField(
        max_length=20,
        choices=VendorSubscriptionType.choices
    )
    price = models.DecimalField(max_digits=10, decimal_places=2)
    billing_period = models.CharField(
        max_length=20,
        choices=[
            ('monthly', _('Monthly')),
            ('yearly', _('Yearly')),
            ('custom', _('Custom'))
        ],
        default='monthly'
    )
    duration_days = models.PositiveIntegerField(null=True, blank=True)
    max_sites = models.IntegerField()
    max_events = models.IntegerField()
    can_upload_images = models.BooleanField(default=False)
    max_images_per_site = models.IntegerField(default=5)
    can_add_events = models.BooleanField(default=False)
    can_add_special_offers = models.BooleanField(default=False)
    support_level = models.CharField(
        max_length=20,
        choices=[
            ('email', _('Email Support')),
            ('priority', _('Priority Support')),
            ('dedicated', _('Dedicated Support'))
        ],
        default='email'
    )
    is_active = models.BooleanField(default=True)
    can_create_promotions = models.BooleanField(default=False)
    max_active_promotions = models.IntegerField(default=0)

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = _('Subscription Plan')
        verbose_name_plural = _('Subscription Plans')
        ordering = ['price', 'name']
 
class PlanFeature(models.Model):
    FEATURE_TYPES = [
        ('boolean', 'Boolean'),
        ('numeric', 'Numeric'),
        ('text', 'Text')
    ]
    
    name = models.CharField(max_length=100)
    description = models.TextField()
    feature_type = models.CharField(max_length=20, choices=FEATURE_TYPES)
    
    class Meta:
        verbose_name = _('Plan Feature')
        verbose_name_plural = _('Plan Features')
 
class PlanFeatureValue(models.Model):
    plan = models.ForeignKey(SubscriptionPlan, on_delete=models.CASCADE, related_name='feature_values')
    feature = models.ForeignKey(PlanFeature, on_delete=models.CASCADE)
    boolean_value = models.BooleanField(null=True, blank=True)
    numeric_value = models.IntegerField(null=True, blank=True)
    text_value = models.CharField(max_length=255, null=True, blank=True)
    
    def get_value(self):
        if self.feature.feature_type == 'boolean':
            return self.boolean_value
        elif self.feature.feature_type == 'numeric':
            return self.numeric_value
        return self.text_value

    def __str__(self):
        return f"{self.plan.name} - {self.feature.name}: {self.get_value()}"
 
class VendorSubscription(models.Model):
    vendor = models.ForeignKey(Vendor, on_delete=models.CASCADE, related_name='subscriptions')
    plan = models.ForeignKey(SubscriptionPlan, on_delete=models.PROTECT)
    start_date = models.DateTimeField()
    end_date = models.DateTimeField()
    status = models.CharField(
        max_length=20,
        choices=[
            ('active', _('Active')),
            ('cancelled', _('Cancelled')),
            ('expired', _('Expired')),
            ('pending', _('Pending'))
        ],
        default='pending'
    )
    payment_status = models.CharField(
        max_length=20,
        choices=[
            ('pending', _('Pending')),
            ('completed', _('Completed')),
            ('failed', _('Failed')),
            ('refunded', _('Refunded'))
        ],
        default='pending'
    )
    payment_id = models.CharField(max_length=255, blank=True)
    sites_quota_used = models.IntegerField(default=0)
    events_quota_used = models.IntegerField(default=0)
    auto_renew = models.BooleanField(default=True)
    cancelled_at = models.DateTimeField(null=True, blank=True)
    
    class Meta:
        verbose_name = _('Vendor Subscription')
        verbose_name_plural = _('Vendor Subscriptions')
        ordering = ['-start_date']

    def is_active(self):
        return (
            self.status == 'active' and 
            self.start_date <= timezone.now() <= self.end_date
        )

    def has_available_quota(self, site_type='site'):
        if site_type == 'site':
            return self.sites_quota_used < self.plan.max_sites
        return self.events_quota_used < self.plan.max_events

    def add_site(self, site):
        """Add a site to this subscription"""
        if not self.has_available_quota():
            raise ValidationError(_("No available site quota in this subscription"))
        
        if not self.is_active():
            raise ValidationError(_("Subscription is not active"))

        # Create the site subscription
        site_subscription = VendorSiteSubscription.objects.create(
            vendor_subscription=self,
            site=site
        )

        # Update quota
        self.sites_quota_used += 1
        self.save()

        return site_subscription
    
    def remove_site(self, site):
        """Remove a site from this subscription"""
        site_subscription = self.site_subscriptions.filter(site=site).first()
        if site_subscription:
            site_subscription.is_active = False
            site_subscription.save()
            
            # Update site status
            site.subscription_status = 'expired'
            site.is_published = False
            site.save()

            # Update quota
            self.sites_quota_used -= 1
            self.save()

    def get_active_sites(self):
        """Get all active sites under this subscription"""
        return self.site_subscriptions.filter(
            is_active=True
        ).select_related('site')
            
class VendorSiteSubscription(models.Model):
    vendor_subscription = models.ForeignKey(
        'VendorSubscription', 
        on_delete=models.CASCADE, 
        related_name='site_subscriptions'
    )
    site = models.ForeignKey(
        'sites.Site', 
        on_delete=models.CASCADE, 
        related_name='vendor_subscriptions'
    )
    activated_at = models.DateTimeField(auto_now_add=True)
    is_active = models.BooleanField(default=True)

    class Meta:
        unique_together = ['vendor_subscription', 'site']

    def save(self, *args, **kwargs):
        is_new = not self.pk
        super().save(*args, **kwargs)
        
        if is_new:
            self.activate_site()

    def activate_site(self):
        site = self.site
        subscription = self.vendor_subscription
        
        site.subscription_status = 'subscribed'
        site.is_published = True
        site.subscription_start_date = subscription.start_date
        site.subscription_end_date = subscription.end_date
        site.save()
 
class SubscriptionTransaction(models.Model):
    subscription = models.ForeignKey(
        VendorSubscription, 
        on_delete=models.CASCADE, 
        related_name='transactions'
    )
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    transaction_id = models.CharField(max_length=255)
    status = models.CharField(
        max_length=20,
        choices=[
            ('pending', _('Pending')),
            ('completed', _('Completed')),
            ('failed', _('Failed')),
            ('refunded', _('Refunded'))
        ]
    )
    created_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        verbose_name = _('Subscription Transaction')
        verbose_name_plural = _('Subscription Transactions')
 
class SubscriptionUsageReport(models.Model):
    subscription = models.ForeignKey(
        VendorSubscription,
        on_delete=models.CASCADE,
        related_name='usage_reports'
    )
    report_date = models.DateField()
    sites_count = models.IntegerField()
    sites_quota_percentage = models.DecimalField(
        max_digits=5,
        decimal_places=2
    )
    events_count = models.IntegerField()
    events_quota_percentage = models.DecimalField(
        max_digits=5,
        decimal_places=2
    )
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = _('Subscription Usage Report')
        verbose_name_plural = _('Subscription Usage Reports')
        unique_together = ['subscription', 'report_date']
        ordering = ['-report_date']

    def __str__(self):
        return f"{self.subscription.vendor.company_name} - {self.report_date}"
 
class SubscriptionAuditLog(models.Model):
    ACTIONS = [
        ('created', _('Created')),
        ('renewed', _('Renewed')),
        ('upgraded', _('Upgraded')),
        ('downgraded', _('Downgraded')),
        ('cancelled', _('Cancelled')),
        ('expired', _('Expired')),
        ('payment_failed', _('Payment Failed')),
    ]

    subscription = models.ForeignKey(
        VendorSubscription,
        on_delete=models.CASCADE,
        related_name='audit_logs'
    )
    action = models.CharField(max_length=20, choices=ACTIONS)
    old_plan = models.ForeignKey(
        SubscriptionPlan,
        on_delete=models.SET_NULL,
        null=True,
        related_name='+'
    )
    new_plan = models.ForeignKey(
        SubscriptionPlan,
        on_delete=models.SET_NULL,
        null=True,
        related_name='+'
    )
    details = models.JSONField(default=dict)
    created_by = models.ForeignKey(
        'users.CustomUser',
        on_delete=models.SET_NULL,
        null=True
    )
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = _('Subscription Audit Log')
        verbose_name_plural = _('Subscription Audit Logs')
        ordering = ['-created_at']

    def __str__(self):
        return f"{self.subscription.vendor.company_name} - {self.action}"