from django.contrib.auth.models import Group
from django.core.validators import RegexValidator, URLValidator
from django.db.models import DurationField, ExpressionWrapper, F
from django.forms import ValidationError
from django.utils import translation
from django.utils.timezone import now
from django.utils.translation import gettext_lazy as _
from rest_framework import serializers
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework import viewsets, filters, status
from rest_framework.decorators import action
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from django.core.exceptions import ValidationError
from django_filters.rest_framework import DjangoFilterBackend
  
from local_secrets.payment.models import PaymentMethod
from local_secrets.sites.models import Site, SiteImage
from .models import CustomUser, GroupDescription, Notification, SubscriptionUsageReport, Tag
from ..core.custom_exceptions import CustomApiException
from ..core.serializers import ThumbnailJSONSerializer
from ..languages.serializers import LanguageSerializer
from ..operations.models import DefaultText
from local_secrets.users.models import (VendorVerificationStatus,
    Vendor, VendorSubscription, SubscriptionPlan, VendorContact, VendorReview,
    SiteClaim, VendorDocument, SubscriptionTransaction, BusinessCategory, BusinessType, TranslatedBusinessCategory, TranslatedBusinessType
)
from ..cities.models import City
from rest_framework.permissions import BasePermission
from .validators import VendorValidators
from .choices import BUSINESS_TYPE_CHOICES
from django.utils.timezone import now
from datetime import timedelta, datetime, timezone
from local_secrets.users.services.subscription_services import SubscriptionService
from django.utils.text import slugify
from django.db import transaction
import logging

logger = logging.getLogger(__name__)

class IsVendor(BasePermission):
    """
    Permission check for vendor access
    """
    def has_permission(self, request, view):
        return bool(request.user and request.user.is_vendor)

class TagOutputSerializer(serializers.ModelSerializer):
    is_selected = serializers.BooleanField()

    def to_representation(self, instance):
        representation = super(TagOutputSerializer, self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
        else:
            return representation

        for field in self.fields:
            if field in [
                'title',
            ]:
                representation[field] = instance.display_text(field, language)
        return representation

    class Meta:
        model = Tag
        fields = '__all__'
 
class TagOutputWithoutSelectionSerializer(TagOutputSerializer):
    is_selected = None
 
class TagSelectionInputSerializer(serializers.Serializer):
    tags = serializers.ListField(child=serializers.IntegerField())
 
class UserInputSerializer(serializers.ModelSerializer):
    username = serializers.CharField()
    email = serializers.EmailField()
    first_name = serializers.CharField(required=False)
    last_name = serializers.CharField(required=False)
    password = serializers.CharField(
        validators=[
            RegexValidator(
                regex=r'^(?=.*[A-Z])*(?=.*\d)(?=.*[@$!%*#?&.])*(?=.*\d)[A-Z\d@$!%*#?&.]*(?=.*\d).{7,}$',
                message=_(
                    'The password must be 8 characters long, have an uppercase letter, a number and '
                    'a special character'
                ),
                code='invalid_password',
            )
        ],
        required=False,
    )
    phone = serializers.CharField(
        required=False,
        allow_blank=True,
        validators=[
            RegexValidator(
                regex=r'^\d{9,15}$',
                message=_('The phone must contain between 9 and 15 numbers'),
                code='invalid_phone',
            )
        ],
    )
    phone_prefix = serializers.CharField(required=False)

    def is_valid(self, raise_exception=False):
        #with translation.override(self.context.get('request').headers.get('language')):
        is_valid = super().is_valid(raise_exception)
        users = CustomUser.objects.all()
        if not self.context.get('request').user.is_anonymous:
            users = users.exclude(id=self.context.get('request').user.id)
        if users.filter(email=self.validated_data.get('email')).exists():
            if raise_exception:
                raise CustomApiException(message=_('There is an user with the same email'), status_code=400)
            return False
        if users.filter(username=self.validated_data.get('username')).exists():
            if raise_exception:
                raise CustomApiException(message=_('There is an user with the same username'), status_code=400)
            return False
        return is_valid

    class Meta:
        model = CustomUser
        fields = ('username', 'email', 'first_name', 'last_name', 'password', 'phone', 'phone_prefix')
 
class GroupSerializer(serializers.ModelSerializer):
    def to_representation(self, instance):
        representation = super(GroupSerializer, self).to_representation(instance)
        request = self.context.get('request')

        if request and request.headers.get('language'):
            language = request.headers.get('language')
        else:
            return representation

        for field in self.fields:
            if field in ['description', 'name']:
                try:
                    representation[field] = GroupDescription.objects.get(group__id=instance.id).display_text(
                        field, language
                    )
                except BaseException as e:
                    print(e)
                    pass
        return representation

    description = serializers.SerializerMethodField()
    general_description = serializers.SerializerMethodField()

    def get_description(self, obj):
        try:
            return GroupDescription.objects.get(group__id=obj.id).description
        except GroupDescription.DoesNotExist:
            return ""

    def get_general_description(self, obj):
        try:
            return (
                DefaultText.objects.filter(title='group_description')
                .first()
                .display_text('text', self.context.get('request').headers.get('language'))
            )
        except Exception:
            return ''

    class Meta:
        model = Group
        fields = ('name', 'description', 'general_description')
 
class UserOutputSerializer(serializers.ModelSerializer):
    tags = TagOutputWithoutSelectionSerializer(many=True)
    profile_picture = ThumbnailJSONSerializer(alias='', read_only=True)
    next_trip = serializers.SerializerMethodField(allow_null=True)
    current_trip = serializers.SerializerMethodField(allow_null=True)
    is_on_trip = serializers.SerializerMethodField()
    num_of_past_travels = serializers.IntegerField()
    num_of_upcoming_travels = serializers.IntegerField()
    visited_places = serializers.IntegerField(default=0)
    visited_events = serializers.IntegerField(default=0)
    groups = GroupSerializer(many=True, read_only=True, allow_null=True)
    language = LanguageSerializer()

    def get_is_on_trip(self, obj):
        current_time = now()
        for travel in obj.travels.all():
            if travel.initial_date <= current_time.date() <= travel.end_date:
                return True
        return False

    def get_next_trip(self, obj):
        from ..travels.serializers import TravelListSerializer

        subquery = ExpressionWrapper((F('initial_date') - now().date()), output_field=DurationField())
        travels = obj.travels.filter(initial_date__gte=now().date())
        if not travels.exists():
            return None
        return TravelListSerializer(
            travels.annotate(days_until_trip=subquery).order_by('initial_date').first(),
            context=self.context,
        ).data

    def get_current_trip(self, obj):
        from ..travels.serializers import TravelListSerializer

        travels = obj.travels.filter(end_date__gte=now().date(), initial_date__lte=now().date())
        if not travels.exists():
            return None
        return TravelListSerializer(
            travels.annotate(
                days_until_trip=ExpressionWrapper((F('initial_date') - now().date()), output_field=DurationField())
            )
            .order_by('-initial_date')
            .first(),
            context=self.context,
        ).data

    def get_profile_picture(self, obj):
        if obj.profile_picture:
            return self.context['request'].build_absolute_uri(obj.profile_picture.url)
        return None

    class Meta:
        model = CustomUser
        exclude = (
            'password',
            'is_superuser',
            'is_staff',
            'is_active',
            'user_permissions',
        )
 
class UserCommentOutputSerializer(serializers.ModelSerializer):
    profile_picture = ThumbnailJSONSerializer(alias='', read_only=True)

    class Meta:
        model = CustomUser
        fields = ('username', 'profile_picture')
 
class NotificationSerializer(serializers.ModelSerializer):
    # Add extra fields
    has_been_seen = serializers.BooleanField()

    class Meta:
        model = Notification
        fields = ('id', 'title', 'body', 'link', 'site', 'has_been_seen', 'created_at')
 
class NotificationDetailSerializer(serializers.ModelSerializer):
    class Meta:
        model = Notification
        # fields = '__all__'
        fields = ('id', 'title', 'body', 'link', 'site', 'created_at')
 
class EmailSerializer(serializers.Serializer):
    email = serializers.EmailField()
 
class ResetPasswordSerializer(serializers.Serializer):
    old_password = serializers.CharField(allow_null=False)
    new_password = serializers.CharField(allow_null=False, allow_blank=False)

    def is_valid(self, raise_exception=False):
        is_valid = super().is_valid(raise_exception)
        old_password = self.validated_data['old_password']
        new_password = self.validated_data['new_password']
        if old_password == new_password:
            raise serializers.ValidationError(
                code=400, detail='The new password cannot be the same as the previous ' 'password'
            )
        return is_valid

class SubscriptionPlanSerializer(serializers.ModelSerializer):
    """
    Serializer for SubscriptionPlan model that includes all relevant fields
    and computed features
    """
    features = serializers.SerializerMethodField()
    
    class Meta:
        model = SubscriptionPlan
        fields = [
            'id', 'name', 'type', 'price', 'billing_period',
            'duration_days', 'max_sites', 'max_events',
            'can_upload_images', 'max_images_per_site',
            'can_add_events', 'can_add_special_offers',
            'support_level', 'is_active', 'features'
        ]

    def get_features(self, obj):
        """
        Convert individual feature fields into a structured features list
        """
        features = []
        
        # Basic features
        if obj.max_sites > 0:
            features.append({
                'name': 'sites',
                'value': obj.max_sites,
                'label': _('Sites allowed'),
                'type': 'numeric'
            })
            
        if obj.max_events > 0:
            features.append({
                'name': 'events',
                'value': obj.max_events,
                'label': _('Events allowed'),
                'type': 'numeric'
            })

        # Image features
        if obj.can_upload_images:
            features.append({
                'name': 'images',
                'value': obj.max_images_per_site,
                'label': _('Images per site'),
                'type': 'numeric'
            })

        # Event features
        if obj.can_add_events:
            features.append({
                'name': 'can_add_events',
                'value': True,
                'label': _('Event management'),
                'type': 'boolean'
            })

        # Special offers feature
        if obj.can_add_special_offers:
            features.append({
                'name': 'special_offers',
                'value': True,
                'label': _('Special offers'),
                'type': 'boolean'
            })

        # Support level
        features.append({
            'name': 'support',
            'value': obj.support_level,
            'label': _('Support Level'),
            'type': 'text'
        })

        return features

class VendorSubscriptionSerializer(serializers.ModelSerializer):
    plan_details = SubscriptionPlanSerializer(source='plan', read_only=True)
    remaining_site_quota = serializers.SerializerMethodField()
    remaining_event_quota = serializers.SerializerMethodField()
    
    class Meta:
        model = VendorSubscription
        fields = [
            'id', 'vendor', 'plan', 'plan_details', 'start_date', 'end_date',
            'status', 'payment_status', 'sites_quota_used', 'events_quota_used',
            'remaining_site_quota', 'remaining_event_quota', 'auto_renew'
        ]
        read_only_fields = ['status', 'payment_status', 'sites_quota_used', 'events_quota_used']

    def get_remaining_site_quota(self, obj):
        return obj.plan.max_sites - obj.sites_quota_used

    def get_remaining_event_quota(self, obj):
        return obj.plan.max_events - obj.events_quota_used
 
class VendorDocumentSerializer(serializers.ModelSerializer):
    class Meta:
        model = VendorDocument
        fields = ['id', 'name', 'file', 'document_type', 'uploaded_at', 'verified']
        read_only_fields = ['verified']
 
class SiteClaimSerializer(serializers.ModelSerializer):
    documents = VendorDocumentSerializer(many=True, required=False)
    
    class Meta:
        model = SiteClaim
        fields = [
            'id', 'site', 'vendor', 'status', 'documents',
            'created_at', 'processed_at', 'notes'
        ]
        read_only_fields = ['status', 'processed_at', 'processed_by']

    def validate(self, data):
        site = data.get('site')
        if site.vendor:
            raise serializers.ValidationError(_("This site is already claimed by a vendor"))
        return data
 
class SubscriptionTransactionSerializer(serializers.ModelSerializer): 
    class Meta:
        model = SubscriptionTransaction
        fields = ['id', 'subscription', 'amount', 'transaction_id', 'status', 'created_at']
        read_only_fields = ['status', 'created_at']
 
class BusinessCategorySerializer(serializers.ModelSerializer):
    class Meta:
        model = BusinessCategory
        fields = ['id', 'name', 'slug', 'description', 'icon']

    def to_representation(self, instance):
        representation = super().to_representation(instance)
        request = self.context.get('request')
        
        if request and request.headers.get('language'):
            language = request.headers.get('language')
            try:
                translation = instance.translations.get(language__code=language)
                representation['name'] = translation.name
                representation['description'] = translation.description
            except TranslatedBusinessCategory.DoesNotExist:
                pass
        
        return representation
 
class BusinessTypeSerializer(serializers.ModelSerializer):
    category = BusinessCategorySerializer(read_only=True)
    
    class Meta:
        model = BusinessType
        fields = [
            'id', 'category', 'name', 'slug', 'description',
            'service_level', 'target_audience', 'operating_season'
        ]

    def to_representation(self, instance):
        representation = super().to_representation(instance)
        request = self.context.get('request')
        
        if request and request.headers.get('language'):
            language = request.headers.get('language')
            try:
                translation = instance.translations.get(language__code=language)
                representation['name'] = translation.name
                representation['description'] = translation.description
            except TranslatedBusinessType.DoesNotExist:
                pass
        
        return representation

class VendorRegistrationSerializer(serializers.Serializer):
    company_name = serializers.CharField(max_length=255)
    contact_email = serializers.EmailField()
    business_type = serializers.IntegerField()
    description = serializers.CharField(required=False, allow_blank=True)
    tax_id = serializers.CharField(required=False, allow_blank=True)
    website = serializers.URLField(required=False, allow_blank=True)
    short_description = serializers.CharField(required=False, allow_blank=True)
    founded_year = serializers.IntegerField(required=False)
    facebook_url = serializers.URLField(required=False, allow_blank=True)
    instagram_url = serializers.URLField(required=False, allow_blank=True)
    twitter_url = serializers.URLField(required=False, allow_blank=True)
    primary_address = serializers.CharField(required=False, allow_blank=True)
    cities = serializers.ListField(
        child=serializers.IntegerField(),
        required=False
    )

    def validate_business_type(self, value):
        try:
            return BusinessType.objects.get(id=value)
        except BusinessType.DoesNotExist:
            raise serializers.ValidationError("Invalid business type")

    def validate_cities(self, value):
        if value:
            cities = City.objects.filter(id__in=value)
            if len(cities) != len(value):
                raise serializers.ValidationError("One or more invalid city IDs")
            return cities
        return []

    def validate_company_name(self, value):
        """
        Check that the company name is unique for active vendors
        """
        # Only check against active vendors (not deleted or rejected)
        existing_vendor = Vendor.objects.filter(
            company_name__iexact=value,
            verification_status__in=[
                VendorVerificationStatus.PENDING,
                VendorVerificationStatus.VERIFIED
            ]
        ).exists()
        
        if existing_vendor:
            raise serializers.ValidationError('A company with this name already exists')
        return value

    def validate_contact_email(self, value):
        """
        Check that the contact email is unique for active vendors
        """
        # Check if email belongs to requesting user
        user = self.context['request'].user
        if user.email.lower() == value.lower():
            return value
            
        # Only check against active vendors
        existing_vendor = Vendor.objects.filter(
            contact_email__iexact=value,
            verification_status__in=[
                VendorVerificationStatus.PENDING,
                VendorVerificationStatus.VERIFIED
            ]
        ).exists()
        
        if existing_vendor:
            raise serializers.ValidationError('This email is already registered as a vendor')
        return value

    def validate(self, data):
        user = self.context['request'].user
        
        # Check if user has an active vendor profile
        existing_vendor = Vendor.objects.filter(
            user=user,
            verification_status__in=[
                VendorVerificationStatus.PENDING,
                VendorVerificationStatus.VERIFIED
            ]
        ).first()
        
        if existing_vendor:
            if existing_vendor.verification_status == VendorVerificationStatus.PENDING:
                raise serializers.ValidationError({
                    "detail": "You already have a pending vendor registration"
                })
            else:
                raise serializers.ValidationError({
                    "detail": "You already have an active vendor profile"
                })
        
        # If user had a rejected vendor profile, allow them to try again
        rejected_vendor = Vendor.objects.filter(
            user=user,
            verification_status=VendorVerificationStatus.REJECTED
        ).first()
        
        if rejected_vendor:
            # Update the existing rejected vendor instead of creating new
            self.instance = rejected_vendor
        
        return data

    def create(self, validated_data):
        user = self.context['request'].user
        
        # Extract M2M fields
        cities = validated_data.pop('cities', [])
        
        if hasattr(self, 'instance') and self.instance:
            # Update existing rejected vendor
            for attr, value in validated_data.items():
                setattr(self.instance, attr, value)
            self.instance.verification_status = VendorVerificationStatus.PENDING
            self.instance.save()
            vendor = self.instance
        else:
            # Create new vendor
            vendor = Vendor.objects.create(
                user=user,
                verification_status=VendorVerificationStatus.PENDING,
                **validated_data
            )
        
        # Set M2M fields
        if cities:
            vendor.cities.set(cities)
            
        # Add to vendor group
        vendor_group, _ = Group.objects.get_or_create(name='Vendor')
        user.groups.add(vendor_group)
        
        return vendor
 
    def validate_founded_year(self, value):
        """
        Validate that founded_year is not in the future and not too far in the past
        """
        current_year = datetime.now().year
        if value > current_year:
            raise serializers.ValidationError("Founded year cannot be in the future")
        if value < 1800:  # You can adjust this minimum year as needed
            raise serializers.ValidationError("Founded year seems too far in the past")
        return value

    def validate_facebook_url(self, value):
        """Validate Facebook URL format"""
        if value:
            if not value.startswith('https://facebook.com/') and not value.startswith('https://www.facebook.com/'):
                raise serializers.ValidationError("Invalid Facebook URL format")
        return value

    def validate_instagram_url(self, value):
        """Validate Instagram URL format"""
        if value:
            if not value.startswith('https://instagram.com/') and not value.startswith('https://www.instagram.com/'):
                raise serializers.ValidationError("Invalid Instagram URL format")
        return value

    def validate_twitter_url(self, value):
        """Validate Twitter URL format"""
        if value:
            if not value.startswith('https://twitter.com/') and not value.startswith('https://www.twitter.com/'):
                raise serializers.ValidationError("Invalid Twitter URL format")
        return value

    def _setup_vendor_services(self, vendor):
        """Initialize vendor services and subscription"""
        try:
            # Create free subscription
            subscription = SubscriptionService.create_subscription(
                vendor=vendor,
                plan=SubscriptionPlan.objects.get(name='Free'),
                payment_method=None
            )
            
            # Create initial usage report
            SubscriptionUsageReport.objects.create(
                subscription=subscription,
                sites_count=0,
                events_count=0,
                sites_quota_percentage=0,
                events_quota_percentage=0,
                created_at=now(),
                updated_at=now()
            )
            
            # Setup payment method
            SubscriptionService.create_payment_method(
                vendor=vendor,
                payment_type='FREE',
                is_default=True,
                is_verified=True
            )
            
            # Setup automatic billing
            SubscriptionService.create_automatic_billing(
                vendor=vendor,
                enabled=False,
                billing_day=1,
                next_billing_date=now() + timedelta(days=30),
                payment_method=None,
                retry_count=0
            )
            
        except Exception as e:
            raise serializers.ValidationError(f"Error setting up vendor services: {str(e)}")

    def _generate_unique_slug(self, company_name):
        """
        Generate a unique slug from company name
        """
        base_slug = slugify(company_name)
        slug = base_slug
        counter = 1
        
        while Vendor.objects.filter(slug=slug).exists():
            slug = f"{base_slug}-{counter}"
            counter += 1
            
        return slug
        
    def create(self, validated_data):
        user = self.context['request'].user
        
        # Extract M2M fields
        cities = validated_data.pop('cities', [])
        
        # Generate slug from company name
        slug = self._generate_unique_slug(validated_data['company_name'])
        
        with transaction.atomic():
            if hasattr(self, 'instance') and self.instance:
                # Update existing rejected vendor
                for attr, value in validated_data.items():
                    setattr(self.instance, attr, value)
                self.instance.verification_status = VendorVerificationStatus.PENDING
                self.instance.slug = slug
                self.instance.save()
                vendor = self.instance
            else:
                # Create new vendor
                vendor = Vendor.objects.create(
                    user=user,
                    verification_status=VendorVerificationStatus.PENDING,
                    slug=slug,  # Add the slug here
                    **validated_data
                )
            
            # Set M2M fields
            if cities:
                vendor.cities.set(cities)
                
            # Add to vendor group
            vendor_group, _ = Group.objects.get_or_create(name='Vendor')
            user.groups.add(vendor_group)
            
            return vendor

    def to_representation(self, instance):
        """
        Customize the response data
        """
        return {
            'id': instance.id,
            'company_name': instance.company_name,
            'slug': instance.slug,
            'status': instance.verification_status,
            'message': 'Vendor registration successful. Awaiting approval.'}
 

class VendorSerializer(serializers.ModelSerializer):
    business_type = BusinessTypeSerializer(read_only=True)
    secondary_business_types = BusinessTypeSerializer(many=True, read_only=True)
    categories = BusinessCategorySerializer(many=True, read_only=True)
    average_rating = serializers.FloatField(read_only=True)
    
    class Meta:
        model = Vendor
        fields = [
            'id', 'uuid', 'company_name', 'slug', 'business_type',
            'secondary_business_types', 'tax_id', 'website',
            'description', 'short_description', 'founded_year',
            'verification_status', 'verified_at',
            'subscription_type', 'subscription_expires',
            'contact_email', 'contact_phone',
            'facebook_url', 'instagram_url', 'twitter_url',
            'logo', 'header_image', 'cities', 'categories',
            'primary_address', 'is_featured', 'average_rating',
            'created_at', 'updated_at'
        ]

class VendorListSerializer(serializers.ModelSerializer):
    business_type = BusinessTypeSerializer(read_only=True)
    average_rating = serializers.FloatField(read_only=True)
    
    class Meta:
        model = Vendor
        fields = [
            'id', 'uuid', 'company_name', 'slug', 'business_type',
            'short_description', 'logo', 'is_featured',
            'average_rating'
        ]

class VendorContactSerializer(serializers.ModelSerializer):
    class Meta:
        model = VendorContact
        fields = ['id', 'name', 'position', 'email', 'phone', 'is_primary']

class VendorReviewSerializer(serializers.ModelSerializer):
    class Meta:
        model = VendorReview
        fields = ['id', 'rating', 'title', 'comment', 'created_at']
 
class VendorStatusSerializer(serializers.ModelSerializer):
    business_type = serializers.SerializerMethodField()
    secondary_business_types = serializers.SerializerMethodField()
    subscription_status = serializers.SerializerMethodField()
    categories = serializers.SerializerMethodField()
    operating_cities = serializers.SerializerMethodField()
    verification_details = serializers.SerializerMethodField()
    sites_count = serializers.SerializerMethodField()
    
    class Meta:
        model = Vendor
        fields = [
            'id',
            'uuid',
            'company_name',
            'slug',
            'business_type',
            'secondary_business_types',
            'verification_status',
            'subscription_type',
            'subscription_status',
            'verification_details',
            'categories',
            'operating_cities',
            'sites_count',
            'is_featured',
            'created_at',
            'updated_at'
        ]
        read_only_fields = fields

    def get_business_type(self, obj):
        return {
            'id': obj.business_type.id,
            'name': obj.business_type.name,
            'slug': obj.business_type.slug
        } if obj.business_type else None

    def get_secondary_business_types(self, obj):
        return [
            {
                'id': bt.id,
                'name': bt.name,
                'slug': bt.slug
            } for bt in obj.secondary_business_types.all()
        ]

    def get_subscription_status(self, obj):
        return {
            'type': obj.subscription_type,
            'is_active': obj.get_active_subscription(),
            'expires_at': obj.subscription_expires,
            'can_manage_events': obj.can_manage_events()
        }

    def get_verification_details(self, obj):
        return {
            'status': obj.verification_status,
            'verified_at': obj.verified_at,
            'is_verified': obj.is_verified(),
            'documents_count': obj.documents.count() if hasattr(obj, 'documents') else 0
        }

    def get_categories(self, obj):
        return [
            {
                'id': category.id,
                'name': category.name,
                'slug': category.slug
            } for category in obj.categories.all()
        ]

    def get_operating_cities(self, obj):
        return [
            {
                'id': city.id,
                'name': city.name,
            } for city in obj.cities.all()
        ]

    def get_sites_count(self, obj):
        return obj.get_managed_sites().count()

    def to_representation(self, instance):
        data = super().to_representation(instance)
        
        # Add business analytics if requested
        request = self.context.get('request')
        if request and request.query_params.get('include_analytics'):
            data['analytics'] = instance.get_business_analytics()
            
        return data

class BusinessDetailsSerializer(serializers.ModelSerializer):
    tax_id = serializers.CharField(validators=[VendorValidators.tax_id_pattern])
    website = serializers.URLField(required=False)
    founded_year = serializers.IntegerField(validators=[VendorValidators.validate_founded_year])
    
    class Meta:
        model = Vendor
        fields = ['business_type', 'tax_id', 'website', 'description', 
                 'founded_year', 'primary_address']

    def validate_website(self, value):
        url_validator = URLValidator()
        try:
            url_validator(value)
        except ValidationError:
            raise serializers.ValidationError('Enter a valid URL')
        return value
 
class SubscriptionSerializer(serializers.ModelSerializer):
    plan_details = SubscriptionPlanSerializer(source='plan', read_only=True)
    status_display = serializers.SerializerMethodField()
    
    class Meta:
        model = VendorSubscription
        fields = [
            'id', 'vendor', 'plan', 'plan_details', 
            'start_date', 'end_date', 'status', 'status_display',
            'auto_renew', 'next_billing_date', 'created_at'
        ]
        read_only_fields = ['status', 'created_at']
    
    def get_status_display(self, obj):
        return obj.get_status_display()

class SubscriptionActionSerializer(serializers.Serializer):
    plan_id = serializers.IntegerField(required=True)
    payment_method_id = serializers.CharField(required=False)
    coupon_code = serializers.CharField(required=False, allow_blank=True)
    auto_renew = serializers.BooleanField(default=True)
    
    def validate_plan_id(self, value):
        try:
            plan = SubscriptionPlan.objects.get(id=value, is_active=True)
            return value
        except SubscriptionPlan.DoesNotExist:
            raise serializers.ValidationError(_("Invalid or inactive subscription plan"))

class SubscriptionCancellationSerializer(serializers.Serializer):
    reason = serializers.CharField(required=True)
    feedback = serializers.CharField(required=False, allow_blank=True)
    cancel_immediately = serializers.BooleanField(default=False)
    
    REASON_CHOICES = [
        'too_expensive',
        'not_using',
        'missing_features',
        'technical_issues',
        'poor_support',
        'other'
    ]
    
    def validate_reason(self, value):
        if value not in self.REASON_CHOICES:
            raise serializers.ValidationError(_("Invalid cancellation reason"))
        return value

from rest_framework import serializers
from django.core.validators import FileExtensionValidator

class BusinessVerificationSerializer(serializers.Serializer):
    document_type = serializers.ChoiceField(
        choices=[
            ('business_license', _('Business License')),
            ('tax_id', _('Tax ID Document')),
            ('proof_of_address', _('Proof of Address')),
            ('id_document', _('ID Document')),
            ('other', _('Other Document'))
        ]
    )
    description = serializers.CharField(required=False, max_length=255)
    documents = serializers.ListField(
        child=serializers.FileField(
            validators=[
                FileExtensionValidator(
                    allowed_extensions=['pdf', 'jpg', 'jpeg', 'png']
                )
            ],
            max_length=10485760  # 10MB max
        ),
        min_length=1,
        max_length=5
    )

    def validate(self, data):
        if not data.get('documents'):
            raise serializers.ValidationError(
                _("At least one document is required for verification")
            )
        
        total_size = sum(doc.size for doc in data['documents'])
        if total_size > 52428800:  # 50MB total
            raise serializers.ValidationError(
                _("Total document size cannot exceed 50MB")
            )
        
        return data
 
class SubscriptionPlanDetailSerializer(serializers.ModelSerializer):
    """Detailed serializer for subscription plans"""
    features = serializers.SerializerMethodField()
    
    class Meta:
        model = SubscriptionPlan
        fields = [
            'id', 'name', 'description', 'price', 'currency',
            'billing_period', 'features', 'max_sites', 'max_events',
            'is_featured', 'is_active'
        ]
    
    def get_features(self, obj):
        return [
            {
                'name': feature.name,
                'description': feature.description,
                'is_available': feature.is_available
            }
            for feature in obj.features.all()
        ]

class PaymentMethodSerializer(serializers.ModelSerializer):
    """Serializer for payment methods"""
    class Meta:
        model = PaymentMethod
        fields = [
            'id', 'type', 'last_four', 'expiry_date',
            'is_default', 'is_expired', 'brand'
        ]

class SubscriptionDetailSerializer(serializers.ModelSerializer):
    """Detailed serializer for subscriptions with all related information"""
    plan = SubscriptionPlanDetailSerializer()
    payment_method = PaymentMethodSerializer()
    status_display = serializers.SerializerMethodField()
    days_remaining = serializers.SerializerMethodField()
    usage_statistics = serializers.SerializerMethodField()
    billing_history = serializers.SerializerMethodField()
    
    class Meta:
        model = VendorSubscription
        fields = [
            'id', 'vendor', 'plan', 'status', 'status_display',
            'start_date', 'end_date', 'days_remaining',
            'auto_renew', 'payment_method', 'next_billing_date',
            'usage_statistics', 'billing_history',
            'is_cancelled', 'cancellation_date',
            'created_at', 'updated_at'
        ]
    
    def get_status_display(self, obj):
        return obj.get_status_display()
    
    def get_days_remaining(self, obj):
        if obj.end_date:
            delta = obj.end_date - timezone.now()
            return max(0, delta.days)
        return None
    
    def get_usage_statistics(self, obj):
        # Get latest usage report
        usage_report = obj.usage_reports.first()
        if usage_report:
            return {
                'sites': {
                    'used': usage_report.sites_count,
                    'total': obj.plan.max_sites,
                    'percentage': usage_report.sites_quota_percentage
                },
                'events': {
                    'used': usage_report.events_count,
                    'total': obj.plan.max_events,
                    'percentage': usage_report.events_quota_percentage
                }
            }
        return None
    
    def get_billing_history(self, obj):
        return [{
            'date': transaction.created_at,
            'amount': transaction.amount,
            'currency': transaction.currency,
            'status': transaction.status,
            'invoice_url': transaction.get_invoice_url()
        } for transaction in obj.transactions.all()[:5]]  # Last 5 transactions

class SubscriptionChangeSerializer(serializers.Serializer):
    """Serializer for subscription plan changes"""
    plan_id = serializers.IntegerField(required=True)
    immediate = serializers.BooleanField(default=False)
    payment_method_id = serializers.CharField(required=False)
    proration_date = serializers.DateTimeField(required=False)
    
    def validate_plan_id(self, value):
        try:
            plan = SubscriptionPlan.objects.get(id=value, is_active=True)
            
            # Get current subscription
            current_subscription = self.context['request'].user.vendor_profile.get_active_subscription()
            if current_subscription and current_subscription.plan.id == value:
                raise serializers.ValidationError(
                    _("You are already subscribed to this plan")
                )
            
            # Check if plan change is allowed
            if current_subscription and not current_subscription.can_change_to_plan(plan):
                raise serializers.ValidationError(
                    _("Cannot change to this plan. Please contact support.")
                )
                
            return value
        except SubscriptionPlan.DoesNotExist:
            raise serializers.ValidationError(_("Invalid or inactive subscription plan"))
    
    def validate_payment_method_id(self, value):
        if value:
            try:
                payment_method = PaymentMethod.objects.get(
                    id=value,
                    vendor__user=self.context['request'].user,
                    is_active=True
                )
                if payment_method.is_expired:
                    raise serializers.ValidationError(
                        _("This payment method has expired")
                    )
                return value
            except PaymentMethod.DoesNotExist:
                raise serializers.ValidationError(
                    _("Invalid or inactive payment method")
                )
        return value
    
    def validate(self, data):
        # Additional validation logic
        plan = SubscriptionPlan.objects.get(id=data['plan_id'])
        current_subscription = self.context['request'].user.vendor_profile.get_active_subscription()
        
        # Check if payment method is required
        if plan.price > 0 and not data.get('payment_method_id') and not current_subscription:
            raise serializers.ValidationError(
                _("Payment method is required for paid plans")
            )
        
        # Check if immediate change is allowed
        if data.get('immediate') and current_subscription and not current_subscription.can_change_immediately:
            raise serializers.ValidationError(
                _("Immediate plan change is not allowed for your current subscription")
            )
        
        return data

from rest_framework import serializers

class VendorRegistrationSerializer(serializers.ModelSerializer):
    class Meta:
        model = Vendor
        fields = [
            'company_name', 'business_type', 'contact_email',
            'contact_phone', 'website', 'description'
        ]
 
class VendorOnboardingSerializer(serializers.ModelSerializer):
    onboarding_progress = serializers.SerializerMethodField()
    
    class Meta:
        model = Vendor
        fields = [
            'id', 'onboarding_progress', 'verification_status',
            'business_type', 'company_name', 'contact_email',
            'contact_phone'  # Only including fields that exist in your Vendor model
        ]
    
    def get_onboarding_progress(self, obj):
        # Calculate onboarding progress based on completed steps
        steps_completed = 0
        total_steps = 4  # Adjust based on your actual number of onboarding steps
        
        if obj.company_name:
            steps_completed += 1
        if obj.business_type:
            steps_completed += 1
        if obj.documents.exists():
            steps_completed += 1
        if obj.primary_address:
            steps_completed += 1
            
        return {
            'completed_steps': steps_completed,
            'total_steps': total_steps,
            'percentage': (steps_completed / total_steps) * 100
        }

class VendorSiteSerializer(serializers.ModelSerializer):
    city_name = serializers.CharField(source='city.name', read_only=True)
    vendor_name = serializers.CharField(source='vendor.company_name', read_only=True)
    created_by_name = serializers.CharField(source='created_by.get_full_name', read_only=True)
    
    class Meta:
        model = Site
        fields = [
            'id', 
            'title',
            'description',
            'type',
            'status',
            'address',
            'city',
            'city_name',
            'vendor',
            'vendor_name',
            'created_by',
            'created_by_name',
            'reviewed_by',
            'active_subscription',
            'images',
            'tags',
            'created_at',
            'updated_at'
        ]
        read_only_fields = [
            'vendor',
            'created_by',
            'reviewed_by',
            'active_subscription',
            'created_at',
            'updated_at'
        ]

    def validate(self, data):
        """
        Custom validation for the site data
        """
        # Example validation rules
        if data.get('type') not in ['place', 'event', 'route']:
            raise serializers.ValidationError("Invalid site type")
            
        if not data.get('address'):
            raise serializers.ValidationError("Address is required")
            
        if not data.get('city'):
            raise serializers.ValidationError("City is required")
            
        return data

    def create(self, validated_data):
        """
        Custom create method to handle related fields
        """
        images_data = self.context['request'].FILES.getlist('images', [])
        tags_data = validated_data.pop('tags', [])
        
        site = Site.objects.create(**validated_data)
        
        # Handle images
        for image_data in images_data:
            SiteImage.objects.create(site=site, image=image_data)
            
        # Handle tags
        if tags_data:
            site.tags.set(tags_data)
            
        return site

    def update(self, instance, validated_data):
        """
        Custom update method to handle related fields
        """
        images_data = self.context['request'].FILES.getlist('images', [])
        tags_data = validated_data.pop('tags', None)
        
        # Update basic fields
        for attr, value in validated_data.items():
            setattr(instance, attr, value)
        instance.save()
        
        # Handle images if new ones are provided
        if images_data:
            instance.images.all().delete()  # Remove old images
            for image_data in images_data:
                SiteImage.objects.create(site=instance, image=image_data)
                
        # Handle tags if provided
        if tags_data is not None:
            instance.tags.set(tags_data)
            
        return instance

 

class DocumentVerificationStatus:
    PENDING = 'pending'
    PARTIAL = 'partial'
    COMPLETE = 'complete'
    REJECTED = 'rejected'

class DocumentVerificationSerializer(serializers.ModelSerializer):
    skip_verification = serializers.BooleanField(required=False, default=False)
    document_status = serializers.CharField(read_only=True)
    verification_notes = serializers.CharField(required=False, allow_blank=True)
    
    class Meta:
        model = VendorDocument
        fields = [
            'id', 'name', 'file', 'document_type',
            'uploaded_at', 'verified', 'skip_verification',
            'document_status', 'verification_notes'
        ]
        extra_kwargs = {
            'file': {'required': False},
            'document_type': {'required': True}
        }

    def validate(self, data):
        if not data.get('file') and not data.get('skip_verification'):
            raise serializers.ValidationError({
                'file': 'Either file must be provided or skip_verification must be true'
            })
        return data

class DocumentVerificationViewSet(viewsets.ModelViewSet):
    """
    ViewSet for vendor document verification with support for partial submissions
    """
    permission_classes = [IsAuthenticated]
    serializer_class = DocumentVerificationSerializer

    def get_queryset(self):
        return VendorDocument.objects.filter(
            vendor__user=self.request.user
        ).select_related('vendor')

    @action(detail=False, methods=['POST'])
    def submit_documents(self, request):
        """
        Step 3: Upload verification documents with support for partial submission
        """
        try:
            with transaction.atomic():
                vendor = request.user.vendor_profile
                documents_data = request.data if isinstance(request.data, list) else [request.data]
                response_data = []
                verification_status = DocumentVerificationStatus.COMPLETE

                for doc_data in documents_data:
                    serializer = self.get_serializer(data=doc_data)
                    if serializer.is_valid():
                        # Handle skipped verification
                        if serializer.validated_data.get('skip_verification'):
                            doc = VendorDocument.objects.create(
                                vendor=vendor,
                                document_type=serializer.validated_data['document_type'],
                                name=serializer.validated_data.get('name', 'Pending Document'),
                                verification_notes='Document submission pending',
                                document_status=DocumentVerificationStatus.PENDING
                            )
                            verification_status = DocumentVerificationStatus.PARTIAL
                        else:
                            # Handle actual document upload
                            doc = VendorDocument.objects.create(
                                vendor=vendor,
                                file=serializer.validated_data['file'],
                                document_type=serializer.validated_data['document_type'],
                                name=serializer.validated_data.get('name'),
                                document_status=DocumentVerificationStatus.PENDING
                            )
                        
                        response_data.append(self.get_serializer(doc).data)
                    else:
                        raise serializers.ValidationError(serializer.errors)

                # Update vendor verification status
                vendor.verification_status = verification_status
                vendor.verification_updated_at = timezone.now()
                vendor.save(update_fields=['verification_status', 'verification_updated_at'])

                return Response({
                    'status': 'success',
                    'verification_status': verification_status,
                    'documents': response_data,
                    'next_step': 'subscription_selection' if verification_status == DocumentVerificationStatus.COMPLETE else 'pending_documents',
                    'message': 'Documents submitted successfully' if verification_status == DocumentVerificationStatus.COMPLETE else 'Documents partially submitted'
                })

        except serializers.ValidationError as e:
            return Response({
                'status': 'error',
                'errors': e.detail
            }, status=status.HTTP_400_BAD_REQUEST)
        
        except Exception as e:
            logger.error(f"Document submission error: {str(e)}")
            return Response({
                'status': 'error',
                'message': 'An error occurred while processing your request'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    @action(detail=False, methods=['GET'])
    def verification_status(self, request):
        """
        Get the current verification status and pending documents
        """
        vendor = request.user.vendor_profile
        documents = self.get_queryset()
        
        pending_documents = documents.filter(document_status=DocumentVerificationStatus.PENDING)
        complete_documents = documents.filter(document_status=DocumentVerificationStatus.COMPLETE)
        
        return Response({
            'status': vendor.verification_status,
            'pending_documents': self.get_serializer(pending_documents, many=True).data,
            'complete_documents': self.get_serializer(complete_documents, many=True).data,
            'last_updated': vendor.verification_updated_at
        })

    @action(detail=True, methods=['PUT'])
    def update_document(self, request, pk=None):
        """
        Update a pending document with actual file
        """
        try:
            document = self.get_queryset().get(pk=pk)
            
            if document.document_status != DocumentVerificationStatus.PENDING:
                return Response({
                    'status': 'error',
                    'message': 'Only pending documents can be updated'
                }, status=status.HTTP_400_BAD_REQUEST)

            serializer = self.get_serializer(document, data=request.data, partial=True)
            if serializer.is_valid():
                serializer.save(
                    document_status=DocumentVerificationStatus.COMPLETE,
                    verification_notes='Document updated successfully'
                )
                
                # Check if all documents are complete
                vendor = request.user.vendor_profile
                if not vendor.documents.filter(document_status=DocumentVerificationStatus.PENDING).exists():
                    vendor.verification_status = DocumentVerificationStatus.COMPLETE
                    vendor.save(update_fields=['verification_status'])

                return Response({
                    'status': 'success',
                    'document': serializer.data
                })
            
            return Response({
                'status': 'error',
                'errors': serializer.errors
            }, status=status.HTTP_400_BAD_REQUEST)

        except VendorDocument.DoesNotExist:
            return Response({
                'status': 'error',
                'message': 'Document not found'
            }, status=status.HTTP_404_NOT_FOUND)
