# local_secrets/users/services/analytics_service.py

from django.db.models import Count, Avg, Sum, F, ExpressionWrapper, FloatField
from django.db.models.functions import TruncDate, TruncMonth
from django.utils import timezone
from datetime import timedelta

class SubscriptionAnalytics:
    @staticmethod
    def get_subscription_overview(date_from=None, date_to=None):
        """Get overview of subscription metrics"""
        if not date_from:
            date_from = timezone.now() - timedelta(days=30)
        if not date_to:
            date_to = timezone.now()

        return {
            'active_subscriptions': VendorSubscription.objects.filter(
                status='active'
            ).count(),
            'revenue': SubscriptionTransaction.objects.filter(
                status='completed',
                created_at__range=(date_from, date_to)
            ).aggregate(total=Sum('amount'))['total'] or 0,
            'avg_subscription_duration': VendorSubscription.objects.filter(
                status='active'
            ).annotate(
                duration=ExpressionWrapper(
                    F('end_date') - F('start_date'),
                    output_field=FloatField()
                )
            ).aggregate(avg=Avg('duration'))['avg'],
            'churn_rate': self.calculate_churn_rate(date_from, date_to)
        }

    @staticmethod
    def get_subscription_trends(days=30):
        """Get subscription trends over time"""
        date_from = timezone.now() - timedelta(days=days)
        
        return {
            'new_subscriptions': VendorSubscription.objects.filter(
                created_at__gte=date_from
            ).annotate(
                date=TruncDate('created_at')
            ).values('date').annotate(
                count=Count('id')
            ).order_by('date'),
            
            'cancellations': VendorSubscription.objects.filter(
                cancelled_at__gte=date_from
            ).annotate(
                date=TruncDate('cancelled_at')
            ).values('date').annotate(
                count=Count('id')
            ).order_by('date'),
            
            'revenue_trend': SubscriptionTransaction.objects.filter(
                created_at__gte=date_from,
                status='completed'
            ).annotate(
                date=TruncDate('created_at')
            ).values('date').annotate(
                total=Sum('amount')
            ).order_by('date')
        }

    @staticmethod
    def get_plan_performance():
        """Analyze performance of different subscription plans"""
        return SubscriptionPlan.objects.annotate(
            active_subscriptions=Count(
                'vendorsubscription',
                filter=Q(vendorsubscription__status='active')
            ),
            total_revenue=Sum(
                'vendorsubscription__transactions__amount',
                filter=Q(vendorsubscription__transactions__status='completed')
            ),
            avg_duration=Avg(
                ExpressionWrapper(
                    F('vendorsubscription__end_date') - F('vendorsubscription__start_date'),
                    output_field=FloatField()
                )
            )
        ).values('name', 'active_subscriptions', 'total_revenue', 'avg_duration')

    @staticmethod
    def get_usage_statistics():
        """Get subscription usage statistics"""
        return {
            'sites_usage': SubscriptionUsageReport.objects.filter(
                report_date=timezone.now().date() - timedelta(days=1)
            ).aggregate(
                avg_usage=Avg('sites_quota_percentage'),
                high_usage_count=Count(
                    'id',
                    filter=Q(sites_quota_percentage__gte=90)
                )
            ),
            
            'events_usage': SubscriptionUsageReport.objects.filter(
                report_date=timezone.now().date() - timedelta(days=1)
            ).aggregate(
                avg_usage=Avg('events_quota_percentage'),
                high_usage_count=Count(
                    'id',
                    filter=Q(events_quota_percentage__gte=90)
                )
            )
        }

    @staticmethod
    def calculate_churn_rate(date_from, date_to):
        """Calculate subscription churn rate"""
        start_active = VendorSubscription.objects.filter(
            status='active',
            start_date__lt=date_from
        ).count()
        
        churned = VendorSubscription.objects.filter(
            status='cancelled',
            cancelled_at__range=(date_from, date_to)
        ).count()
        
        return (churned / start_active * 100) if start_active > 0 else 0
