# local_secrets/users/services/claim_services.py

from django.db import transaction
from django.utils import timezone
from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _
from ..emails import VendorEmailNotifications
from local_secrets.users.models import Vendor, SiteClaim, CustomUser
from local_secrets.sites.models import Site


class ClaimService:
    @staticmethod
    @transaction.atomic
    def create_claim(site, vendor, documents=None):
        """Create a new site claim"""
        # Check if site is already claimed
        if SiteClaim.objects.filter(site=site, status='APPROVED').exists():
            raise ValidationError(_("This site has already been claimed"))

        # Check if vendor has pending claim for this site
        if SiteClaim.objects.filter(
            site=site,
            vendor=vendor,
            status='PENDING'
        ).exists():
            raise ValidationError(_("You already have a pending claim for this site"))

        claim = SiteClaim.objects.create(
            site=site,
            vendor=vendor,
            status='PENDING'
        )

        if documents:
            claim.proof_documents.set(documents)

        return claim

    @staticmethod
    @transaction.atomic
    def process_claim(claim_id, admin_user, approve=False, notes=None):
        """Process a site claim"""
        try:
            claim = SiteClaim.objects.get(id=claim_id)
        except SiteClaim.DoesNotExist:
            raise ValidationError(_("Claim not found"))

        if claim.status != 'PENDING':
            raise ValidationError(_("This claim has already been processed"))

        claim.status = 'APPROVED' if approve else 'REJECTED'
        claim.processed_at = timezone.now()
        claim.processed_by = admin_user
        claim.notes = notes or ''
        claim.save()

        if approve:
            # Update site ownership
            claim.site.vendor = claim.vendor
            claim.site.save()

            # Reject other pending claims for this site
            SiteClaim.objects.filter(
                site=claim.site,
                status='PENDING'
            ).exclude(id=claim.id).update(
                status='REJECTED',
                processed_at=timezone.now(),
                processed_by=admin_user,
                notes=_("Site was claimed by another vendor")
            )

        return claim

    @staticmethod
    def get_vendor_claims(vendor):
        """Get all claims for a vendor"""
        return SiteClaim.objects.filter(vendor=vendor)

    @staticmethod
    def get_pending_claims():
        """Get all pending claims"""
        return SiteClaim.objects.filter(status='PENDING')
