# local_secrets/users/services/notification_service.py
from django.conf import settings
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.utils.translation import gettext as _
from local_secrets.sites.models import Site
from local_secrets.users.models import Notification, CustomUser, Vendor

class NotificationService:
    """
    Manages system notifications
    """
    def notify_site_activation(self, site: Site):
        """
        Sends notifications for site activation
        """
        # Create system notification
        notification = Notification.objects.create(
            title=_("Site Activated"),
            body=_("Your site '%(title)s' is now active") % {'title': site.title},
            site=site
        )

        # Notify vendor
        self._notify_vendor(
            site.vendor,
            notification,
            'site_activation'
        )

        # Notify admins if needed
        self._notify_admins(
            site,
            'site_activation',
            extra_context={'vendor': site.vendor}
        )

    def _notify_vendor(self, vendor: Vendor, notification: Notification, template: str):
        """
        Sends notification to vendor
        """
        context = {
            'vendor': vendor,
            'notification': notification,
            'site_url': self._get_site_url(notification.site)
        }

        # Send email
        self._send_email(
            template_name=f'users/vendor/{template}',
            context=context,
            recipient_email=vendor.email
        )

        # Send push notification if device ID exists
        if vendor.device_id:
            self._send_push_notification(
                vendor.device_id,
                notification.title,
                notification.body
            )

    def _notify_admins(self, site: Site, template: str, extra_context: dict = None):
        """
        Sends notification to admin users
        """
        admin_users = CustomUser.objects.filter(is_staff=True)
        
        context = {
            'site': site,
            'admin_url': self._get_admin_url(site),
            **(extra_context or {})
        }

        for admin in admin_users:
            self._send_email(
                template_name=f'users/admin/{template}',
                context=context,
                recipient_email=admin.email
            )

    def _send_email(self, template_name: str, context: dict, recipient_email: str):
        """
        Sends email notification
        """
        html_content = render_to_string(f'{template_name}.html', context)
        text_content = render_to_string(f'{template_name}.txt', context)

        send_mail(
            subject=context.get('notification', {}).get('title', 'Notification'),
            message=text_content,
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=[recipient_email],
            html_message=html_content
        )

    def _send_push_notification(self, device_id: str, title: str, body: str):
        """
        Sends push notification
        """
        # Implement push notification logic here
        pass

    def _get_site_url(self, site: Site) -> str:
        """
        Gets site URL
        """
        return f"{settings.FRONTEND_URL}/sites/{site.id}"

    def _get_admin_url(self, site: Site) -> str:
        """
        Gets admin URL for site
        """
        return f"{settings.ADMIN_URL}/sites/site/{site.id}/change/"
