# local_secrets/users/services/quota_notification_service.py

from functools import cache
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.conf import settings
from django.utils import timezone
from typing import Dict, Any

class QuotaNotificationService:
    WARNING_THRESHOLDS = {
        'high': 90,
        'medium': 75,
        'low': 50
    }

    def __init__(self, subscription):
        self.subscription = subscription
        self.vendor = subscription.vendor

    def check_and_notify(self, resource_type: str, usage: int, limit: int) -> None:
        """
        Checks usage against thresholds and sends notifications if needed
        """
        percentage = (usage / limit * 100) if limit > 0 else 100
        
        # Check if we've already notified for this level today
        cache_key = f"quota_notification_{self.subscription.id}_{resource_type}_{percentage}"
        if cache.get(cache_key):
            return

        for level, threshold in self.WARNING_THRESHOLDS.items():
            if percentage >= threshold:
                self._send_notification(
                    resource_type,
                    usage,
                    limit,
                    percentage,
                    level
                )
                # Cache the notification for 24 hours
                cache.set(cache_key, True, 86400)
                break

    def _send_notification(
        self,
        resource_type: str,
        usage: int,
        limit: int,
        percentage: float,
        level: str
    ) -> None:
        """
        Sends quota warning notification
        """
        context = {
            'vendor_name': self.vendor.company_name,
            'resource_type': resource_type,
            'usage': usage,
            'limit': limit,
            'percentage': percentage,
            'level': level,
            'subscription_name': self.subscription.plan.name
        }

        subject = self._get_notification_subject(resource_type, percentage, level)
        message = render_to_string('users/emails/quota_warning.html', context)

        send_mail(
            subject,
            message,
            settings.DEFAULT_FROM_EMAIL,
            [self.vendor.user.email],
            html_message=message
        )

        # Log notification
        self.subscription.audit_logs.create(
            action='quota_warning',
            details={
                'resource_type': resource_type,
                'usage': usage,
                'limit': limit,
                'percentage': percentage,
                'level': level
            }
        )

    def _get_notification_subject(
        self,
        resource_type: str,
        percentage: float,
        level: str
    ) -> str:
        """
        Generates notification subject based on context
        """
        return f"[{level.upper()}] {percentage:.0f}% of your {resource_type} quota used"
