# local_secrets/users/tasks.py (Celery tasks)
from celery import shared_task
from local_secrets.users.models import Vendor
from .services.email_service import VendorEmailService 
import logging
from django.core.mail import send_mail 
from django.conf import settings
from django.utils.translation import gettext_lazy as _
from django.core.mail import send_mail
from django.template.loader import render_to_string
from django.utils.html import strip_tags
from django.contrib.auth import get_user_model
from django.db import transaction
from django.utils import timezone

logger = logging.getLogger(__name__)
User = get_user_model()

@shared_task
def send_welcome_email(vendor_id):
    try:
        vendor = Vendor.objects.get(id=vendor_id)
        VendorEmailService.send_welcome_email(vendor)
    except Exception as e:
        logger.error(f"Failed to send welcome email to vendor {vendor_id}: {str(e)}")

@shared_task
def send_verification_update_email(vendor_id, status, rejection_reason=None):
    try:
        vendor = Vendor.objects.get(id=vendor_id)
        VendorEmailService.send_verification_update(vendor, status, rejection_reason)
    except Exception as e:
        logger.error(f"Failed to send verification update email to vendor {vendor_id}: {str(e)}")
 
User = get_user_model()

@shared_task
def notify_admins(notification_type, **kwargs):
    """
    Notify admin users about important vendor actions
    
    Args:
        notification_type (str): Type of notification (e.g., 'vendor_verification')
        **kwargs: Additional data needed for the notification
    """
    admin_emails = User.objects.filter(is_staff=True, is_active=True).values_list('email', flat=True)
    
    if notification_type == 'vendor_verification':
        subject = f"New Vendor Verification Request: {kwargs.get('user_company_name')}"
        
        # Render email template with context
        html_message = render_to_string('emails/admin/vendor_verification.html', {
            'vendor_id': kwargs.get('vendor_id'),
            'documents': kwargs.get('documents', []),
            'user_name': kwargs.get('user_name'),
            'user_email': kwargs.get('user_email'),
            'user_phone': kwargs.get('user_phone'),
            'company_name': kwargs.get('user_company_name'),
            'business_type': kwargs.get('user_business_type'),
            'admin_url': f"{settings.ADMIN_SITE_URL}/vendors/{kwargs.get('vendor_id')}/verify"
        })
        
        # Send email to all admins
        send_mail(
            subject=subject,
            message=strip_tags(html_message),  # Plain text version
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=list(admin_emails),
            html_message=html_message,
            fail_silently=False
        )
 
    elif notification_type == 'subscription_change':
        # Handle subscription change notifications
        pass
    
    elif notification_type == 'vendor_report':
        # Handle vendor reporting notifications
        pass
    
 