# validators.py
import os
from django.core.validators import RegexValidator, URLValidator
from django.core.exceptions import ValidationError
from django.utils.translation import gettext_lazy as _

class VendorValidators:
    # Common validation patterns
    tax_id_pattern = RegexValidator(
        regex=r'^[0-9A-Z]{8,15}$',
        message=_('Tax ID must be 8-15 characters of numbers and uppercase letters')
    )
    
    phone_pattern = RegexValidator(
        regex=r'^\+?1?\d{9,15}$',
        message=_('Phone number must be entered in the format: "+999999999".')
    )
    
    @staticmethod
    def validate_company_name(value):
        if len(value.strip()) < 3:
            raise ValidationError(_('Company name must be at least 3 characters long'))
        if len(value) > 255:
            raise ValidationError(_('Company name cannot exceed 255 characters'))

    @staticmethod
    def validate_business_type(value):
        from .models import BusinessType
        if not BusinessType.objects.filter(id=value).exists():
            raise ValidationError(_('Invalid business type selected'))

    @staticmethod
    def validate_founded_year(value):
        from django.utils import timezone
        current_year = timezone.now().year
        if value > current_year:
            raise ValidationError(_('Founded year cannot be in the future'))
        if value < 1800:
            raise ValidationError(_('Founded year cannot be before 1800'))

    @staticmethod
    def validate_document_files(files):
        allowed_extensions = ['.pdf', '.jpg', '.jpeg', '.png']
        max_file_size = 5 * 1024 * 1024  # 5MB

        for file in files:
            # Check file extension
            ext = os.path.splitext(file.name)[1].lower()
            if ext not in allowed_extensions:
                raise ValidationError(
                    _('Invalid file type. Allowed types: PDF, JPG, JPEG, PNG')
                )
            
            # Check file size
            if file.size > max_file_size:
                raise ValidationError(_('File size cannot exceed 5MB'))

