# local_secrets/users/views.py
import csv
import io
from pyexpat.errors import messages
import random
import string
from datetime import timedelta, timezone
from itertools import chain
from operator import attrgetter
from venv import logger
from django.conf import settings
from django.contrib.auth.models import Group
from django.core.cache import cache
from django.core.files.images import get_image_dimensions
from django.db.models.functions import Random
from django.db import IntegrityError, transaction
from django.http import HttpResponse
from django.shortcuts import get_object_or_404
from django.urls import reverse, reverse_lazy
from django.utils import translation
from django.utils.timezone import now
from django.utils.translation import gettext_lazy as _
from oauth2_provider.models import Application, get_access_token_model, get_refresh_token_model
from rest_framework import viewsets, mixins, filters, status, serializers
from rest_framework.decorators import action
from rest_framework.exceptions import ValidationError
from rest_framework.pagination import PageNumberPagination
from rest_framework.permissions import AllowAny, IsAuthenticated
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.views import APIView
from rest_framework.viewsets import ViewSet
from rest_framework.request import Request
from rest_framework.response import Response
from rest_framework.viewsets import GenericViewSet
from local_secrets.cities.models import Address
from local_secrets.languages.models import Language
from local_secrets.payment.exceptions import PaymentServiceError
from local_secrets.payment.services import PaymentService
from local_secrets.routes.models import Route
from local_secrets.routes.serializers import RoutesAndSitesSerializer
from local_secrets.sites.models import ImageSize, Site
from local_secrets.users.models import CustomUser, Notification, SubscriptionAuditLog, Tag, VendorDocument
from local_secrets.users.serializers import (
    EmailSerializer,
    GroupSerializer,
    NotificationDetailSerializer,
    NotificationSerializer,
    ResetPasswordSerializer,
    TagOutputSerializer,
    TagSelectionInputSerializer,
    UserInputSerializer,
    UserOutputSerializer,
    VendorRegistrationSerializer,
    VendorListSerializer,

)
from django_filters import rest_framework as filters
 
from local_secrets.users.models import (
    Vendor, VendorSubscription, SubscriptionPlan, VendorContact, VendorReview,
    SiteClaim, SubscriptionTransaction, BusinessCategory, BusinessType, SubscriptionUsageReport, VendorVerificationStatus, 
)
 
from local_secrets.users.services.analytics_service import SubscriptionAnalytics
from local_secrets.users.services.bulk_operations import BulkOperationsService
from local_secrets.users.services.subscription_services import SubscriptionService
from local_secrets.users.services.claim_services import ClaimService
from local_secrets.users.tasks import notify_admins, send_welcome_email
from .serializers import (
    BusinessDetailsSerializer, BusinessVerificationSerializer, DocumentVerificationSerializer, SubscriptionActionSerializer, SubscriptionCancellationSerializer, SubscriptionChangeSerializer, SubscriptionDetailSerializer, SubscriptionPlanSerializer, SubscriptionSerializer, VendorDocumentSerializer, VendorOnboardingSerializer, VendorSiteSerializer, VendorSubscriptionSerializer,
    SiteClaimSerializer, SubscriptionTransactionSerializer,
    BusinessCategorySerializer, BusinessTypeSerializer, VendorSerializer,
    VendorListSerializer, VendorContactSerializer, VendorReviewSerializer,
    VendorStatusSerializer
)
from local_secrets.payment.models import Payment, PaymentMethod 
from local_secrets.users.permissions import IsVendorOwner, HasActiveSubscription, HasAvailableQuota, IsAuthenticated
from django.contrib.auth.decorators import login_required
from django.shortcuts import redirect
from django.views.generic.edit import FormView, UpdateView
from django.views.generic import TemplateView, ListView
from django.contrib.auth.mixins import LoginRequiredMixin
from .forms import VendorProfileForm, VendorRegistrationForm
from rest_framework import viewsets, status, filters 
from rest_framework.permissions import IsAdminUser, IsAuthenticated 
from local_secrets.events.models import Event 
from django.contrib.auth.models import Permission
from django.utils.decorators import method_decorator
from django.views.decorators.csrf import csrf_exempt
from django.core.mail import send_mail
from rest_framework.parsers import MultiPartParser, FormParser
from .serializers import VendorSubscriptionSerializer 
from .tasks import send_welcome_email
import logging

logger = logging.getLogger(__name__)

class UserViewSet(
    GenericViewSet,
):
    permission_classes = (IsAuthenticated,)

    def get_queryset(self):
        return CustomUser.objects.all()

    def get_serializer_class(self, *args, **kwargs):
        if self.action in ['post', 'update', 'patch', 'update_me']:
            return UserInputSerializer
        return UserOutputSerializer

    def get_serializer_context(self):
        return {'request': self.request}

    @action(detail=False, methods=['get'], permission_classes=(IsAuthenticated,))
    def me(self, request):
        user = CustomUser.objects.all().annotate_travel_variables().get(id=request.user.id)
        return Response(UserOutputSerializer(user, context={'request': request}).data)

    @action(detail=False, methods=['post'], permission_classes=(IsAuthenticated,))
    def logout(self, request):
        get_access_token_model().objects.filter(user=request.user).delete()
        get_refresh_token_model().objects.filter(user=request.user).delete()
        user = CustomUser.objects.get(id=request.user.id)
        user.device_id = None
        user.save()
        if request.data.get('create_token'):
            allowed_chars = ''.join((string.ascii_letters, string.digits))

            access_unique_id = ''.join(random.choice(allowed_chars) for _ in range(30))
            access_token = get_access_token_model().objects.create(
                user=request.user, expires=now() + timedelta(minutes=1), token=access_unique_id, scope="read write"
            )
            print(access_token)

            refresh_unique_id = ''.join(random.choice(allowed_chars) for _ in range(30))
            new_refresh_token = get_refresh_token_model().objects.create(
                user=request.user,
                access_token=access_token,
                application=Application.objects.first(),
                token=refresh_unique_id,
            )
            print(new_refresh_token)
            return Response({'detail': new_refresh_token.token})

        return Response({'detail': _('The access and refresh tokens have been deleted successfully')})

    @action(detail=False, methods=['post'], permission_classes=(IsAuthenticated,))
    def reset_password(self, request):
        serializer = ResetPasswordSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        if not request.user.check_password(serializer.validated_data['old_password']):
            raise ValidationError(code=400, detail='The given password does not match the old password')
        print(request.user)
        request.user.set_password(serializer.validated_data.get('new_password'))
        request.user.save()
        return Response({'detail': 'Password reset successfully'})

    @action(detail=False, methods=['get'], permission_classes=(IsAuthenticated,))
    def preferences(self, request):
        tags = Tag.objects.annotate_is_selected(request.user)
        return Response(TagOutputSerializer(tags, many=True, context={'user': request.user, 'request': request}).data)

    @action(detail=False, methods=['post'], permission_classes=(IsAuthenticated,))
    def select_preferences(self, request):
        serializer = TagSelectionInputSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        request.user.update_tags(serializer.validated_data.get('tags'))
        return Response({'detail': 'Preferences updated'}, status=201)

    @me.mapping.delete
    def delete_me(self, request: Request):
        request.user.delete()
        return Response({'detail': 'The user was deleted'}, status=200)

    @me.mapping.patch
    @me.mapping.put
    def update_me(self, request: Request):
        user = CustomUser.objects.all().annotate_travel_variables().get(id=request.user.id)
        serializer = self.get_serializer(user, data=request.data, context={'request': request}, partial=True)
        serializer.is_valid(raise_exception=True)
        serializer.save()
        if 'password' in serializer.validated_data:
            user.set_password(serializer.validated_data.get('password'))
            user.save()

        return Response(UserOutputSerializer(user, context={'request': request}).data)

    #  Get All notifications
    @action(detail=False, methods=['get'])
    def notifications(self, request):
        #  Define Pagination
        paginator = PageNumberPagination()
        paginator.page_size = 10
        paginate = request.query_params.get('page')

        #  Get Notifications, order by created_at desc
        notifications = Notification.objects.all().annotate_has_been_seen(request.user).order_by('-created_at')

        #  If Page Query param exists
        if paginate:
            paginated_queryset = paginator.paginate_queryset(notifications, request)
            output = NotificationSerializer(paginated_queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        else:
            return Response(NotificationSerializer(notifications, many=True).data)

    #  Mark notifications as seen
    @action(detail=False, methods=['post'])
    def notification(self, request):
        # Get user
        user = CustomUser.objects.get(id=request.user.id)
        #  Get Notification
        notification = Notification.objects.get(id=request.data.get('id'))
        #  Mark as seen for request user
        notification.mark_as_seen(user)
        #  Get a fresh instance from db
        notification.refresh_from_db()
        #  Return Notification Detail
        return Response(NotificationDetailSerializer(notification).data)

    # Check if user has notifications that has_been_seen = False
    @action(detail=False, methods=['get'])
    def has_notifications(self, request):
        #  Check has_been_seen=False
        user = CustomUser.objects.get(id=request.user.id)
        _has_notifications = user.notifications.count() < Notification.objects.count()
        if not _has_notifications:
            _has_notifications = user.notifications.filter(has_been_seen=False).count() > 0
        return Response({'detail': _has_notifications})

    @action(detail=False, methods=['post'])
    def update_pfp(self, request):
        try:
            file = request.FILES.getlist('profile_picture')
            if not file:
                raise ValidationError(detail='You need to pass at least one image', code=400)
                
            img_sizes = ImageSize.objects.first()
            if not img_sizes:
                img_sizes = ImageSize.objects.create(
                    min_width=512,
                    min_height=512,
                    max_width=4096,
                    max_height=2160
                )
                
            width, height = get_image_dimensions(file[0])
            
            # Validation checks
            if not (img_sizes.min_width <= width <= img_sizes.max_width and 
                    img_sizes.min_height <= height <= img_sizes.max_height):
                raise ValidationError(
                    detail=f'Image must be between {img_sizes.min_width}x{img_sizes.min_height} '
                    f'and {img_sizes.max_width}x{img_sizes.max_height} pixels',
                    code=400
                )
                
            request.user.update_pfp(file)
            return Response({'detail': _('Profile picture updated')})
            
        except Exception as e:
            return Response(
                {'detail': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )
 
    @action(detail=False, methods=['post'])
    def language(self, request):
        user = CustomUser.objects.get(id=request.user.id)
        user.language = Language.objects.get(id=request.data.get('lang_id'))
        user.save()
        return Response({'detail': _('Language updated')}, status=200)

    @action(detail=False, methods=['get'])
    def groups(self, request):
        groups = Group.objects.all()
        return Response(GroupSerializer(groups, context={'request': request}, many=True).data)

    def validate_vendor_registration(self, data):
        if Vendor.objects.filter(user=self.request.user).exists():
            raise ValidationError("User already has a LS Manager profile")
        
        if not BusinessType.objects.filter(id=data['business_type']).exists():
            raise ValidationError("Invalid business type")
        
        return data

class UserCreationViewSet(viewsets.GenericViewSet, mixins.CreateModelMixin):
    permission_classes = (AllowAny,)
    serializer_class = UserInputSerializer   

    def get_serializer_class(self):
        """
        Return different serializers based on the action
        """
        if self.action == 'restore_password':
            return EmailSerializer
        elif self.action == 'register_as_vendor':
            return VendorRegistrationSerializer
        elif self.action == 'vendor_status':
            return VendorStatusSerializer
        return UserInputSerializer
 
    def create(self, request, *args, **kwargs):
        with translation.override(request.headers.get('language')):
            serializer = UserInputSerializer(data=request.data, context={'request': request})
            serializer.is_valid(raise_exception=True)
            try:
                # Create user
                user = CustomUser.objects.create_user(**serializer.validated_data)
                user = CustomUser.objects.all().annotate_travel_variables().get(id=user.id)
                
                # Create default group if it doesn't exist
                default_group, created = Group.objects.get_or_create(
                    name='Viajero',
                    defaults={'name': 'Viajero'}
                )
                
                # Add user to default group
                user.groups.add(default_group)
                
                output = UserOutputSerializer(user, context={'request': request})
                return Response(output.data, status=201)
            except Exception as e:
                return Response({'detail': str(e)}, status=400)
 
    @action(detail=False, methods=['post'])
    def restore_password(self, request):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        try:
            user = get_object_or_404(CustomUser, email=serializer.validated_data.get('email'))
            user.restore_password()
            return Response({'detail': 'Check your email inbox to restore your password'})
        except Exception as e:
            return Response(
                {'detail': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )


    @action(detail=False, methods=['POST'])
    def register_as_vendor(self, request):
        """Convert existing user to vendor or register new vendor"""
        if request.user.is_authenticated:
            # Path 1: Converting existing user
            return self._convert_to_vendor(request)
        else:
            # Path 2: New vendor registration
            return self._register_new_vendor(request)


    def _convert_to_vendor(self, request):
        if hasattr(request.user, 'vendor_profile'):
            return Response({
                'error': 'User already has a vendor profile'
            }, status=400)
            
        serializer = VendorRegistrationSerializer(data=request.data)
        if serializer.is_valid():
            vendor = Vendor.objects.create(
                user=request.user,
                company_name=serializer.validated_data['company_name'],
                contact_email=request.user.email
            )
            return Response({
                'vendor_id': vendor.id,
                'next_step': 'business_details'
            })
        return Response(serializer.errors, status=400)


    def _register_new_vendor(self, request):
        user_serializer = UserInputSerializer(data=request.data)
        vendor_serializer = VendorRegistrationSerializer(data=request.data)
        
        if user_serializer.is_valid() and vendor_serializer.is_valid():
            with transaction.atomic():
                user = CustomUser.objects.create_user(
                    **user_serializer.validated_data
                )
                vendor = Vendor.objects.create(
                    user=user,
                    **vendor_serializer.validated_data
                )
                return Response({
                    'user_id': user.id,
                    'vendor_id': vendor.id,
                    'next_step': 'business_details'
                })
        return Response({
            'user_errors': user_serializer.errors,
            'vendor_errors': vendor_serializer.errors
        }, status=400)


    @action(detail=False, methods=['get'])
    def vendor_status(self, request):
        if not request.user.is_authenticated:
            return Response(
                {'detail': 'You need to be logged in'},
                status=status.HTTP_401_UNAUTHORIZED
            )

        try:
            vendor = Vendor.objects.get(user=request.user)
            return Response(self.get_serializer(vendor).data)
        except Vendor.DoesNotExist:
            return Response(self._get_default_vendor_status())

    def _create_vendor_profile(self, user, validated_data):
        vendor = Vendor.create(
            user=user,
            company_name=validated_data.get('company_name'),
            business_type=validated_data.get('business_type'),
            contact_email=validated_data.get('contact_email'),
            description=validated_data.get('description'),
        )

        # Create Syspay account
        syspay_response = SubscriptionService.create_syspay_account(vendor=vendor)
        if syspay_response['status'] == 'success':
            vendor.syspay_user_id = syspay_response['user_id']
            vendor.save()
        else:
            raise ValidationError(syspay_response['message'])

        return vendor

    def _setup_vendor_services(self, user, vendor):
        # Create Vendor Group and permissions
        group, created = Group.objects.get_or_create(name='Vendor')
        if created:
            group.permissions.set(Permission.objects.filter(codename__startswith='vendor_'))
        user.groups.add(group)

        # Initialize subscription
        subscription = self._initialize_subscription(vendor)
        
        # Create related services
        self._create_subscription_services(subscription, user)

    def _initialize_subscription(self, vendor):
        return SubscriptionService.create_subscription(
            vendor=vendor,
            plan=SubscriptionPlan.objects.get(name='Free'),
            payment_method=None,
        )

    def _create_subscription_services(self, subscription, user):
        # Create Transaction
        SubscriptionTransaction.objects.create(
            vendor_subscription=subscription,
            transaction_type='FREE',
            amount=0,
            currency='USD',
            status='COMPLETED',
            payment_gateway_reference='FREE',
            created_at=now(),
            completed_at=now(),
        )

        # Create Usage Report
        SubscriptionUsageReport.objects.create(
            subscription=subscription,
            sites_count=0,
            events_count=0,
            sites_quota_percentage=0,
            events_quota_percentage=0,
            created_at=now(),
            updated_at=now(),
        )

        # Create Audit Log
        SubscriptionService.create_audit_log(
            subscription=subscription,
            action='CREATE',
            details='LS Manager registered successfully',
            created_by=user,
        )

        # Create Payment Method
        SubscriptionService.create_payment_method(
            vendor=subscription.vendor,
            payment_type='FREE',
            is_default=True,
            is_verified=True,
        )

        # Create Automatic Billing
        SubscriptionService.create_automatic_billing(
            vendor=subscription.vendor,
            enabled=False,
            billing_day=1,
            next_billing_date=now() + timedelta(days=30),
            payment_method=None,
            syspay_subscription_id=None,
            retry_count=0,
        )

    def _get_default_vendor_status(self):
        return {
            'is_vendor': False,
            'is_verified': False,
            'is_featured': False,
            'is_active': False,
            'is_subscribed': False,
            'is_free': False,
            'is_trial': False,
            'is_expired': False,
            'is_approved': False,
            'is_claimed': False,
            'is_claimed_by': None,
            'is_claimed_at': None,
            'is_claimed_by_user': None,
            'is_claimed_by_vendor': None,
            'is_claimed_by_vendor_id': None,
            'is_claimed_by_vendor_name': None,
            'is_claimed_by_vendor_email': None,
        }
 
class TagViewSet(GenericViewSet):
    permission_classes = (AllowAny,)

    def get_queryset(self):
        return Tag.objects.all()

    def get_serializer_class(self):
        return TagOutputSerializer

    @action(detail=True, methods=['get'])
    def routes_and_sites(self, request, pk):
        #  Define Pagination
        cache_key = f'routes_and_sites_{pk}_{request.query_params.get("city_id")}'
        paginator = PageNumberPagination()
        paginator.page_size = 10
        paginate = request.query_params.get('page')

        tag = self.get_object()
        if cache.get(cache_key):
            result_list = cache.get(cache_key)
        else:
            routes = (
                Route.objects.filter(tags__id=tag.id).filter_by_city_id(request.query_params.get('city_id')).distinct()#.annotate(order=Random() * 100)
            ).distinct()

            sites = (
                tag.site_tags.all()
                .filter_city_id(request.query_params.get('city_id'))
                #.annotate(order=Random() * 100)
                .annotate_is_fav(self.request.user)
            ).distinct()

            result_list = list(chain(sites, routes))
            cache.set(cache_key, result_list)

        #  If Page Query param exists
        if paginate:
            paginated_queryset = paginator.paginate_queryset(result_list, request)
            output = RoutesAndSitesSerializer(paginated_queryset, context={'request': request}, many=True).data
            return paginator.get_paginated_response(output)
        else:
            output = RoutesAndSitesSerializer(result_list, context={'request': request}, many=True).data
            return Response(output)
 
class SubscriptionPlanViewSet(mixins.ListModelMixin,
                            mixins.RetrieveModelMixin,
                            GenericViewSet):
    queryset = SubscriptionPlan.objects.filter(is_active=True)
    serializer_class = SubscriptionPlanSerializer
    permission_classes = [IsAuthenticated]
 
class VendorSubscriptionViewSet(mixins.CreateModelMixin, mixins.RetrieveModelMixin, mixins.ListModelMixin, mixins.UpdateModelMixin, viewsets.GenericViewSet):
    """
    ViewSet for managing vendor subscriptions.
    Provides endpoints for creating, retrieving, updating, and managing subscriptions.
    """
    permission_classes = [IsAuthenticated, IsVendorOwner]
    serializer_class = VendorSubscriptionSerializer  # Add this line
    
    def get_serializer_class(self):
        """
        Return different serializers based on the action
        """
        if self.action in ['subscribe', 'change_plan']:
            return SubscriptionActionSerializer
        elif self.action == 'cancel':
            return SubscriptionCancellationSerializer
        return self.serializer_class

    def get_queryset(self):
        return VendorSubscription.objects.filter(
            vendor__user=self.request.user
        ).select_related('plan', 'vendor')

    @action(detail=False, methods=['post'])
    def subscribe(self, request):
        """
        Create a new subscription for the vendor
        """
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        with transaction.atomic():
            try:
                # Get vendor instance
                vendor = request.user.vendor_profile
                if not vendor:
                    raise ValidationError(_("User does not have a vendor profile"))

                # Get plan instance
                plan = SubscriptionPlan.objects.get(
                    id=serializer.validated_data['plan_id'],
                    is_active=True
                )

                # Validate current subscription
                current_subscription = vendor.active_subscription
                if current_subscription and current_subscription.status == 'active':
                    if current_subscription.plan.id == plan.id:
                        raise ValidationError(_("Already subscribed to this plan"))

                # Get payment method
                payment_method = PaymentMethod.objects.get(
                    id=serializer.validated_data['payment_method_id'],
                    vendor=vendor,
                    is_active=True
                )

                # Create new subscription
                subscription = SubscriptionService.create_subscription(
                    vendor=vendor,
                    plan=plan,
                    payment_method=payment_method
                )

                # Process payment
                payment_result = self._process_subscription_payment(
                    subscription=subscription,
                    payment_method=payment_method
                )

                if not payment_result.success:
                    raise ValidationError(payment_result.error)

                return Response(
                    VendorSubscriptionSerializer(subscription).data,
                    status=status.HTTP_201_CREATED
                )

            except Exception as e:
                transaction.set_rollback(True)
                raise ValidationError(str(e))

    @action(detail=True, methods=['post'])
    def change_plan(self, request, pk=None):
        """
        Change subscription plan (upgrade/downgrade)
        """
        subscription = self.get_object()
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        try:
            new_plan = SubscriptionPlan.objects.get(
                id=serializer.validated_data['plan_id'],
                is_active=True
            )

            updated_subscription = SubscriptionService.change_plan(
                subscription=subscription,
                new_plan=new_plan,
                user=request.user
            )

            return Response(
                VendorSubscriptionSerializer(updated_subscription).data
            )

        except (SubscriptionPlan.DoesNotExist, ValidationError) as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )

    @action(detail=True, methods=['post'])
    def cancel(self, request, pk=None):
        """
        Cancel subscription
        """
        subscription = self.get_object()
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        try:
            cancelled_subscription = SubscriptionService.cancel_subscription(
                subscription=subscription,
                user=request.user,
                reason=serializer.validated_data.get('reason')
            )

            return Response(
                VendorSubscriptionSerializer(cancelled_subscription).data
            )

        except ValidationError as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )

    @action(detail=True, methods=['get'])
    def audit_logs(self, request, pk=None):
        """
        Get subscription audit logs
        """
        subscription = self.get_object()
        logs = subscription.audit_logs.all()[:100]

        return Response({
            'logs': SubscriptionAuditLogSerializer(logs, many=True).data
        })

    def _process_subscription_payment(self, subscription, payment_method):
        """
        Process the payment for a subscription using PaymentService
        """
        try:
            payment_service = PaymentService(
                gateway=payment_method.get_gateway_interface()
            )

            with transaction.atomic():
                # Create and process payment
                payment = payment_service.create_payment(
                    vendor=subscription.vendor,
                    amount=subscription.plan.price,
                    currency=subscription.plan.currency,
                    payment_method=payment_method,
                    metadata={
                        'subscription_id': subscription.id,
                        'plan_id': subscription.plan.id,
                        'payment_type': 'subscription'
                    }
                )

                # Update subscription based on payment status
                if payment.status == 'completed':
                    subscription.status = 'active'
                    subscription.payment_id = payment.id
                    subscription.save()

                    # Create subscription audit log
                    SubscriptionAuditLog.objects.create(
                        subscription=subscription,
                        action='payment_success',
                        metadata={
                            'payment_id': payment.id,
                            'amount': str(payment.amount),
                            'currency': payment.currency
                        }
                    )

                    return {
                        'success': True,
                        'payment_id': payment.id,
                        'status': payment.status,
                        'transaction_id': payment.payment_gateway_reference
                    }
                else:
                    subscription.status = 'payment_failed'
                    subscription.save()

                    # Create subscription audit log
                    SubscriptionAuditLog.objects.create(
                        subscription=subscription,
                        action='payment_failed',
                        metadata={
                            'payment_id': payment.id,
                            'error': payment.response_data.get('error')
                        }
                    )

                    return {
                        'success': False,
                        'error': payment.response_data.get('error', 'Payment processing failed')
                    }

        except PaymentServiceError as e:
            logger.error(
                "Subscription payment processing error",
                extra={
                    'subscription_id': subscription.id,
                    'payment_method_id': payment_method.id,
                    'error': str(e)
                },
                exc_info=True
            )
            
            subscription.status = 'payment_failed'
            subscription.save()

            return {
                'success': False,
                'error': str(e)
            }
        except Exception as e:
            logger.error(
                "Unexpected error during subscription payment processing",
                extra={
                    'subscription_id': subscription.id,
                    'payment_method_id': payment_method.id,
                    'error': str(e)
                },
                exc_info=True
            )
            
            subscription.status = 'payment_failed'
            subscription.save()

            return {
                'success': False,
                'error': 'An unexpected error occurred during payment processing'
            }


class VendorViewSet(GenericViewSet):
    permission_classes = [IsAuthenticated]
    
    def get_permissions(self):
        if self.action in ['update', 'partial_update', 'destroy']:
            return [IsAuthenticated(), IsVendorOwner()]
        return [IsAuthenticated()]

    @action(detail=False, methods=['GET'])
    def vendor_status(self, request):
        """
        Get comprehensive vendor status including:
        - Basic information
        - Business type and categories
        - Subscription status
        - Verification status
        - Operating locations
        - Site statistics
        """
        try:
            vendor = request.user.vendor_profile
        except Vendor.DoesNotExist:
            return Response(
                {'detail': 'Vendor profile not found'}, 
                status=status.HTTP_404_NOT_FOUND
            )

        serializer = VendorStatusSerializer(
            vendor, 
            context={'request': request}
        )
        return Response(serializer.data)

    @action(detail=False, methods=['POST'])
    def verify_business(self, request):
        """Handle business verification request"""
        vendor = request.user.vendor_profile
        serializer = BusinessVerificationSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        documents = request.FILES.getlist('documents')
        if not documents:
            raise ValidationError("Business verification documents required")

        # Create documents
        for doc in documents:
            VendorDocument.objects.create(
                vendor=vendor,
                document=doc,
                document_type=serializer.validated_data.get('document_type')
            )

        # Update vendor status
        vendor.verification_status = VendorVerificationStatus.PENDING
        vendor.save()

        # Notify admins
        notify_admins.delay(
            notification_type='vendor_verification',
            vendor_id=vendor.id,
            documents=[doc.name for doc in documents],
            user_id=request.user.id,
            user_email=request.user.email,
            user_name=request.user.get_full_name(),
            user_phone=request.user.phone_number,
            user_company_name=vendor.company_name,
            user_business_type=vendor.business_type.name if vendor.business_type else None,
        )

        return Response({
            'detail': 'Business verification documents submitted successfully',
            'status': vendor.verification_status
        })
 
class SiteClaimViewSet(mixins.CreateModelMixin, mixins.RetrieveModelMixin, mixins.ListModelMixin, GenericViewSet):
    permission_classes = [IsAuthenticated, HasActiveSubscription, HasAvailableQuota]
    
    def get_queryset(self):
        user = self.request.user
        if user.is_staff:
            return SiteClaim.objects.all().select_related(
                'site', 'vendor', 'processed_by'
            )
        return SiteClaim.objects.filter(
            vendor__user=user
        ).select_related('site', 'vendor', 'processed_by')

    def perform_create(self, serializer):
        serializer.save(vendor=self.request.user.vendor_profile)

    @action(detail=True, methods=['post'])
    def process_claim(self, request, pk=None):
        if not request.user.is_staff:
            return Response(
                {'error': _("You don't have permission to process claims")},
                status=status.HTTP_403_FORBIDDEN
            )

        claim = self.get_object()
        approve = request.data.get('approve', False)
        notes = request.data.get('notes', '')

        try:
            processed_claim = ClaimService.process_claim(
                claim_id=claim.id,
                admin_user=request.user,
                approve=approve,
                notes=notes
            )
            serializer = self.get_serializer(processed_claim)
            return Response(serializer.data)
        except Exception as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )

class SubscriptionAnalyticsViewSet(mixins.ListModelMixin,mixins.RetrieveModelMixin,GenericViewSet):
    permission_classes = [IsAuthenticated, IsVendorOwner]
    
    def get_queryset(self):
        return SubscriptionUsageReport.objects.filter(
            subscription__vendor__user=self.request.user
        ).select_related('subscription')
    
    @action(detail=False, methods=['get'])
    def summary(self, request):
        """Get subscription usage summary"""
        subscription = request.user.vendor_profile.subscriptions.filter(
            status='active'
        ).first()
        
        if not subscription:
            return Response(
                {'error': 'No active subscription found'},
                status=status.HTTP_404_NOT_FOUND
            )
        
        # Get latest report
        latest_report = subscription.usage_reports.first()
        
        if not latest_report:
            return Response(
                {'error': 'No usage data available'},
                status=status.HTTP_404_NOT_FOUND
            )
        
        return Response({
            'sites': {
                'used': latest_report.sites_count,
                'total': subscription.plan.max_sites,
                'percentage': latest_report.sites_quota_percentage
            },
            'events': {
                'used': latest_report.events_count,
                'total': subscription.plan.max_events,
                'percentage': latest_report.events_quota_percentage
            },
            'subscription_ends': subscription.end_date,
            'auto_renew': subscription.auto_renew
        })

class BusinessCategoryViewSet(GenericViewSet, mixins.ListModelMixin):
    queryset = BusinessCategory.objects.all()
    serializer_class = BusinessCategorySerializer
    filter_backends = [filters.SearchFilter, filters.OrderingFilter]
    search_fields = ['name', 'slug']
    ordering_fields = ['name', 'order']

class BusinessTypeViewSet(GenericViewSet, mixins.ListModelMixin):
    queryset = BusinessType.objects.all()
    serializer_class = BusinessTypeSerializer
    filter_backends = [filters.SearchFilter, filters.OrderingFilter]
    search_fields = ['name', 'slug', 'category__name']
    ordering_fields = ['name', 'order']
    filterset_fields = ['category', 'service_level', 'target_audience', 'operating_season']

class VendorContactViewSet(viewsets.ModelViewSet):
    queryset = VendorContact.objects.all()
    serializer_class = VendorContactSerializer
    #filter_backends = [DjangoFilterBackend]
    filterset_fields = ['vendor', 'is_primary']

#Flow Control and Progress Tracking

class VendorSignupProgressManager:
    STEPS = {
        'registration': {
            'order': 1,
            'required': True,
            'next': 'business_details'
        },
        'business_details': {
            'order': 2,
            'required': True,
            'next': 'document_verification'
        },
        'document_verification': {
            'order': 3,
            'required': False,
            'next': 'subscription_selection'
        },
        'subscription_selection': {
            'order': 4,
            'required': False,
            'next': 'payment'
        },
        'payment': {
            'order': 5,
            'required': False,  # Not required for free trial
            'next': 'activation'
        },
        'activation': {
            'order': 6,
            'required': True,
            'next': 'onboarding'
        },
        'onboarding': {
            'order': 7,
            'required': False,
            'next': None
        }
    }

    @classmethod
    def get_current_step(cls, vendor):
        """Get vendor's current signup step"""
        if not vendor.user:
            return 'registration'
        if not vendor.business_type:
            return 'business_details'
        if vendor.verification_status == 'PENDING':
            return 'document_verification'
        if not vendor.subscription_type:
            return 'subscription_selection'
        if not vendor.get_active_subscription():
            return 'payment'
        return 'completed'
 
class VendorRegistrationViewSet(viewsets.ModelViewSet):
    """
    ViewSet for vendor registration process
    """
    permission_classes = [IsAuthenticated]
    serializer_class = VendorRegistrationSerializer

    def get_queryset(self):
        return Vendor.objects.filter(user=self.request.user)
    
    @action(detail=False, methods=['POST'])
    def register(self, request):
        """Step 1: Basic user account creation"""
        serializer = VendorRegistrationSerializer(data=request.data)
        if serializer.is_valid():
            with transaction.atomic():
                # Create user account
                user = CustomUser.objects.create_user(
                    email=serializer.validated_data['email'],
                    password=serializer.validated_data['password'],
                    first_name=serializer.validated_data['first_name'],
                    last_name=serializer.validated_data['last_name']
                )
                
                # Initialize vendor profile
                vendor = Vendor.objects.create(
                    user=user,
                    company_name=serializer.validated_data['company_name'],
                    contact_email=serializer.validated_data['email']
                )
                
                return Response({
                    'user_id': user.id,
                    'vendor_id': vendor.id,
                    'next_step': 'business_details'
                }, status=201)
        return Response(serializer.errors, status=400)

class BusinessDetailsViewSet(viewsets.ModelViewSet):
    """
    ViewSet for managing vendor business details
    """
    permission_classes = [IsAuthenticated]
    serializer_class = BusinessDetailsSerializer

    def get_queryset(self):
        return Vendor.objects.filter(user=self.request.user).select_related(
            'business_type',
            'primary_address'
        ).prefetch_related(
            'secondary_business_types',
            'categories'
        )
    
    @action(detail=False, methods=['POST'])
    def submit_details(self, request):
        """Step 2: Collect business information"""
        serializer = BusinessDetailsSerializer(data=request.data)
        if serializer.is_valid():
            vendor = request.user.vendor_profile
            vendor.business_type = serializer.validated_data['business_type']
            vendor.description = serializer.validated_data['description']
            vendor.tax_id = serializer.validated_data['tax_id']
            vendor.website = serializer.validated_data['website']
            vendor.founded_year = serializer.validated_data['founded_year']
            vendor.save()
            
            # Handle address
            address = Address.objects.create(
                **serializer.validated_data['address']
            )
            vendor.primary_address = address
            vendor.save()
            
            return Response({
                'status': 'completed',
                'next_step': 'document_verification'
            })
        return Response(serializer.errors, status=400)


class DocumentVerificationViewSet(viewsets.ModelViewSet):
    """
    ViewSet for vendor document verification with support for partial submissions
    """
    permission_classes = [IsAuthenticated]
    serializer_class = DocumentVerificationSerializer

    def get_queryset(self):
        return VendorDocument.objects.filter(
            vendor__user=self.request.user
        ).select_related('vendor')

    @action(detail=False, methods=['POST'])
    def submit_documents(self, request):
        """
        Step 3: Upload verification documents with support for partial submission
        """
        try:
            with transaction.atomic():
                vendor = request.user.vendor_profile
                documents_data = request.data if isinstance(request.data, list) else [request.data]
                response_data = []
                all_verified = True

                for doc_data in documents_data:
                    serializer = self.get_serializer(data=doc_data)
                    if serializer.is_valid():
                        # Handle skipped verification
                        if serializer.validated_data.get('skip_verification'):
                            doc = VendorDocument.objects.create(
                                vendor=vendor,
                                document_type=serializer.validated_data['document_type'],
                                name=serializer.validated_data.get('name', 'Pending Document'),
                                verified=False
                            )
                            all_verified = False
                        else:
                            # Handle actual document upload
                            doc = VendorDocument.objects.create(
                                vendor=vendor,
                                file=serializer.validated_data['file'],
                                document_type=serializer.validated_data['document_type'],
                                name=serializer.validated_data.get('name'),
                                verified=False
                            )
                        
                        response_data.append(self.get_serializer(doc).data)
                    else:
                        raise serializers.ValidationError(serializer.errors)

                # Update vendor verification status
                vendor.verification_status = (
                    VendorVerificationStatus.VERIFIED if all_verified 
                    else VendorVerificationStatus.PENDING
                )
                vendor.save()

                return Response({
                    'status': 'success',
                    'verification_status': vendor.verification_status,
                    'documents': response_data,
                    'next_step': 'subscription_selection' if all_verified else 'pending_documents',
                    'message': 'Documents submitted successfully' if all_verified else 'Documents partially submitted'
                })

        except serializers.ValidationError as e:
            return Response({
                'status': 'error',
                'errors': e.detail
            }, status=status.HTTP_400_BAD_REQUEST)
        
        except Exception as e:
            logger.error(f"Document submission error: {str(e)}")
            return Response({
                'status': 'error',
                'message': 'An error occurred while processing your request'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    @action(detail=False, methods=['GET'])
    def verification_status(self, request):
        """
        Get the current verification status and pending documents
        """
        vendor = request.user.vendor_profile
        documents = self.get_queryset()
        
        pending_documents = documents.filter(verified=False)
        verified_documents = documents.filter(verified=True)
        
        return Response({
            'status': vendor.verification_status,
            'pending_documents': self.get_serializer(pending_documents, many=True).data,
            'verified_documents': self.get_serializer(verified_documents, many=True).data,
            'last_updated': vendor.updated_at if hasattr(vendor, 'updated_at') else None
        })

    @action(detail=True, methods=['PUT'])
    def update_document(self, request, pk=None):
        """
        Update a pending document with actual file
        """
        try:
            document = self.get_queryset().get(pk=pk)
            
            if document.verified:
                return Response({
                    'status': 'error',
                    'message': 'Cannot update verified documents'
                }, status=status.HTTP_400_BAD_REQUEST)

            serializer = self.get_serializer(document, data=request.data, partial=True)
            if serializer.is_valid():
                serializer.save()
                
                # Check if all documents are verified
                vendor = request.user.vendor_profile
                if not vendor.documents.filter(verified=False).exists():
                    vendor.verification_status = VendorVerificationStatus.VERIFIED
                    vendor.save()

                return Response({
                    'status': 'success',
                    'document': serializer.data
                })
            
            return Response({
                'status': 'error',
                'errors': serializer.errors
            }, status=status.HTTP_400_BAD_REQUEST)

        except VendorDocument.DoesNotExist:
            return Response({
                'status': 'error',
                'message': 'Document not found'
            }, status=status.HTTP_404_NOT_FOUND)

class SubscriptionSelectionViewSet(viewsets.ModelViewSet):
    """
    ViewSet for subscription plan selection
    """
    permission_classes = [IsAuthenticated]
    serializer_class = SubscriptionPlanSerializer

    def get_queryset(self):
        return SubscriptionPlan.objects.filter(
            is_active=True
        ) 
    
    @action(detail=False, methods=['GET'])
    def available_plans(self, request):
        """Step 4: Show available subscription plans"""
        plans = SubscriptionPlan.objects.filter(is_active=True)
        return Response({
            'plans': SubscriptionPlanSerializer(plans, many=True).data,
            'free_trial_available': self.check_free_trial_eligibility(request.user)
        })

    @action(detail=False, methods=['POST'])
    def select_plan(self, request):
        """Step 4.1: Select and process subscription"""
        plan = get_object_or_404(
            SubscriptionPlan, 
            id=request.data.get('plan_id')
        )
        
        if plan.is_free_trial:
            return self.activate_free_trial(request.user.vendor_profile, plan)
        else:
            return self.initiate_payment(request.user.vendor_profile, plan)

class SubscriptionPaymentViewSet(viewsets.ModelViewSet):
    @action(detail=False, methods=['POST'])
    def process_payment(self, request):
        """Step 5: Handle payment for selected plan"""
        vendor = request.user.vendor_profile
        plan = get_object_or_404(
            SubscriptionPlan, 
            id=request.data.get('plan_id')
        )
        
        try:
            # Create payment session
            payment = Payment.objects.create(
                variant='stripe',  # or your payment provider
                amount=plan.price,
                currency=plan.currency,
                description=f"Subscription: {plan.name}",
                billing_email=vendor.contact_email,
                extra_data={
                    'vendor_id': vendor.id,
                    'plan_id': plan.id
                }
            )
            
            return Response({
                'payment_url': payment.get_payment_url(),
                'payment_id': payment.id
            })
            
        except Exception as e:
            return Response({
                'error': str(e)
            }, status=400)

 
class AccountActivationViewSet(viewsets.ModelViewSet):
    permission_classes = [IsAuthenticated]
    serializer_class = VendorSubscriptionSerializer
    
    def get_queryset(self):
        return VendorSubscription.objects.filter(
            vendor__user=self.request.user
        ).select_related(
            'vendor',
            'plan'
        )

    @action(detail=False, methods=['POST'])
    def activate_account(self, request):
        """Step 6: Complete account activation"""
        vendor = request.user.vendor_profile
        payment_id = request.data.get('payment_id')
        
        try:
            with transaction.atomic():
                # Verify payment exists
                try:
                    payment = Payment.objects.get(
                        id=payment_id,
                        vendor=vendor
                    )
                except Payment.DoesNotExist:
                    raise ValidationError('Invalid payment ID')

                # Check payment status
                if payment.status != 'confirmed':
                    raise ValidationError('Payment not confirmed')
                
                # Verify plan exists
                try:
                    plan = SubscriptionPlan.objects.get(
                        id=payment.extra_data.get('plan_id')
                    )
                except SubscriptionPlan.DoesNotExist:
                    raise ValidationError('Invalid subscription plan')

                # Cancel any existing subscriptions
                VendorSubscription.objects.filter(
                    vendor=vendor,
                    status='active'
                ).update(
                    status='cancelled',
                    cancelled_at=timezone.now()
                )

                # Create new subscription
                subscription = VendorSubscription.objects.create(
                    vendor=vendor,
                    plan=plan,
                    payment=payment,
                    start_date=timezone.now(),
                    end_date=timezone.now() + timedelta(days=plan.duration_days),
                    status='active',
                    auto_renew=request.data.get('auto_renew', False)
                )

                # Update vendor status
                vendor.subscription_type = plan.type
                vendor.subscription_expires = subscription.end_date
                vendor.status = 'active'
                vendor.save(update_fields=[
                    'subscription_type',
                    'subscription_expires',
                    'status'
                ])

                # Schedule welcome email
                try:
                    task = send_welcome_email.delay(vendor.id)
                    # Store task id for potential cancellation
                    subscription.email_task_id = task.id
                    subscription.save(update_fields=['email_task_id'])
                except Exception as e:
                    # Log email scheduling error but don't halt activation
                    logger.error(f"Failed to schedule welcome email: {str(e)}")

                return Response({
                    'status': 'activated',
                    'subscription': self.get_serializer(subscription).data,
                    'next_step': 'dashboard'
                })

        except ValidationError as e:
            return Response({
                'error': str(e)
            }, status=status.HTTP_400_BAD_REQUEST)
        except Exception as e:
            # Log unexpected errors
            logger.error(f"Account activation failed: {str(e)}")
            return Response({
                'error': 'Account activation failed. Please try again.'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    @action(detail=False, methods=['POST'])
    def cancel_activation(self, request):
        """Cancel ongoing account activation process"""
        vendor = request.user.vendor_profile
        
        try:
            with transaction.atomic():
                # Cancel any pending subscriptions
                pending_subs = VendorSubscription.objects.filter(
                    vendor=vendor,
                    status='pending'
                )
                
                for sub in pending_subs:
                    # Send cancellation notification email
                    try:
                        send_mail(
                            subject='Subscription Activation Cancelled',
                            message=f'Your subscription activation has been cancelled.',
                            from_email=settings.DEFAULT_FROM_EMAIL,
                            recipient_list=[vendor.user.email],
                            fail_silently=True
                        )
                    except Exception as e:
                        logger.warning(f"Failed to send cancellation email: {str(e)}")
                    
                    sub.status = 'cancelled'
                    sub.cancelled_at = timezone.now()
                    sub.cancellation_reason = request.data.get('reason', 'User cancelled')
                    sub.save(update_fields=['status', 'cancelled_at', 'cancellation_reason'])

                # Update vendor status if needed
                if vendor.status == 'pending_activation':
                    vendor.status = 'inactive'
                    vendor.save(update_fields=['status'])

                return Response({
                    'status': 'cancelled',
                    'message': 'Activation process cancelled successfully'
                })

        except Exception as e:
            logger.error(f"Failed to cancel activation: {str(e)}")
            return Response({
                'error': 'Failed to cancel activation'
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

class VendorOnboardingViewSet(viewsets.ModelViewSet):
    """
    ViewSet for vendor onboarding process
    """
    permission_classes = [IsAuthenticated]
    serializer_class = VendorOnboardingSerializer

    def get_queryset(self):
        return Vendor.objects.filter(
            user=self.request.user
        ).select_related(
            'business_type',
            'primary_address',
            'automatic_billing'
        ).prefetch_related(
            'documents',
            'payment_methods'
        )
    
    @action(detail=False, methods=['GET'])
    def onboarding_steps(self, request):
        """Step 7: Show onboarding steps"""
        vendor = request.user.vendor_profile
        return Response({
            'completed_steps': self.get_completed_steps(vendor),
            'next_steps': self.get_next_steps(vendor),
            'quick_links': {
                'add_site': reverse('api:sites-create'),
                'manage_profile': reverse('api:vendor-profile'),
                'view_analytics': reverse('api:vendor-analytics')
            }
        })
 
class VendorReviewViewSet(viewsets.ModelViewSet):
    queryset = VendorReview.objects.all()
    serializer_class = VendorReviewSerializer
    filter_backends = [filters.OrderingFilter]
    filterset_fields = ['vendor', 'rating']
    ordering_fields = ['created_at', 'rating']

    def get_queryset(self):
        return super().get_queryset().select_related('vendor', 'user')
 
class AnalyticsViewSet(viewsets.ViewSet):
    permission_classes = [IsAdminUser]

    @action(detail=False, methods=['get'])
    def subscription_overview(self, request):
        days = int(request.query_params.get('days', 30))
        overview = SubscriptionAnalytics.get_subscription_overview(
            date_from=timezone.now() - timedelta(days=days)
        )
        return Response(overview)

    @action(detail=False, methods=['get'])
    def subscription_trends(self, request):
        days = int(request.query_params.get('days', 30))
        trends = SubscriptionAnalytics.get_subscription_trends(days=days)
        return Response(trends)

    @action(detail=False, methods=['get'])
    def plan_performance(self, request):
        performance = SubscriptionAnalytics.get_plan_performance()
        return Response(performance)

    @action(detail=False, methods=['get'])
    def usage_statistics(self, request):
        statistics = SubscriptionAnalytics.get_usage_statistics()
        return Response(statistics)

class BulkOperationsViewSet(viewsets.ViewSet):
    permission_classes = [IsAdminUser]

    @action(detail=False, methods=['post'])
    def process_sites(self, request):
        site_ids = request.data.get('site_ids', [])
        action = request.data.get('action')
        
        try:
            result = BulkOperationsService.bulk_site_operations(
                site_ids, action, user=request.user, **request.data
            )
            return Response(result)
        except ValidationError as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )

    @action(detail=False, methods=['post'])
    def process_vendors(self, request):
        vendor_ids = request.data.get('vendor_ids', [])
        action = request.data.get('action')
        
        try:
            result = BulkOperationsService.bulk_vendor_operations(
                vendor_ids, action, **request.data
            )
            return Response(result)
        except ValidationError as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )

    @action(detail=False, methods=['post'])
    def process_subscriptions(self, request):
        subscription_ids = request.data.get('subscription_ids', [])
        action = request.data.get('action')
        
        try:
            result = BulkOperationsService.bulk_subscription_updates(
                subscription_ids, action, **request.data
            )
            return Response(result)
        except ValidationError as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )
  
class SubscriptionManagementViewSet(viewsets.ViewSet):
    permission_classes = [IsAuthenticated]
    
    @action(detail=False, methods=['get'])
    def my_subscription(self, request):
        """Get current user's subscription details"""
        subscription = request.user.vendor.get_active_subscription()
        return Response(SubscriptionSerializer(subscription).data)

    @action(detail=False, methods=['post'])
    def upgrade_plan(self, request):
        """Upgrade current subscription plan"""
        new_plan_id = request.data.get('plan_id')
        try:
            result = SubscriptionService.upgrade_plan(
                request.user.vendor, new_plan_id
            )
            return Response(result)
        except ValidationError as e:
            return Response({'error': str(e)}, status=400)

    @action(detail=False, methods=['post'])
    def cancel_subscription(self, request):
        """Cancel current subscription"""
        reason = request.data.get('reason')
        try:
            result = SubscriptionService.cancel_subscription(
                request.user.vendor, reason
            )
            return Response(result)
        except ValidationError as e:
            return Response({'error': str(e)}, status=400)

class VendorAnalyticsViewSet(viewsets.ViewSet):
    permission_classes = [IsAuthenticated]
    
    @action(detail=False, methods=['get'])
    def usage_metrics(self, request):
        """Get vendor's usage metrics"""
        vendor = request.user.vendor
        return Response(
            SubscriptionAnalytics.get_vendor_metrics(vendor)
        )

    @action(detail=False, methods=['get'])
    def site_statistics(self, request):
        """Get vendor's site statistics"""
        vendor = request.user.vendor
        return Response(
            SubscriptionAnalytics.get_site_statistics(vendor)
        )

class AdminDashboardViewSet(viewsets.ViewSet):
    permission_classes = [IsAdminUser]
    
    @action(detail=False, methods=['get'])
    def system_health(self, request):
        """Get system health metrics"""
        return Response(
            SubscriptionAnalytics.get_system_health()
        )

    @action(detail=False, methods=['get'])
    def revenue_metrics(self, request):
        """Get detailed revenue metrics"""
        return Response(
            SubscriptionAnalytics.get_revenue_metrics()
        )

    @action(detail=False, methods=['post'])
    def generate_report(self, request):
        """Generate custom analytics report"""
        report_type = request.data.get('report_type')
        parameters = request.data.get('parameters', {})
        return Response(
            SubscriptionAnalytics.generate_custom_report(
                report_type, parameters
            )
        )
 
class SiteFilter(filters.BaseFilterBackend):
    status = filters.SearchFilter()
    type = filters.SearchFilter()
    title = filters.SearchFilter()
    description = filters.SearchFilter()
    created_after = filters.SearchFilter()
    created_before = filters.SearchFilter()

    class Meta:
        model = Site
        fields = '__all__'

class SiteManagementViewSet(viewsets.ModelViewSet):
    permission_classes = [IsAuthenticated]
    serializer_class = VendorSiteSerializer
    filter_backends = [DjangoFilterBackend, filters.SearchFilter]
 
    search_fields = ['title', 'description']

    def get_queryset(self):
        return Site.objects.filter(
            vendor__user=self.request.user
        ).select_related(
            'address',
            'city',
            'created_by',
            'vendor',
            'reviewed_by',
            'active_subscription'
        ).prefetch_related(
            'images',
            'tags'
        )

    def perform_create(self, serializer):
        serializer.save(
            vendor=self.request.user.vendor_profile,
            created_by=self.request.user
        )

    @action(detail=True, methods=['post'])
    def change_visibility(self, request, pk=None):
        """Change site visibility status"""
        site = self.get_object()
        visibility = request.data.get('visibility')
        try:
            result = site.change_visibility(visibility)
            return Response(result)
        except ValidationError as e:
            return Response({'error': str(e)}, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=['post'])
    def transfer_ownership(self, request, pk=None):
        """Transfer site ownership to another vendor"""
        site = self.get_object()
        new_vendor_id = request.data.get('vendor_id')
        try:
            result = site.transfer_ownership(new_vendor_id)
            return Response(result)
        except ValidationError as e:
            return Response({'error': str(e)}, status=status.HTTP_400_BAD_REQUEST)

    def get_renderer_context(self):
        context = super().get_renderer_context()
        context['template_pack'] = 'rest_framework/vertical/'
        return context
    
class VendorRegistrationView(FormView):
    template_name = 'users/vendor_registration.html'
    form_class = VendorRegistrationForm
    success_url = reverse_lazy('vendor_registration_success')

    def dispatch(self, request, *args, **kwargs):
        if not request.user.is_authenticated:
            messages.warning(request, 'Please log in first to register as a vendor.')
            return redirect('admin:login')  # or wherever you want to redirect
        
        # Check if user already has a vendor account
        if Vendor.objects.filter(user=request.user).exists():
            messages.warning(request, 'You already have a vendor account.')
            return redirect('vendor_dashboard')  # or wherever you want to redirect
            
        return super().dispatch(request, *args, **kwargs)

    def form_valid(self, form):
        try:
            # Check again before saving (double-check)
            if Vendor.objects.filter(user=self.request.user).exists():
                messages.error(self.request, 'You already have a vendor account.')
                return redirect('vendor_dashboard')

            vendor = form.save(commit=False)
            vendor.user = self.request.user
            vendor.save()
            messages.success(self.request, 'Registration successful! Your vendor account is under review.')
            return super().form_valid(form)
            
        except IntegrityError as e:
            messages.error(self.request, 'You already have a vendor account.')
            return redirect('vendor_dashboard')
        except Exception as e:
            messages.error(self.request, f'Registration failed: {str(e)}')
            return self.form_invalid(form)

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['user'] = self.request.user
        return context
 
class VendorRequiredMixin(LoginRequiredMixin):
    """Verify that the current user is authenticated and is a vendor."""
    
    def handle_no_permission(self):
        if not self.request.user.is_authenticated:
            return super().handle_no_permission()
        # If user is authenticated but not a vendor
        messages.warning(self.request, 'You need to register as a vendor first.')
        return redirect('vendor_registration')

class VendorDashboardView(LoginRequiredMixin, TemplateView):
    template_name = 'users/vendor/dashboard.html'

    def dispatch(self, request, *args, **kwargs):
        if not request.user.is_authenticated:
            return self.handle_no_permission()
        
        # Check if user is a vendor
        if not hasattr(request.user, 'vendor'):
            messages.warning(request, 'You need to register as a vendor first.')
            return redirect('vendor_registration')
            
        return super().dispatch(request, *args, **kwargs)

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        vendor = self.request.user.vendor
        
        # Basic statistics
        context.update({
            'vendor': vendor,
            'total_sites': Site.objects.filter(vendor=vendor).count() if hasattr(Site, 'objects') else 0,
            'total_events': Event.objects.filter(vendor=vendor).count() if hasattr(Event, 'objects') else 0,
            'recent_sites': Site.objects.filter(vendor=vendor).order_by('-created_at')[:5] if hasattr(Site, 'objects') else [],
            'recent_events': Event.objects.filter(vendor=vendor).order_by('-created_at')[:5] if hasattr(Event, 'objects') else [],
        })
        
        # Add subscription info if available
        if hasattr(vendor, 'get_active_subscription'):
            context['subscription'] = vendor.get_active_subscription()
        
        return context

class VendorProfileUpdateView(VendorRequiredMixin, UpdateView):
    model = Vendor
    form_class = VendorProfileForm
    template_name = 'users/vendor/profile.html'
    success_url = reverse_lazy('vendor_dashboard')

    def get_object(self, queryset=None):
        return self.request.user.vendor

    def form_valid(self, form):
        messages.success(self.request, 'Profile updated successfully!')
        return super().form_valid(form)

class VendorSitesListView(VendorRequiredMixin, ListView):
    template_name = 'users/vendor/sites.html'
    context_object_name = 'sites'
    paginate_by = 10

    def get_queryset(self):
        return Site.objects.filter(vendor=self.request.user.vendor)

class VendorEventsListView(VendorRequiredMixin, ListView):
    template_name = 'users/vendor/events.html'
    context_object_name = 'events'
    paginate_by = 10

    def get_queryset(self):
        return Event.objects.filter(vendor=self.request.user.vendor)
 
class VendorRegistrationSuccessView(TemplateView):
    template_name = 'users/vendor_registration_success.html'

 

class AnalyticsExportView(APIView):
    permission_classes = [IsAuthenticated]
    
    def get(self, request, *args, **kwargs):
        """Export analytics data to CSV"""
        export_type = request.query_params.get('type', 'subscription')
        date_from = request.query_params.get('date_from')
        date_to = request.query_params.get('date_to')
        
        try:
            # Get data based on export type
            if export_type == 'subscription':
                data = self.get_subscription_data(date_from, date_to)
            elif export_type == 'revenue':
                data = self.get_revenue_data(date_from, date_to)
            else:
                return Response(
                    {'error': 'Invalid export type'},
                    status=status.HTTP_400_BAD_REQUEST
                )
            
            # Create CSV file
            buffer = io.StringIO()
            writer = csv.writer(buffer)
            writer.writerow(data['headers'])
            writer.writerows(data['rows'])
            
            # Create response
            response = HttpResponse(buffer.getvalue(), content_type='text/csv')
            response['Content-Disposition'] = f'attachment; filename="{export_type}_report.csv"'
            return response
            
        except Exception as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )
    
    def get_subscription_data(self, date_from, date_to):
        # Implement subscription data export logic
        return {
            'headers': ['Date', 'Plan', 'Status', 'Revenue'],
            'rows': []  # Add your data here
        }
    
    def get_revenue_data(self, date_from, date_to):
        # Implement revenue data export logic
        return {
            'headers': ['Date', 'Revenue', 'Subscriptions', 'Churn Rate'],
            'rows': []  # Add your data here
        }

class SubscriptionManagementViewSet(ViewSet):
    permission_classes = [IsAuthenticated]
    
    def get_subscription(self):
        """Helper method to get current subscription"""
        return self.request.user.vendor_profile.get_active_subscription()
    
    def list(self, request):
        """Get current subscription details"""
        subscription = self.get_subscription()
        if not subscription:
            return Response({
                'status': 'no_subscription',
                'message': 'No active subscription found'
            })
        
        serializer = SubscriptionDetailSerializer(subscription)
        return Response(serializer.data)
    
    @action(detail=False, methods=['post'])
    def change_plan(self, request):
        """Change subscription plan"""
        serializer = SubscriptionChangeSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        try:
            subscription = self.get_subscription()
            new_plan = SubscriptionPlan.objects.get(
                id=serializer.validated_data['plan_id']
            )
            
            result = subscription.change_plan(
                new_plan,
                immediate=serializer.validated_data.get('immediate', False)
            )
            
            return Response(result)
        except Exception as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )
    
    @action(detail=False, methods=['post'])
    def cancel(self, request):
        """Cancel subscription"""
        serializer = SubscriptionCancellationSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        
        try:
            subscription = self.get_subscription()
            result = subscription.cancel(
                reason=serializer.validated_data['reason'],
                feedback=serializer.validated_data.get('feedback'),
                immediate=serializer.validated_data.get('immediate', False)
            )
            
            return Response(result)
        except Exception as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )
 


class SubscriptionWebhookView(APIView):
    permission_classes = []  # Webhooks typically bypass standard auth
    
    @method_decorator(csrf_exempt)
    def post(self, request, *args, **kwargs):
        """Handle webhook notifications from payment provider"""
        webhook_secret = request.headers.get('X-Webhook-Secret')
        
        if not self.verify_webhook_signature(webhook_secret):
            return Response(
                {'error': 'Invalid webhook signature'},
                status=status.HTTP_401_UNAUTHORIZED
            )
            
        event_type = request.data.get('type')
        event_data = request.data.get('data', {})
        
        try:
            handler = self.get_event_handler(event_type)
            result = handler(event_data)
            return Response(result, status=status.HTTP_200_OK)
        except Exception as e:
            logger.error(f"Webhook error: {str(e)}", exc_info=True)
            return Response(
                {'error': 'Webhook processing failed'},
                status=status.HTTP_400_BAD_REQUEST
            )
    
    def verify_webhook_signature(self, signature):
        # Implement signature verification logic
        return True
    
    def get_event_handler(self, event_type):
        handlers = {
            'subscription.created': self.handle_subscription_created,
            'subscription.updated': self.handle_subscription_updated,
            'subscription.cancelled': self.handle_subscription_cancelled,
            'payment.succeeded': self.handle_payment_succeeded,
            'payment.failed': self.handle_payment_failed,
        }
        return handlers.get(event_type, self.handle_unknown_event)

class SubscriptionCallbackView(APIView):
    permission_classes = [IsAuthenticated]
    
    def get(self, request, *args, **kwargs):
        """Handle return URL from payment provider"""
        subscription_id = request.query_params.get('subscription_id')
        status = request.query_params.get('status')
        
        try:
            subscription = VendorSubscription.objects.get(id=subscription_id)
            if status == 'success':
                subscription.activate()
            elif status == 'failed':
                subscription.mark_payment_failed()
            
            return Response({
                'status': subscription.status,
                'message': 'Subscription status updated'
            })
        except VendorSubscription.DoesNotExist:
            return Response(
                {'error': 'Subscription not found'},
                status=status.HTTP_404_NOT_FOUND
            )

class DocumentUploadView(APIView):
    parser_classes = (MultiPartParser, FormParser)
    permission_classes = [IsAuthenticated]
    
    def post(self, request, *args, **kwargs):
        """Handle document uploads"""
        document_type = request.data.get('document_type')
        files = request.FILES.getlist('files')
        
        if not files:
            return Response(
                {'error': 'No files provided'},
                status=status.HTTP_400_BAD_REQUEST
            )
            
        try:
            uploaded_docs = []
            for file in files:
                doc = VendorDocument.objects.create(
                    vendor=request.user.vendor_profile,
                    document=file,
                    document_type=document_type
                )
                uploaded_docs.append(doc)
            
            return Response({
                'message': f'{len(uploaded_docs)} documents uploaded successfully',
                'documents': VendorDocumentSerializer(uploaded_docs, many=True).data
            })
        except Exception as e:
            return Response(
                {'error': str(e)},
                status=status.HTTP_400_BAD_REQUEST
            )


"""
Add cronjobs to system:
python manage.py crontab add
python manage.py crontab show
python manage.py crontab remove

"""